// Copyright (C) 2010, 2011 EPITA Research and Development Laboratory
// (LRDE)
//
// This file is part of Olena.
//
// Olena is free software: you can redistribute it and/or modify it under
// the terms of the GNU General Public License as published by the Free
// Software Foundation, version 2 of the License.
//
// Olena is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Olena.  If not, see <http://www.gnu.org/licenses/>.
//
// As a special exception, you may use this file as part of a free
// software project without restriction.  Specifically, if other files
// instantiate templates or use macros or inline functions from this
// file, or you compile this file and link it with other files to produce
// an executable, this file does not by itself cause the resulting
// executable to be covered by the GNU General Public License.  This
// exception does not however invalidate any other reasons why the
// executable file might be covered by the GNU General Public License.

#ifndef SCRIBO_IO_XML_SAVE_HH
# define SCRIBO_IO_XML_SAVE_HH

/// \file
///
/// \brief Save document information as XML.

# include <libgen.h>
# include <fstream>
# include <sstream>

# include <map>

# include <scribo/core/document.hh>
# include <scribo/core/line_set.hh>

namespace scribo
{

  namespace io
  {

    namespace xml
    {

      /*! \brief Save document information as XML.

	We use a XML Schema part of the PAGE (Page Analysis and Ground
	truth Elements) image representation framework.

	This schema was used in the Page Segmentation COMPetition
	(PSCOMP) for ICDAR 2009.

	Its XSD file is located here:
	http://schema.primaresearch.org/PAGE/gts/pagecontent/2009-03-16/pagecontent.xsd

      */
      template <typename L>
      void
      save(const document<L>& doc,
	   const std::string& output_name,
	   bool allow_extensions);


# ifndef MLN_INCLUDE_ONLY

      namespace internal
      {

	inline
	std::string&
	html_markups_replace(std::string& input,
			     std::map<char, std::string>& map)
	{
	  for (unsigned i = 0; i < input.size(); ++i)
	  {
	    std::map<char, std::string>::iterator it = map.find(input.at(i));
	    if (it != map.end())
	    {
	      input.replace(i, 1, it->second);
	      i += it->second.size() - 1;
	    }
	  }
	  return input;
	}


	inline
	void print_box_coords(std::ofstream& ostr, const box2d& b,
			      const char *space)
	{
	  std::string sc = space;
	  std::string sp = sc + "  ";
	  ostr << sc << "<coords>" << std::endl
	       << sp << "<point x=\"" << b.pmin().col()
	       << "\" y=\"" << b.pmin().row() << "\"/>"
	       << std::endl
	       << sp << "<point x=\"" << b.pmax().col()
	       << "\" y=\"" << b.pmin().row() << "\"/>"
	       << std::endl
	       << sp << "<point x=\"" << b.pmax().col()
	       << "\" y=\"" << b.pmax().row() << "\"/>"
	       << std::endl
	       << sp << "<point x=\"" << b.pmin().col()
	       << "\" y=\"" << b.pmax().row() << "\"/>"
	       << std::endl
	       << sc << "</coords>" << std::endl;

	}



	template <typename L>
	void
	save(const document<L>& doc,
	     const std::string& output_name)
	{
	  trace::entering("scribo::io::xml:save_text_lines");

	  std::ofstream file(output_name.c_str());
	  if (! file)
	  {
	    std::cerr << "error: cannot open file '" << doc.filename() << "'!";
	    abort();
	  }

	  std::map<char, std::string> html_map;
	  html_map['\"'] = "&quot;";
	  html_map['<'] = "&lt;";
	  html_map['>'] = "&gt;";
	  html_map['&'] = "&amp;";

	  file << "<?xml version=\"1.0\"?>" << std::endl;
	  file << "<pcGts xmlns=\"http://schema.primaresearch.org/PAGE/gts/pagecontent/2009-03-16\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:schemaLocation=\"http://schema.primaresearch.org/PAGE/gts/pagecontent/2009-03-16 http://schema.primaresearch.org/PAGE/gts/pagecontent/2009-03-16/pagecontent.xsd\" pcGtsId=\"" << doc.filename() << "\">" << std::endl;

	  file << "  <pcMetadata>" << std::endl;
	  file << "    <pcCreator>LRDE</pcCreator>" << std::endl;
	  file << "    <pcCreated/>" << std::endl;
	  file << "    <pcLastChange/>" << std::endl;
	  file << "    <pcComments>Generated by Scribo from Olena.</pcComments>" << std::endl;
	  file << "  </pcMetadata>" << std::endl;

	  file << "  <page image_filename=\"" << doc.filename()
	       << "\" image_width=\"" << doc.width()
	       << "\" image_height=\"" << doc.height()
	       << "\">" << std::endl;

	  // Text
	  if (doc.has_text())
	  {
	    const line_set<L>& lines = doc.lines();
	    const paragraph_set<L>& parset = doc.paragraphs();

	    for_all_paragraphs(p, parset)
	    {
	      const mln::util::array<line_id_t>& line_ids = parset(p).line_ids();

	      // FIXME: compute that information on the whole paragraph
	      // and use them here.
	      line_id_t fid = line_ids(0);
	      file << "    <text_region id=\"" << p
		   << "\" txt_orientation=\"" << lines(fid).orientation()
		   << "\" txt_reading_orientation=\"" << lines(fid).reading_orientation()
		   << "\" txt_reading_direction=\"" << lines(fid).reading_direction()
		   << "\" txt_text_type=\"" << lines(fid).type()
		   << "\" txt_reverse_video=\"" << (lines(fid).reverse_video() ? "true" : "false")
		   << "\" txt_indented=\"" << (lines(fid).indented() ? "true" : "false")
		   << "\" kerning=\"" << lines(fid).char_space()
		   << "\">"
		   << std::endl;

	      internal::print_box_coords(file, parset(p).bbox(), "      ");

	      file << "    </text_region>" << std::endl;
	    }
	  }

	  // Page elements (Pictures, ...)
	  if (doc.has_elements())
	  {
	    const component_set<L>& elts = doc.elements();
	    for_all_comps(e, elts)
	      if (elts(e).is_valid())
	      {
		file << "    <image_region id=\"ir" << elts(e).id()
		     << "\" img_colour_type=\"24_Bit_Colour\""
		     << " img_orientation=\"0.000000\" "
		     << " img_emb_text=\"No\" "
		     << " img_bgcolour=\"White\">" << std::endl;

		internal::print_box_coords(file, elts(e).bbox(), "      ");

		file << "    </image_region>" << std::endl;
	      }
	  }


	  file << "  </page>" << std::endl;
	  file << "</pcGts>" << std::endl;

	  trace::exiting("scribo::io::xml::save_text_lines");
	}




	template <typename L>
	void
	save_extended(const document<L>& doc,
		      const std::string& output_name)
	{
	  trace::entering("scribo::io::xml:save_text_lines");

	  std::ofstream file(output_name.c_str());
	  if (! file)
	  {
	    std::cerr << "error: cannot open file '" << doc.filename() << "'!";
	    abort();
	  }

	  std::map<char, std::string> html_map;
	  html_map['\"'] = "&quot;";
	  html_map['<'] = "&lt;";
	  html_map['>'] = "&gt;";
	  html_map['&'] = "&amp;";

	  file << "<?xml version=\"1.0\"?>" << std::endl;
	  file << "<pcGts>" << std::endl;

	  file << "  <pcMetadata>" << std::endl;
	  file << "    <pcCreator>LRDE</pcCreator>" << std::endl;
	  file << "    <pcCreated/>" << std::endl;
	  file << "    <pcLastChange/>" << std::endl;
	  file << "    <pcComments>Generated by Scribo from Olena.</pcComments>" << std::endl;
	  file << "  </pcMetadata>" << std::endl;

	  file << "  <page image_filename=\"" << doc.filename()
	       << "\" image_width=\"" << doc.width()
	       << "\" image_height=\"" << doc.height()
	       << "\">" << std::endl;

	  // Text
	  if (doc.has_text())
	  {
	    const line_set<L>& lines = doc.lines();
	    const paragraph_set<L>& parset = doc.paragraphs();

	    for_all_paragraphs(p, parset)
	    {
	      const mln::util::array<line_id_t>& line_ids = parset(p).line_ids();

	      // FIXME: compute that information on the whole paragraph
	      // and use them here.
	      line_id_t fid = line_ids(0);
	      file << "    <text_region id=\"" << p
		   << "\" txt_orientation=\"" << lines(fid).orientation()
		   << "\" txt_reading_orientation=\"" << lines(fid).reading_orientation()
		   << "\" txt_reading_direction=\"" << lines(fid).reading_direction()
		   << "\" txt_text_type=\"" << lines(fid).type()
		   << "\" txt_reverse_video=\"" << (lines(fid).reverse_video() ? "true" : "false")
		   << "\" txt_indented=\"" << (lines(fid).indented() ? "true" : "false")
		   << "\" kerning=\"" << lines(fid).char_space();

	      // EXTENSIONS - Not officially supported
	      file << "\" baseline=\"" << lines(fid).baseline()
		   << "\" meanline=\"" << lines(fid).meanline()
		   << "\" x_height=\"" << lines(fid).x_height()
		   << "\" d_height=\"" << lines(fid).d_height()
		   << "\" a_height=\"" << lines(fid).a_height()
		   << "\" char_width=\"" << lines(fid).char_width();
	      // End of EXTENSIONS
	      file << "\">"
		   << std::endl;

	      internal::print_box_coords(file, parset(p).bbox(), "      ");


	      // EXTENSIONS - Not officially supported
	      for_all_paragraph_lines(lid, line_ids)
	      {
		line_id_t l = line_ids(lid);

		if (lines(l).has_text())
		{
		  std::string tmp = lines(l).text();
		  tmp = internal::html_markups_replace(tmp, html_map);

		  file << "        <line text=\"" << tmp << "\" ";
		}
		else
		  file << "        <line " << std::endl;

		file << "id=\"" << lines(l).id()
		     << "\" txt_orientation=\"" << lines(l).orientation()
		     << "\" txt_reading_orientation=\"" << lines(l).reading_orientation()
		     << "\" txt_reading_direction=\"" << lines(l).reading_direction()
		     << "\" txt_text_type=\"" << lines(l).type()
		     << "\" txt_reverse_video=\"" << (lines(l).reverse_video() ? "true" : "false")
		     << "\" txt_indented=\"" << (lines(l).indented() ? "true" : "false")
		     << "\" kerning=\"" << lines(l).char_space()
		     << "\" baseline=\"" << lines(l).baseline()
		     << "\" meanline=\"" << lines(l).meanline()
		     << "\" x_height=\"" << lines(l).x_height()
		     << "\" d_height=\"" << lines(l).d_height()
		     << "\" a_height=\"" << lines(l).a_height()
		     << "\" char_width=\"" << lines(l).char_width()
		     << "\">" << std::endl;

		internal::print_box_coords(file, lines(l).bbox(), "          ");

		file << "        </line>" << std::endl;
	      }

	      file << "    </text_region>" << std::endl;
	    }
	  }
	  // End of EXTENSIONS

	  // Page elements (Pictures, ...)
	  if (doc.has_elements())
	  {
	    const component_set<L>& elts = doc.elements();
	    for_all_comps(e, elts)
	      if (elts(e).is_valid())
	      {
		switch (elts(e).type())
		{
		  case component::Separator:
		  {
		    file << "    <separator_region id=\"sr" << elts(e).id()
			 << "\" sep_orientation=\"0.000000\" "
			 << " sep_colour=\"Black\" "
			 << " sep_bgcolour=\"White\">" << std::endl;

		    internal::print_box_coords(file, elts(e).bbox(), "      ");

		    file << "    </separator_region>" << std::endl;
		    break;
		    break;
		  }

		  default:
		  case component::Image:
		  {
		    file << "    <image_region id=\"ir" << elts(e).id()
			 << "\" img_colour_type=\"24_Bit_Colour\""
			 << " img_orientation=\"0.000000\" "
			 << " img_emb_text=\"No\" "
			 << " img_bgcolour=\"White\">" << std::endl;

		    internal::print_box_coords(file, elts(e).bbox(), "      ");

		    file << "    </image_region>" << std::endl;
		    break;
		  }
		}
	      }
	  }


	  file << "  </page>" << std::endl;
	  file << "</pcGts>" << std::endl;

	  trace::exiting("scribo::io::xml::save_text_lines");
	}

      } // end of namespace scribo::io::xml::internal


      // FACADE

      template <typename L>
      void
      save(const document<L>& doc,
	   const std::string& output_name,
	   bool allow_extensions)
      {
	if (allow_extensions)
	  internal::save_extended(doc, output_name);
	else
	  internal::save(doc, output_name);
      }


# endif // ! MLN_INCLUDE_ONLY

    } // end of namespace scribo::io::xml

  } // end of namespace scribo::io

} // end of namespace scribo


#endif // ! SCRIBO_IO_XML_SAVE_HH
