//
// Copyright (C) 1995  Lars Berntzon
//
#include <sadbconfig.hh>
#include <iostream.h>
extern "C" {
#include <stdio.h>
#ifdef HAVE_STRING_H
#include <string.h>
#endif
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <signal.h>
#include <sys/wait.h>
}

#define DEFAULT_TIMEOUT 60

static int timeoutFlag = 0;

//////////////////////////////////////////////////////////////////
//		S A T I M E O U T
//		-----------------
// Description:
//	Execute a command with timeout. If it does not terminate
//	within timeout it will be killed.
//
// Usage:
//	satimeout [-seconds] command arguments
//
// By:
//	Lars Berntzon
//
// Revision:
//	$Id: satimeout.cc,v 1.6 1996/09/14 18:33:41 lasse Exp $
//
//////////////////////////////////////////////////////////////////
static const char *prog;	/* Name of this program		*/
static int pid;			/* Process id of child command	*/


//////////////////////////////////////////////////////////////////
//		U S A G E
//		---------
// Description:
// 	Print usage information and exit.
//
//////////////////////////////////////////////////////////////////
void
usage(void)
{
    (void)fprintf(stderr, "usage: %s [-seconds] command [argument ...]\n", prog);
    (void)fprintf(stderr, "       Default timeout is %d seconds.\n", DEFAULT_TIMEOUT);

    exit(1);
}


//////////////////////////////////////////////////////////////////
//		H D L _T I M E O U T
//		--------------------
// Description:
// 	This is the routine that is called when a timeout actually
//	occurs.
//
//////////////////////////////////////////////////////////////////
void
hdl_timeout(int)
{
    kill(pid, SIGKILL);
    timeoutFlag = 1;
}


//////////////////////////////////////////////////////////////////
//		M A I N
//		-------
// Description:
// 	This is the main routine for timeout.
//
//////////////////////////////////////////////////////////////////
int
main(int argc, char **argv)
{
    int timeout = DEFAULT_TIMEOUT;
    int status = 0;

    /*
     * Calculate name of program and strip directory part.
     */
    if ((prog = strrchr(argv[0], '/'))) prog++;
    else prog = argv[0];

    if (argc < 2) {
	usage();
    }

    /*
     * Examinte what timeout to use.
     */
    if (argv[1][0] == '-') {
	timeout = atoi(argv[1] + 1);
	if (timeout == 0) {
	    (void)fprintf(stderr, "%s: timeout value must be grater than zero.\n", prog);
	    exit(1);
	}
	argv++;
	argc--;
    }

    /*
     * Start executing the command.
     */
    if ((pid = fork()) == 0)
    {
	execvp(argv[1], argv + 1);
	(void) fprintf(stderr, "%s: failed to execute command: %s\n", prog, argv[1]);
    }

    signal(SIGALRM, hdl_timeout);
    alarm(timeout);
    wait(&status);
    alarm(0);
    
    /*
     * Return correct value. It should be error if the
     * command suffered timeout and it should be error
     * if the actual command returned error.
     */
    if (timeoutFlag) {
    	cerr << "timeout\n";
	return 1;
    } else {
	return WEXITSTATUS(status);
    }
}

//
// History of changes:
// $Log: satimeout.cc,v $
// Revision 1.6  1996/09/14 18:33:41  lasse
// Added some things to the TODO and added pargs
//
// Revision 1.5  1996/05/24 19:48:50  lasse
// Print error message about timeout when timeout occurs
//
// Revision 1.4  1996/03/12 19:43:17  lasse
// Checking in from remote.
//
// Revision 1.3  1995/09/23  13:46:14  lasse
// Imported from remote
//
// Revision 1.1.1.1  1995/09/11  09:23:11  qdtlarb
// THis is version 0.6
//
// Revision 1.2  1995/09/10  20:43:27  lasse
// Added copyright everywhere
//
// Revision 1.1  1995/08/31  18:17:27  lasse
// Created
//
// Revision 1.3  1995/08/08  09:10:10  qdtlarb
// added sarand
//
// Revision 1.2  1995/08/07  08:08:11  qdtlarb
// use correct include files
//
// Revision 1.1  1995/08/07  08:01:40  qdtlarb
// Created
//
//
