//
// Copyright (C) 1995  Lars Berntzon
//
#include <sadblib.hh>
#include <iostream.h>
extern "C" {
#include <fcntl.h>
#include <unistd.h>
#include <sys/stat.h>
#ifdef HAVE_REGEX_H
#include <regex.h>
#endif
}

char SADB::errorMessage[MAXNAME];
int SADB::errno = 0;
static TextList emptyTextList;
static ListList emptyListList;

//
// M e t h o d s .
//
SADB::SADB(void)
{
    dbDirectory[0] = 0;
}

SADB::SADB(const char *dbName)
{
    dbDirectory[0] = 0;
    open(dbName);
}

SADB::~SADB()
{
    errno = 0;
}

//////////////////////////////////////////////////////////////////
//		O P E N
//		-------
// Description:
//	Open a database.
//
//////////////////////////////////////////////////////////////////
int
SADB::open(const char *dbName)
{
    errno = 0;

    //
    // Get directory for database.
    //
    database_directory(dbName, dbDirectory);

    if (access(dbDirectory, F_OK) != 0) {
    	sprintf(errorMessage, "%s: no such database", dbName);
    	errno = 1;
    	return 1;
    }

    return 0;
}
//////////////////////////////////////////////////////////////////
//		C O L U M N _ E X I S T
//		-----------------------
// Description:
//	Check if column exist in database.
//
//////////////////////////////////////////////////////////////////
int
SADB::column_exist(const char *name)
{
    errno = 0;
    int rc;
    char *fileName = new char[strlen(dbDirectory) + strlen(name) + 20];
#ifdef USE_NDBM
    sprintf(fileName, "%s/%s.pag", dbDirectory, name);
#endif
#ifdef USE_GDBM
    sprintf(fileName, "%s/%s", dbDirectory, name);
#endif

    rc =  access(fileName, O_RDONLY);
    delete [] fileName;

    return rc;
}

///////////////////////////////////////////////////////////////////
//		E X T R A C T _ C O L U M N S
//		-----------------------------
// Description:
//	Extract data for each entry. This is done by making a new list
//	with each entry as a ListEntry type and the values of everything
//	that should be shown is converted into one TextEntry per shown column
//	in this sublist.
//
// Arguments:
//	columns		List of columns to extract
//	entries		List of entries to examine
//
///////////////////////////////////////////////////////////////////

ListList
SADB::extract_columns(TextList &columns, TextList &entries)
{
    errno = 0;
    ListList returnList;	// List of columns to be returned.
    char fileName[PATH_MAX];	// Temporary filename.
    const char *data;		// data and key for DBM-work.
    Pix i;			// Index;
    Pix col;			// Column index.

    //
    // Make return list to be a list of lists with one entry
    // per each input list entry.
    //
    for(i = entries.first(); i != 0; entries.next(i))
    {
	returnList.append(ListEntry(entries(i).name()));
    }

    //
    // Loop through all columns to be fetched.
    //
    for(col = columns.first(); col != 0; columns.next(col))
    {
	//
	// Open the dbm file for current column.
	//
	sprintf(fileName, "%s/%s", dbDirectory, columns(col).name());
	DB_OF_CHOICE db(fileName);
	if (!db.ok()) {
	    sprintf(errorMessage, "%s: failed to open database", fileName);
	    errno = 1;
	    return emptyListList;
	}

	//
	// Loop through all entries in order to extract all column
	// values for that entry.
	//
	for(i = returnList.first(); i != 0; returnList.next(i))
	{
	    //
	    // Fetch data from current column for current entry.
	    //
	    data = db.fetch(returnList(i).name());
	    if (data == 0) {
	    	data = "";
	    }

	    //
	    // Make list entry with column name and column value.
	    //
	    TextList &tlist = returnList(i).list();
	    tlist.append(TextEntry(columns(col).name(), data));
	}
    }

    return returnList;
}


///////////////////////////////////////////////////////////////////
//		L I S T _ D A T A B A S E S
//		---------------------------
// Description:
//	Return a list of available databases.
//
///////////////////////////////////////////////////////////////////
TextList
SADB::list_databases(void)
{
    DIR *dp;
    struct dirent *entry;
    TextList returnList;
    char topDBDirectory[PATH_MAX];
    char tmpString[MAXNAME];
    int len;

    //
    // Get base directory.
    //
    base_directory(topDBDirectory);

    //
    // Open directory for reading entries.
    //
    if ((dp = opendir(topDBDirectory)) == NULL) {
    	sprintf(errorMessage, "%s: can open directory", topDBDirectory);
    	errno = 1;
    	return emptyTextList;
    }

    //
    // Read each entry. Att least 3 entries must exist to be a
    // database directory.
    //
    while ((entry = readdir(dp)))
    {
	len = strlen(entry->d_name);
    	if (len < 3) {	// 3 is the length of ".db" which is minimum.
    	    continue;
	}

    	// 
    	// Check if entry matches *.db.
    	//
    	if ((strncmp(entry->d_name + len - 3, ".db", 3) == 0))
	{
    	    strncpy(tmpString, entry->d_name, len - 3);
    	    tmpString[len - 3] = 0;
    	    returnList.append(TextEntry(tmpString));
    	}
    }

    closedir(dp);

    return returnList;
}


///////////////////////////////////////////////////////////////////
//		L I S T _ C O L U M N S
//		-----------------------
// Description:
//	Return a list of valid columns.
//
///////////////////////////////////////////////////////////////////
TextList
SADB::list_columns(void)
{
    errno = 0;
    DIR *dp;
    struct dirent *entry;
    TextList returnList;
    char tmpString[MAXNAME];
    int len;

    //
    //Open directory for reading entries, two entries
    //per valid column (.pag and .dir).
    //
    if ((dp = opendir(dbDirectory)) == NULL) {
    	sprintf(errorMessage, "%s: can open directory", dbDirectory);
    	errno = 1;
    	return emptyTextList;
    }

    //
    //Read each entry. Only record .dir files (could be .pag but
    //must be only one).
    //
    while ((entry = readdir(dp)))
    {
	len = strlen(entry->d_name);
#ifdef USE_NDBM
    	if (len < 4) {	// 4 is the length of ".dir" */
    	    continue;
	}

    	// 
    	// Add to list if extension is ".dir" (but without
    	// the .dir part), also the key column shall allways be
    	// skipped since it must allways exist.
    	//
    	if ((strncmp(entry->d_name, "key.dir", len) != 0)  &&
	    (strncmp(entry->d_name + len - 4, ".dir", 4) == 0))
	{
    	    strncpy(tmpString, entry->d_name, len - 4);
    	    tmpString[len - 4] = 0;
	    returnList.append(TextEntry(tmpString));
    	}
#endif
#ifdef USE_GDBM
	//
	// Skip hidden files (like . and ..).
	//
	if (entry->d_name[0] == '.') {
	    continue;
	}
    	//
    	// Skip the key, its always there.
    	//
    	if (strncmp(entry->d_name, "key.dir", len) == 0) {
    	    continue;
	}
	strncpy(tmpString, entry->d_name, len);
	tmpString[len] = 0;
	returnList.append(TextEntry(tmpString));
#endif
    }

    closedir(dp);

    return returnList;
}


//////////////////////////////////////////////////////////////////
//		L I S T _ E N T R I E S
//		-----------------------
// Description:
//	List all entries in the database (the key values).
//
//////////////////////////////////////////////////////////////////
TextList
SADB::list_entries(void)
{
    errno = 0;
    const char *key;
    TextList returnList;
    char fileName[PATH_MAX];

    sprintf(fileName, "%s/key", dbDirectory);
    DB_OF_CHOICE db(fileName);
    if (!db.ok()) {
    	sprintf(errorMessage, "%s: failed to open database", fileName);
	errno = 1;
	return emptyTextList;
    }

    //
    // Make a list of all entries. Except those that begin with
    // a description. string.
    //
    for(key = db.firstkey(); key != 0;  key = db.nextkey())
    {
    	//
    	// Skip entries that describe the columns.
    	//
    	if (strncmp(key, "description.", 12) != 0) {
	    returnList.append(TextEntry(key));
	}
    }

    return returnList;
}


//////////////////////////////////////////////////////////////////
//		M A T C H
//		---------
//
// Description:
//	Remove all items in input list which doesn't match.
//
// Return:
//	0 for ok and 1 for error.
//
//////////////////////////////////////////////////////////////////
int
SADB::match(const char *name, const char *str, TextList &entries, TextList &returnList)
{
    errno = 0;
    char buf[MAXNAME];
    char fileName[PATH_MAX];
    const char *data;
    int rc;
    Pix i;
    char *tok;
    int invertMode = 0;
    int regexpMode = 0;
    int matchFlag = 0;
#ifdef HAVE_REGEX_H
    regex_t preg;
#endif

    //
    // No need to match an empty list.
    //
    if (entries.empty()) {
	return 0;
    }

    //
    // Check matching mode, match or not matching. Not match
    // is specified by a ^. Regular expression match is specified
    // by a ~ (like perl or AWK).
    //
    while(1)
    {
	if (name[0] == '^') {
	    name++;
	    invertMode = 1;
	    continue;
	}
	else if (name[0] == '~') {
#ifndef HAVE_REGEX_H
	    strcpy("regular expressions not supported (regexp.h not found)", errorMessage);
	    return errno = 1;
#else
	    name++;
	    regexpMode = 1;
	    if ((rc = regcomp(&preg, str, 0)) != 0) {
	    	regerror(rc, &preg, errorMessage, sizeof errorMessage);
	    	return errno = 1;
	    }
	    continue;
#endif
	}
	break;
    }

    //
    // Open column database.
    //
    sprintf(fileName, "%s/%s", dbDirectory, name);

    DB_OF_CHOICE db(fileName);

    if (!db.ok()) {
    	sprintf(errorMessage, "%s: failed to open database", fileName);
	return errno = 1;
    }

    //
    // Loop through all entries.
    //
    for(i = entries.first(); i != 0; entries.next(i))
    {
	// Reset match flag.
	matchFlag = 0;

	//
	// Read data for entry.
	//
	data = db.fetch(entries(i).name());
	if(data == NULL) {
	     data = "";
	}
	strcpy(buf, data);

	//
	// Add entries which does match.
	// Test on all semicolon separated fields.
	// If first token not found use empty string which
	// would otherwise not match (the for-loop would
	// end immediately).
	//
	if ((tok = strtok(buf, ";")) == NULL) {
	    tok = "";
	}

	for(; tok != 0; tok = strtok(0, ";"))
	{
	    if (regexpMode) {
#ifdef HAVE_REGEX_H
		if (regexec(&preg, tok, 0, 0, 0) == 0) {
		    matchFlag = 1;
		}
#endif
	    }
	    else {
		if(strcmp(tok, str) == 0) {
		    matchFlag = 1;
		    break;
		}
	    }
	}

	//
	// Invert match if specified.
	//
	if (invertMode) {
	    matchFlag = (matchFlag == 0);
	}

	//
	// Add entry if match.
	//
	if (matchFlag == 1) {
	    returnList.append(entries(i).name());
	}
    }

    //
    // Free the regular expression area if used.
    //
    if (regexpMode) {
#ifdef HAVE_REGEX_H
	regfree(&preg);
#endif
    }

    return 0;
}


//////////////////////////////////////////////////////////////////
//		F E T C H
//		---------
// Description:
//	Get a column value for specified key.
//
//////////////////////////////////////////////////////////////////
const char *
SADB::fetch(const char *key, const char *column)
{
    errno = 0;
    char fileName[PATH_MAX];
    static char returnBuf[MAXNAME];
    const char *p;

    sprintf(fileName, "%s/%s", dbDirectory, column);

    DB_OF_CHOICE db(fileName);

    if (!db.ok()) {
	sprintf(errorMessage, "%s: failed to open database", fileName);
	errno = 1;
	return NULL;
    }

    p = db.fetch(key);

    if (p != 0) {
       strcpy(returnBuf, p);
    }
    else {
    	return 0;
    }

    return returnBuf;
}


//////////////////////////////////////////////////////////////////
//		S T O R E
//		---------
// Description:
//	Store a value in a column for specified record.
//
//////////////////////////////////////////////////////////////////
int
SADB::store(const char *key, const char *column, const char *value)
{
    errno = 0;
    char fileName[PATH_MAX];

    sprintf(fileName, "%s/%s", dbDirectory, column);

    DB_OF_CHOICE db(fileName, 1);

    if (!db.ok()) {
	sprintf(errorMessage, "%s: failed to open database", fileName);
	errno = 1;
	return 1;
    }

    return db.store(key, value, DB::replace_t);
}


//////////////////////////////////////////////////////////////////
//		R E M O V E
//		-----------
// Description:
//	Remove an entry from a column.
//
//////////////////////////////////////////////////////////////////
int
SADB::remove(const char *key, const char *column)
{
    errno = 0;
    char fileName[PATH_MAX];

    sprintf(fileName, "%s/%s", dbDirectory, column);

    DB_OF_CHOICE db(fileName, 1);

    if (!db.ok()) {
	sprintf(errorMessage, "%s: failed to open database", fileName);
	errno = 1;
	return 1;
    }

    return db.remove(key);
}

//////////////////////////////////////////////////////////////////
//		P A R S E   M A T C H I N G
//		---------------------------
//
// Description:
//	Parse command line arguments for matching and perform
//	the match for all entries in the database which will
//	result in a equal or shorter list of entries. The argument
//	list shall *not* contain the procedure name as index 0.
//
// Arguments:
//	argc		- Number of arguments.
//	argv		- Argument list.
//
// Return:
//	0 for ok and 1 for error.
//
//////////////////////////////////////////////////////////////////
int
SADB::parse_matching(int argc, char **argv, TextList &entries)
{
    errno = 0;
    TextList matchList;			// List of tables to match.
    int i;				// Counter.
    Pix matchIndex;			// Match index.
    const char *matchValue;

    //
    // Parse the command line options.
    //
    for(i = 0; i < argc; i++)
    {
	//
	// Is it a matching option (like -column value)?
	//
	if (argv[i][0] == '-')
	{

	    //
	    // Skip first chars which has to to with
	    // notation.
	    char *p =  argv[i] + 1;
	    if (*p == '^') p++;
	    if (*p == '=') p++;
	    if (*p == '~') p++;

	    if (column_exist(p) != OK) {
	    	sprintf(errorMessage, "%s: no such column", p);
	    	errno = 1;
	    	return 1;
	    }

	    //
	    // Get data to match.
	    //
	    if (i + 1 >= argc) {
	    	sprintf(errorMessage, "%s: needs value for column", p);
	    	errno = 1;
	    	return 1;
	    }
	    matchValue = argv[i + 1];

	    //
	    // If the -=column method is used, then fetch value of key
	    //
	    if (p[-1] == '=') {
		matchValue = fetch(matchValue, p);
		if (matchValue == NULL) {
	 	    matchValue = "";
		}
	    }
	    else {
		p =  argv[i] + 1;
	    }

	    matchList.append(TextEntry(p, matchValue));

	    i++;
	}

	// 
	// Otherwise it is a entry.
	//
	else {
	    entries.append(TextEntry(argv[i]));
	}
    }

    //
    // If no entries on commandline, get every entry.
    //
    if (entries.empty()) {
	entries = list_entries();	
	if (entries.empty()) {
	    cerr << "no entries in database\n";
	    return 1;
	}
    }
    else {
    }

    //
    // Remove all entries that does not match the specified criteria.
    // Every round the direction of lists are reversed. This is to skip
    // one extra list copy per phase.
    //
    if (!matchList.empty())
    {
	TextList tmpList;
	TextList *ll[2] = {&entries, &tmpList};
	i = 0;
	for(i = 0, (matchIndex = matchList.first());
	    matchIndex != 0;
	    i = 1 - i, matchList.next(matchIndex))
	{
	    // Clear the target list.
	    ll[1 - i]->clear();	
	    match(matchList(matchIndex).name(), matchList(matchIndex).text(), *ll[i], *ll[1-i]);
	}

	//
	// If the last round whent to tmpList the return list must be set.
	//
	if (i == 1) {
	    entries = tmpList;
	}
    }
    return 0;
}

int
SADB::create_database(const char *databaseName)
{
    char buf[PATH_MAX];
    int rc;

    rc = database_directory(databaseName, buf);
    if (rc != 0) {
    	return rc;
    }

cout << "Trying to create: " << buf << endl;
    return mkdir(buf, 0777);
}

int
SADB::Tcl_AppInit(Tcl_Interp *interp)
{
    TextList dataBases;
    Pix i;

    //
    // Register functions, one per database.
    //
    dataBases = SADB::list_databases();
    for(i = dataBases.first(); i != 0; dataBases.next(i))
    {
	Tcl_CreateCommand(interp,
			  (char *)dataBases(i).name(),
			  proc_sadb,
			  0,
			  NULL);
    }

    return TCL_OK;
}

//////////////////////////////////////////////////////////////////
//		P R O C _ S A D B
//		-----------------
// Description:
//	Read entries from the database.
//
//////////////////////////////////////////////////////////////////
int
SADB::proc_sadb(ClientData clientData, Tcl_Interp *interp, int argc, char **argv)
{
    Pix i;
    TextList entries;
    SADB db(argv[0]);

    //
    // Shift away routine name.
    //
    argc--, argv++;

    //
    // Make a list of entires which match according to
    // command line arguments.
    //
    db.parse_matching(argc, argv, entries);
    if (db.errno) {
	interp->result = db.errorMessage;
	return TCL_ERROR;
    }

    Tcl_ResetResult(interp);
    for(i = entries.first(); i != 0; entries.next(i))
    {
    	Tcl_AppendElement(interp, (char *)entries(i).name());
    }

    return TCL_OK;
}

int
SADB::database_directory(const char *databaseName, char *directory)
{
    int rc;
    rc = base_directory(directory);
    if (rc != 0) {
    	return rc;
    }
    strcat(directory, "/");
    strcat(directory, databaseName);
    strcat(directory, ".db");

    return 0;
}

int
SADB::base_directory(char *directory)
{
    char *env;

    //
    //Get home directory for satools.
    //
    if ((env = getenv(SADBDIR)) != NULL) {
	sprintf(directory,  "%s", env);
    }
    else {
	if ((env = getenv(SAHOMEENV)) == NULL) {
	    sprintf(errorMessage, "neither %s nor %s is defined\n", SADBDIR, SAHOMEENV);
	    errno = 1;
	    return 1;
	}
    
	sprintf(directory,  "%s/sadb", env);
    }
    return 0;
}
    

//
// History of changes:
// $Log: SADB.cc,v $
// Revision 1.39  1997/07/18 19:57:52  lasse
// sadbcmd now handles -~column for regular expressions.
//
// Revision 1.38  1996/09/14 18:33:31  lasse
// Added some things to the TODO and added pargs
//
// Revision 1.37  1996/07/30 20:41:47  lasse
// Corrected GDBM databases
//
// Revision 1.36  1996/05/01 20:40:42  lasse
// backup before 0.13
//
// Revision 1.35  1996/03/12 19:42:29  lasse
// Checking in from remote.
//
// Revision 1.33  1996/01/22  20:17:21  lasse
// Checking in from mobile
//
// Revision 1.32  1995/11/09  21:21:00  lasse
// To be 0.9
//
// Revision 1.31  1995/11/06  20:53:07  lasse
// Now xsadb can list entries that matches
//
// Revision 1.30  1995/10/22  22:18:03  lasse
// Backup
//
// Revision 1.29  1995/10/22  12:33:26  lasse
// Backup
//
// Revision 1.28  1995/10/16  22:51:46  lasse
// bckup
//
// Revision 1.27  1995/09/23  13:46:05  lasse
// Imported from remote
//
// Revision 1.1.1.1  1995/09/11  09:23:01  qdtlarb
// THis is version 0.6
//
// Revision 1.26  1995/09/10  20:43:21  lasse
// Added copyright everywhere
//
// Revision 1.25  1995/09/10  19:47:44  lasse
// Depend on environment variables.
//
// Revision 1.24  1995/09/10  19:03:40  lasse
// Corrected removed Log keyword
//
// Revision 1.1.1.1  1995/07/17  07:51:30  qdtlarb
// Original V0_3
//
// Revision 1.22  1995/07/16  13:42:25  lasse
// removed merging differences
//
// Revision 1.21  1995/07/05  17:32:23  lasse
// added environment variable SADBDIR
//
// Revision 1.20  1995/06/15  10:56:30  lasse
// Added registration of tcl commands
//
// Revision 1.19  1995/06/09  21:12:21  lasse
// backup
//
// Revision 1.18  1995/06/09  17:13:53  lasse
// Started to convert to use libg++ DLList, sadblib seems ok. sadbcmd
// compiles but does not work.
//
// Revision 1.17  1995/05/21  12:49:53  lasse
// backup
//
// Revision 1.16  1995/05/13  17:55:06  lasse
// backup
//
// Revision 1.15  1995/05/13  17:33:35  lasse
// added parse_matching
//
// Revision 1.14  1995/05/13  17:20:51  lasse
// backup
//
// Revision 1.13  1995/05/13  16:56:55  lasse
// Bug with SADB::~SADB delete cleared
//
// Revision 1.12  1995/05/13  16:32:02  lasse
// It seems much better now.
//
// Revision 1.11  1995/05/13  15:05:17  lasse
// backup
//
// Revision 1.10  1995/05/12  13:42:05  lasse
// backup
//
// Revision 1.9  1995/05/12  11:21:48  lasse
// added support for gdbm
//
// Revision 1.8  1995/05/12  10:01:59  lasse
// backup
//
// Revision 1.7  1995/05/09  21:49:51  lasse
// Backup
//
// Revision 1.6  1995/05/09  18:32:54  lasse
// Backup before converting List into a Proper List and adding Entry
//
// Revision 1.5  1995/05/08  09:36:52  lasse
// backup
//
// Revision 1.4  1995/05/03  15:20:09  lasse
// backup
//
// Revision 1.3  1995/05/03  11:42:13  lasse
// Removed generated file
//
// Revision 1.2  1995/05/02  22:39:18  lasse
// It seems to work slightly
//
// Revision 1.1.1.1  1995/05/02  22:18:19  lasse
// First checkin into cvs
//
//
