//////////////////////////////////////////////////////////////////
// Name:
//	SADBCMD
//
// Synopsis:
//	<sadb command name> [-show <column>] [-<column> <value>] [-u] [-shell | -semi | -export] [entry ...]
//
// Description:
//	SADB is a command line interface to a database base on
//	UNIX DBM files. This database can be used for storing information
//	about lot of things, e.g. workstations in a UNIX network. This command
//	can be use from various system administration scripts to determine
//	which machines to operate on or what data to use per machine.
//
//	The command is by default named *machines* and it will thereby
//	use the DBM database under a directory called $SAHOME/sadb/machines.db,
//	however if the executable is named some thing else, eg. printers
//	it will use a DBM database under $SAHOME/sadb/printers.db so any number
//	of databases can be used.
//	
//	In the database directory ($SAHOME/sadb/<db>) there shall be any number
//	of columns, one column per information type, eg. the column
//	"location" could contain information about a machines location. There
//	is one predefined column which must exist and that is the "key" column
//	which contain a list of all entries in the database, eg: $SAHOME/sadb/
//		$SAHOME/sadb/printers/key.{pag,dir}
//	shall contain all printers that this command should know about.
//	In this key database the key and value is the same namely the machine
//	or printer name. In all other column the key is the machine or
//	printer name and the value is the value of the table, eg. the location
//	format could be:
//		key=printer23	value=1002
//
// Options:
//	-show column	- Tell sadbcmd to list the value of columns for all
//			  entries that match.
//	-semi		- Output in columns separaded by semicolons, suitable for
//			  importing for instance into MS Access.
//	-u		- Show the set of values, not all of the but only
//			  the different types that exist. This option does
//			  not show the entry name (e.g. machine name or printer
//			  name).
//	-shell		- Produce a output that can be evaluated by the shell.
//			  This is used for setting shell variables named after
//		 	  the database column and with a value of that database
//			  column value.
//	-export		- Produce output in the mkmachines format. This is suitable
//			  for exporting a database in text format.
//
// Copyright (C) 1995  Lars Berntzon
//
//
//////////////////////////////////////////////////////////////////

#include <sadblib.hh>
#include <sadbcmd.hh>
#include <iostream.h>

//
//  G l o b a l   s t u f f .
//
char *prog;

//////////////////////////////////////////////////////////////////
//		M A I N
//		-------
//
// Description:
//	Main routine for machines.
//
//////////////////////////////////////////////////////////////////
int
main(int argc, char **argv)
{
    TextList showList;			// List of columns to show.
    TextList entries;			// List of entries.
    ListList columns;			// List of entries with list of columns.
    int unique = 0;			// If -u option.
    int singleMachine = 0;		// Flag true if only one machine spec.
    FILE *out = 0;			// Output filter (sort -u).
    int i = 0;				// Counter.
    int match_argc = 0;			// No of arguments for SADB::parse_matching.
    char **match_argv;			// Arguments for SADB::parse_matching.
    Pix entryIndex;			// Index over entries.
    enum OutputMode outputMode;		// Output mode, either normal, shell, export or semi.

    //
    // Set default output mode.
    //
    outputMode = normal;

    //
    // Get the name of the program. This is later used
    // to determine which database to use.
    //
    if ((prog = strrchr(argv[0], '/'))) prog++;
    else prog = argv[0];

    //
    // Connect to the database.
    //
    SADB database(prog);
    if (database.errno) {
	cerr << prog << ": " << database.errorMessage << endl;
	return 1;
    }

    //
    // Allocate space to match_argv.
    //
    match_argv = (char **)malloc(argc * sizeof (char *));

    //
    // Get command line options.
    //
    for(i = 1; i < argc; i++)
    {
	//
	// Is it an show option.
	//
	if (strcmp(argv[i], "-show") == 0)
	{
	    if (++i >= argc) {
	    	fprintf(stderr, "%s: option -show needs argument\n", prog);
		sadb_usage(database);
	    }

	    //
	    // Check if showing all columns or just some of them.
	    //
	    if (strcmp(argv[i], "all") == 0) {
		if (!showList.empty()) {
		    fprintf(stderr, "%s: -show all: conflicting -show option\n", prog);
		    sadb_usage(database);
		}
		showList = database.list_columns();
	    }
	    else {
		if (database.column_exist(argv[i]) != OK) {
		    fprintf(stderr, "%s: %s: unknown show option\n", prog, argv[i]);
		    sadb_usage(database);
		}
		showList.append(TextEntry(argv[i]));
	    }
	}

	//
	// Unique mode.
	//
	else if (strcmp(argv[i], "-u") == 0) {
	    unique = 1;
	}

	//
	// Upper case for column names
	//
	else if (strcmp(argv[i], "-shell") == 0) {
	    outputMode = shell;
	}

	//
	// Output is to be in semicolon separated list.
	//
	else if (strcmp(argv[i], "-semi") == 0)  {
	    outputMode = semi;
	}

	//
	// Output is to be in export format.
	//
	else if (strcmp(argv[i], "-export") == 0)  {
	    outputMode = export;
	}

	//
	// Give some help.
	//
	else if (strcmp(argv[i], "-help") == 0)
	{
	    fprintf(stderr, "%s is a tool for examining the database of %s,\n", prog, prog);
	    sadb_usage(database);
	}

	//
	// Column matching.
	// HACKMARK: there is possible for conflict with other options
	// above if entries names begins with a -.
	//
	else
	{
	    match_argv[match_argc++] = argv[i];
	}
    }

    //
    // Now parse the matching options, resulting in a list of
    // entries.
    //
    database.parse_matching(match_argc, match_argv, entries);
    if (database.errno) {
    	cerr << prog << ": " << database.errorMessage << endl;
    	return 1;
    }

    // 
    // Check if only one entry was specified which alters
    // the way data is shown as described in show_entries.
    //
    entryIndex = entries.first();
    entries.next(entryIndex);
    if (entryIndex == 0) {
	singleMachine = 1;
    }

    /*
     * Generate list of data to show for each machine, this turns
     * entries into a list of lists with keys (column) and values.
     */
    columns = database.extract_columns(showList, entries);

    /*
     * Open sort pipe
     */
    if (unique) {
	 out = popen("sort -u |  sort -tk -n +2 +0", "w");
    }
    else {
	out = stdout;
    }

    /*
     * Print entries and/or column values.
     */
    show_entries(columns, showList, singleMachine, unique, outputMode, out);

    if (unique) {
	pclose(out);
    }

    return 0;
}

//
// History of changes:
// $Log: sadbcmd.cc,v $
// Revision 1.23  1997/03/18 20:13:56  lasse
// Created quote_escape.cc
//
// Revision 1.22  1996/11/10 00:02:40  lasse
// Don't forget the newline for error messages saying database not found.
//
// Revision 1.21  1996/09/14 18:33:27  lasse
// Added some things to the TODO and added pargs
//
// Revision 1.20  1995/11/06  20:53:06  lasse
// Now xsadb can list entries that matches
//
// Revision 1.19  1995/09/24  16:30:40  lasse
// Print error if SAHOME or SADBDIR is not set
//
// Revision 1.18  1995/09/23  13:46:02  lasse
// Imported from remote
//
// Revision 1.1.1.1  1995/09/11  09:22:59  qdtlarb
// THis is version 0.6
//
// Revision 1.17  1995/09/10  20:43:16  lasse
// Added copyright everywhere
//
// Revision 1.16  1995/09/10  19:03:37  lasse
// Corrected removed Log keyword
//
// Revision 1.1.1.1  1995/07/17  07:51:28  qdtlarb
// Original V0_3
//
// Revision 1.14  1995/07/16  13:42:09  lasse
// removed merging differences
//
// Revision 1.13  1995/06/11  10:35:18  lasse
// backup
//
// Revision 1.12  1995/06/09  21:12:20  lasse
// backup
//
// Revision 1.11  1995/06/09  17:13:51  lasse
// Started to convert to use libg++ DLList, sadblib seems ok. sadbcmd
// compiles but does not work.
//
// Revision 1.10  1995/05/13  17:55:05  lasse
// backup
//
// Revision 1.9  1995/05/13  17:44:09  lasse
// Use a library routine for parsing matching options.
//
// Revision 1.8  1995/05/13  15:05:55  lasse
// ubackup
//
// Revision 1.7  1995/05/12  13:36:39  lasse
// backup
//
// Revision 1.6  1995/05/12  11:44:37  lasse
// renamed sadb to sadbcmd
//
// Revision 1.5  1995/05/12  11:41:27  lasse
// backup
//
// Revision 1.4  1995/05/12  10:01:58  lasse
// backup
//
// Revision 1.3  1995/05/09  11:33:44  lasse
// Backup
//
//
