//
// ********************************************************************
// * License and Disclaimer                                           *
// *                                                                  *
// * The  Geant4 software  is  copyright of the Copyright Holders  of *
// * the Geant4 Collaboration.  It is provided  under  the terms  and *
// * conditions of the Geant4 Software License,  included in the file *
// * LICENSE and available at  http://cern.ch/geant4/license .  These *
// * include a list of copyright holders.                             *
// *                                                                  *
// * Neither the authors of this software system, nor their employing *
// * institutes,nor the agencies providing financial support for this *
// * work  make  any representation or  warranty, express or implied, *
// * regarding  this  software system or assume any liability for its *
// * use.  Please see the license in the file  LICENSE  and URL above *
// * for the full disclaimer and the limitation of liability.         *
// *                                                                  *
// * This  code  implementation is the result of  the  scientific and *
// * technical work of the GEANT4 collaboration.                      *
// * By using,  copying,  modifying or  distributing the software (or *
// * any work based  on the software)  you  agree  to acknowledge its *
// * use  in  resulting  scientific  publications,  and indicate your *
// * acceptance of all terms of the Geant4 Software license.          *
// ********************************************************************
//
/// \file Run.hh
/// \brief Definition of the Run class

#ifndef Run_h
#define Run_h 1

#include "G4Run.hh"
#include "G4VProcess.hh"
#include "globals.hh"

#include <map>

class G4ParticleDefinition;

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

class Run : public G4Run
{
  public:
    Run() = default;
    ~Run() override = default;

  public:
    void ParticleCount(G4String, G4double, G4double);
    void Balance(G4double, G4double);
    void EventTiming(G4double);
    void PrimaryTiming(G4double);
    void EvisEvent(G4double);

    void SetTimeWindow(G4double, G4double);
    void CountInTimeWindow(G4String, G4bool, G4bool, G4bool);

    void SetPrimary(G4ParticleDefinition* particle, G4double energy);
    void EndOfRun();

    void Merge(const G4Run*) override;

  private:
    struct ParticleData
    {
        ParticleData() : fCount(0), fEmean(0.), fEmin(0.), fEmax(0.), fTmean(-1.) {}
        ParticleData(G4int count, G4double ekin, G4double emin, G4double emax, G4double meanLife)
          : fCount(count), fEmean(ekin), fEmin(emin), fEmax(emax), fTmean(meanLife)
        {}
        G4int fCount;
        G4double fEmean;
        G4double fEmin;
        G4double fEmax;
        G4double fTmean;
    };

  private:
    G4ParticleDefinition* fParticle = nullptr;
    G4double fEkin = 0.;

    std::map<G4String, ParticleData> fParticleDataMap;
    G4int fDecayCount = 0, fTimeCount = 0;
    G4double fEkinTot[3] = {0., DBL_MAX, 0.};
    G4double fPbalance[3] = {0., DBL_MAX, 0.};
    G4double fEventTime[3] = {0., DBL_MAX, 0.};
    G4double fPrimaryTime = 0.;
    G4double fEvisEvent[3] = {0., DBL_MAX, 0.};

  private:
    struct ActivityData
    {
        ActivityData() : fNlife_t1(0), fNlife_t2(0), fNdecay_t1t2(0) {}
        ActivityData(G4int n1, G4int n2, G4int nd) : fNlife_t1(n1), fNlife_t2(n2), fNdecay_t1t2(nd)
        {}
        G4int fNlife_t1;
        G4int fNlife_t2;
        G4int fNdecay_t1t2;
    };

    std::map<G4String, ActivityData> fActivityMap;
    G4double fTimeWindow1 = 0., fTimeWindow2 = 0.;
};

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

#endif
