//
// ********************************************************************
// * License and Disclaimer                                           *
// *                                                                  *
// * The  Geant4 software  is  copyright of the Copyright Holders  of *
// * the Geant4 Collaboration.  It is provided  under  the terms  and *
// * conditions of the Geant4 Software License,  included in the file *
// * LICENSE and available at  http://cern.ch/geant4/license .  These *
// * include a list of copyright holders.                             *
// *                                                                  *
// * Neither the authors of this software system, nor their employing *
// * institutes,nor the agencies providing financial support for this *
// * work  make  any representation or  warranty, express or implied, *
// * regarding  this  software system or assume any liability for its *
// * use.  Please see the license in the file  LICENSE  and URL above *
// * for the full disclaimer and the limitation of liability.         *
// *                                                                  *
// * This  code  implementation is the result of  the  scientific and *
// * technical work of the GEANT4 collaboration.                      *
// * By using,  copying,  modifying or  distributing the software (or *
// * any work based  on the software)  you  agree  to acknowledge its *
// * use  in  resulting  scientific  publications,  and indicate your *
// * acceptance of all terms of the Geant4 Software license.          *
// ********************************************************************
//
/// \file PhysicsListMessenger.cc
/// \brief Implementation of the PhysicsListMessenger class

#include "PhysicsListMessenger.hh"

#include "PhysicsList.hh"

#include "G4UIcmdWithABool.hh"
#include "G4UIcmdWithADoubleAndUnit.hh"
#include "G4UIcmdWithAString.hh"
#include "G4UIdirectory.hh"

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

PhysicsListMessenger::PhysicsListMessenger(PhysicsList* physL)
  : G4UImessenger(), fPhysList(physL), fPhysDir(0), fSRTypeCmd(0)
{
  fPhysDir = new G4UIdirectory("/testem/phys/");
  fPhysDir->SetGuidance("physics list commands");

  fSRTypeCmd = new G4UIcmdWithABool("/testem/phys/analyticSR", this);
  fSRTypeCmd->SetGuidance("choose analytic synchrotron radiation");
  fSRTypeCmd->SetParameterName("SRType", true);
  fSRTypeCmd->SetDefaultValue(true);

  fXrayReflectionRoughnessCmd =
    new G4UIcmdWithADoubleAndUnit("/testem/phys/SetXrayReflectionRoughness", this);
  fXrayReflectionRoughnessCmd->SetGuidance(
    "Set the XrayReflection surface roughness, typically between 0 and 10 "
    "nm");
  fXrayReflectionRoughnessCmd->SetParameterName("XrayReflSurfRoughness", false);
  fXrayReflectionRoughnessCmd->SetRange("XrayReflSurfRoughness>=0");
  fXrayReflectionRoughnessCmd->SetUnitCategory("Length");
  fXrayReflectionRoughnessCmd->AvailableForStates(G4State_PreInit, G4State_Idle);
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

void PhysicsListMessenger::SetNewValue(G4UIcommand* command, G4String newValue)
{
  if (command == fSRTypeCmd) {
    fPhysList->SetAnalyticSR(fSRTypeCmd->GetNewBoolValue(newValue));
  }

  if (command == fXrayReflectionRoughnessCmd) {
    fPhysList->SetXrayReflectionRoughness(fXrayReflectionRoughnessCmd->GetNewDoubleValue(newValue));
  }
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......
