<?php
/**
 * Prune file cache for pages, objects, resources, etc.
 *
 * @license GPL-2.0-or-later
 * @file
 * @ingroup Maintenance
 */

use MediaWiki\Maintenance\Maintenance;

// @codeCoverageIgnoreStart
require_once __DIR__ . '/Maintenance.php';
// @codeCoverageIgnoreEnd

/**
 * Maintenance script that prunes file cache for pages, objects, resources, etc.
 *
 * @ingroup Maintenance
 */
class PruneFileCache extends Maintenance {

	/** @var int */
	protected $minSurviveTimestamp;

	public function __construct() {
		parent::__construct();
		$this->addDescription( 'Delete file cache files older than "agedays"' );
		$this->addOption( 'agedays', 'How many days old files must be in order to delete', true, true );
		$this->addOption( 'subdir', 'Prune one $wgFileCacheDirectory subdirectory name', false, true );
	}

	public function execute() {
		global $wgUseFileCache, $wgFileCacheDirectory;

		if ( !$wgUseFileCache ) {
			$this->fatalError( "Nothing to do -- \$wgUseFileCache is disabled." );
		}

		$age = $this->getOption( 'agedays' );
		if ( !ctype_digit( $age ) ) {
			$this->fatalError( "Non-integer 'age' parameter given." );
		}
		// Delete items with a TS older than this
		$this->minSurviveTimestamp = time() - ( 86400 * $age );

		$dir = $wgFileCacheDirectory;
		if ( !is_dir( $dir ) ) {
			$this->fatalError( "Nothing to do -- \$wgFileCacheDirectory directory not found." );
		}

		$subDir = $this->getOption( 'subdir' );
		if ( $subDir !== null ) {
			if ( !is_dir( "$dir/$subDir" ) ) {
				$this->fatalError( "The specified subdirectory `$subDir` does not exist." );
			}
			$this->output( "Pruning `$dir/$subDir` directory...\n" );
			$this->prune_directory( "$dir/$subDir", 'report' );
			$this->output( "Done pruning `$dir/$subDir` directory\n" );
		} else {
			$this->output( "Pruning `$dir` directory...\n" );
			// Note: don't prune things like .cdb files on the top level!
			$this->prune_directory( $dir, 'report' );
			$this->output( "Done pruning `$dir` directory\n" );
		}
	}

	/**
	 * @param string $dir
	 * @param string|bool $report Use 'report' to report the directories being scanned
	 */
	protected function prune_directory( $dir, $report = false ) {
		$tsNow = time();
		$dirHandle = opendir( $dir );
		// phpcs:ignore Generic.CodeAnalysis.AssignmentInCondition.FoundInWhileCondition
		while ( ( $file = readdir( $dirHandle ) ) !== false ) {
			// Skip ".", "..", and also any dirs or files like ".svn" or ".htaccess"
			if ( $file[0] != "." ) {
				// absolute
				$path = $dir . '/' . $file;
				if ( is_dir( $path ) ) {
					if ( $report === 'report' ) {
						$this->output( "Scanning `$path`...\n" );
					}
					$this->prune_directory( $path );
				} else {
					$mts = filemtime( $path );
					// Check the file extension against known cache types
					if ( $mts < $this->minSurviveTimestamp
						&& preg_match( '/\.(?:html|cache)(?:\.gz)?$/', $file )
						&& unlink( $path )
					) {
						$daysOld = round( ( $tsNow - $mts ) / 86400, 2 );
						$this->output( "Deleted `$path` [days=$daysOld]\n" );
					}
				}
			}
		}
		closedir( $dirHandle );
	}
}

// @codeCoverageIgnoreStart
$maintClass = PruneFileCache::class;
require_once RUN_MAINTENANCE_IF_MAIN;
// @codeCoverageIgnoreEnd
