package tar

import (
	"archive/tar"
	"compress/gzip"
	"errors"
	"io"
	"os"
	"path/filepath"

	errors2 "github.com/influxdata/influxdb/pkg/errors"
)

// Untar takes a destination path and a reader; a tar reader loops over the tarfile
// creating the file structure at 'dir' along the way, and writing any files
func Untar(dir string, r io.Reader) (rErr error) {

	gzr, err := gzip.NewReader(r)
	if err != nil {
		return err
	}
	defer errors2.Capture(&rErr, gzr.Close)()

	tr := tar.NewReader(gzr)

	for {
		header, err := tr.Next()

		switch {

		// if no more files are found return
		case errors.Is(err, io.EOF):
			return nil

		// return any other error
		case err != nil:
			return err

		// if the header is nil, just skip it (not sure how this happens)
		case header == nil:
			continue
		}

		// the target location where the dir/file should be created
		target := filepath.Join(dir, header.Name)

		// the following switch could also be done using fi.Mode(), not sure if there
		// a benefit of using one vs. the other.
		// fi := header.FileInfo()

		// check the file type
		switch header.Typeflag {

		// if its a dir and it doesn't exist create it
		case tar.TypeDir:
			if _, err := os.Stat(target); err != nil {
				if err := os.MkdirAll(target, 0755); err != nil {
					return err
				}
			}

		// if it's a file create it
		case tar.TypeReg:
			if err := untarFile(target, tr, header); err != nil {
				return err
			}
		}
	}
}

func untarFile(target string, tr *tar.Reader, header *tar.Header) (rErr error) {
	f, err := os.OpenFile(target, os.O_CREATE|os.O_RDWR, os.FileMode(header.Mode))
	if err != nil {
		return err
	}
	defer errors2.Capture(&rErr, f.Close)()

	// copy over contents
	if _, err := io.Copy(f, tr); err != nil {
		return err
	}
	return nil
}
