// Copyright 2016 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

/** @fileoverview Runs tests for the settings menu. */

// clang-format off
import {flush} from 'chrome://resources/polymer/v3_0/polymer/polymer_bundled.min.js';
import type {SettingsMenuElement, SettingsRoutes} from 'chrome://settings/settings.js';
import {AutofillSettingsReferrer, resetRouterForTesting, loadTimeData, MetricsBrowserProxyImpl, resetPageVisibilityForTesting, Router} from 'chrome://settings/settings.js';
import {assertEquals, assertFalse, assertTrue} from 'chrome://webui-test/chai_assert.js';
import {isVisible, microtasksFinished} from 'chrome://webui-test/test_util.js';
import {flushTasks} from 'chrome://webui-test/polymer_test_util.js';

import {TestMetricsBrowserProxy} from './test_metrics_browser_proxy.js';

// clang-format on

suite('SettingsMenu', function() {
  let settingsMenu: SettingsMenuElement;
  let routes: SettingsRoutes;
  let metricsBrowserProxy: TestMetricsBrowserProxy;

  function createSettingsMenu() {
    routes = Router.getInstance().getRoutes();
    document.body.innerHTML = window.trustedTypes!.emptyHTML;
    settingsMenu = document.createElement('settings-menu');
    document.body.appendChild(settingsMenu);
    flush();
  }

  setup(function() {
    metricsBrowserProxy = new TestMetricsBrowserProxy();
    MetricsBrowserProxyImpl.setInstance(metricsBrowserProxy);
    createSettingsMenu();
  });

  teardown(function() {
    resetPageVisibilityForTesting();
  });

  // Test that navigating via the paper menu always clears the current
  // search URL parameter.
  test('clearsUrlSearchParam', async () => {
    const urlParams = new URLSearchParams('search=foo');
    Router.getInstance().navigateTo(
        Router.getInstance().getRoutes().BASIC, urlParams);
    assertEquals(
        urlParams.toString(),
        Router.getInstance().getQueryParameters().toString());
    settingsMenu.$.people.click();
    await settingsMenu.$.menu.updateComplete;
    assertEquals('', Router.getInstance().getQueryParameters().toString());
  });

  test('openResetSection', function() {
    Router.getInstance().navigateTo(routes.RESET);
    const selector = settingsMenu.$.menu;
    assertTrue(!!selector.selected);
    assertEquals('/reset', selector.selected.toString());
  });

  test('navigateToAnotherSection', function() {
    Router.getInstance().navigateTo(routes.RESET);
    const selector = settingsMenu.$.menu;
    assertTrue(!!selector.selected);
    assertEquals('/reset', selector.selected.toString());

    Router.getInstance().navigateTo(routes.PEOPLE);
    flush();

    assertTrue(!!selector.selected);
    assertEquals('/people', selector.selected.toString());
  });

  test('navigateToBasic', function() {
    Router.getInstance().navigateTo(routes.RESET);
    const selector = settingsMenu.$.menu;
    assertTrue(!!selector.selected);
    assertEquals('/reset', selector.selected.toString());

    Router.getInstance().navigateTo(routes.BASIC);
    flush();

    // BASIC has no sub page selected.
    assertFalse(!!selector.selected);
  });

  test('noExperimental', async function() {
    loadTimeData.overrideValues({showAiPage: false});
    resetRouterForTesting();
    createSettingsMenu();
    await flushTasks();

    const entry = settingsMenu.shadowRoot!.querySelector('a[href=\'/ai\']');
    assertTrue(!!entry);
    assertFalse(isVisible(entry));
  });

  test('navigateToExperimental', async function() {
    loadTimeData.overrideValues({showAiPage: true});
    resetRouterForTesting();
    createSettingsMenu();
    Router.getInstance().navigateTo(routes.AI);
    await flushTasks();

    const entry = settingsMenu.shadowRoot!.querySelector('a[href=\'/ai\']');
    assertTrue(!!entry);
    assertTrue(isVisible(entry));

    const selector = settingsMenu.$.menu;
    assertTrue(!!selector.selected);
    assertEquals('/ai', selector.selected.toString());
  });

  test('pageVisibility', function() {
    function assertPagesHidden(expectedHidden: boolean) {
      const ids = [
        'accessibility', 'appearance',
        // <if expr="not is_chromeos">
        'defaultBrowser',
        // </if>
        'downloads', 'languages', 'onStartup', 'people', 'performance', 'reset',
        // <if expr="not is_chromeos">
        'system',
        // </if>
      ];

      for (const id of ids) {
        assertEquals(
            expectedHidden,
            settingsMenu.shadowRoot!.querySelector<HTMLElement>(
                                        `#${id}`)!.hidden);
      }
    }

    // The default pageVisibility should not cause menu items to be hidden.
    assertPagesHidden(false);

    // Set the visibility of the pages under test to "false".
    resetPageVisibilityForTesting({
      a11y: false,
      appearance: false,
      defaultBrowser: false,
      downloads: false,
      languages: false,
      onStartup: false,
      people: false,
      performance: false,
      reset: false,
      system: false,
    });
    createSettingsMenu();

    // Now, the menu items should be hidden.
    assertPagesHidden(true);
  });

  test('aiPageMenuClick', async function() {
    loadTimeData.overrideValues({
      showAiPage: true,
    });
    resetRouterForTesting();
    createSettingsMenu();
    await flushTasks();

    const entry =
        settingsMenu.shadowRoot!.querySelector<HTMLElement>('a[href=\'/ai\']');
    assertTrue(!!entry);
    assertTrue(isVisible(entry));

    // Ensure UMA is logged.
    entry.click();
    assertEquals(
        'SettingsMenu_AiPageEntryPointClicked',
        await metricsBrowserProxy.whenCalled('recordAction'));

    await microtasksFinished();
    assertEquals(routes.AI, Router.getInstance().getCurrentRoute());
  });

  test('autofillPageMenuClick', async function() {
    const entry = settingsMenu.shadowRoot!.querySelector<HTMLElement>(
      'a[href=\'/autofill\']');
    assertTrue(!!entry);
    assertTrue(isVisible(entry));

    entry.click();
    const [histogramName, referrer] = await metricsBrowserProxy.whenCalled(
        'recordAutofillSettingsReferrer');
    assertEquals(
        'Autofill.AutofillAndPasswordsSettingsPage.VisitReferrer',
        histogramName);
    assertEquals(AutofillSettingsReferrer.SETTINGS_MENU, referrer);

    await microtasksFinished();
    assertEquals(routes.AUTOFILL, Router.getInstance().getCurrentRoute());
  });

  test('yourSavedInfoHiddenWhenFeatureDisabled', async function() {
    loadTimeData.overrideValues({enableYourSavedInfoSettingsPage: false});
    resetRouterForTesting();
    createSettingsMenu();
    await flushTasks();

    const entry = settingsMenu.shadowRoot!.querySelector<HTMLElement>(
        'a[href=\'/yourSavedInfo\']');
    assertTrue(!!entry);
    assertFalse(isVisible(entry));
  });

  test('yourSavedInfoMenuItemClick', async function() {
    loadTimeData.overrideValues({enableYourSavedInfoSettingsPage: true});
    resetRouterForTesting();
    createSettingsMenu();
    await flushTasks();

    const entry = settingsMenu.shadowRoot!.querySelector<HTMLElement>(
        'a[href=\'/yourSavedInfo\']');
    assertTrue(!!entry);
    assertTrue(isVisible(entry));

    entry.click();
    await microtasksFinished();

    const selector = settingsMenu.$.menu;
    assertTrue(!!selector.selected);
    assertEquals('/yourSavedInfo', selector.selected.toString());
    assertEquals(
        routes.YOUR_SAVED_INFO, Router.getInstance().getCurrentRoute());
  });
});
