/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#include "panelElements.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <support.h>
#include <visu_object.h>
#include <visu_tools.h>
#include <visu_data.h>
#include <visu_elements.h>
#include <extraGtkFunctions/gtk_colorComboBoxWidget.h>
#include <extraGtkFunctions/gtk_elementComboBox.h>
#include <extraGtkFunctions/gtk_toolPanelWidget.h>
#include <coreTools/toolColor.h>
#include <openGLFunctions/light.h>

#include "gtkAtomic.h"
#include "gtkSpin.h"

/**
 * SECTION: panelElements
 * @short_description: The tab where #VisuElement characteristics can
 * be tuned.
 */

/**
 * panelElements:
 *
 * A pointer on the #ToolPanel that host the elements chooser.
 */
static GtkWidget *panelElements;

/* Functions used to initialise the
 * specific area in the panel of elements.
 */
static VisuToolsInitFunc listInitRendenringGtkPanelFunc[] = {
  panelElementAtomicInit_gtkPanel,
  panelElementSpinInit_gtkPanel,
  (VisuToolsInitFunc)0};

/* sensitive widgets */
GtkWidget *vBoxElements;
GtkWidget *renderingMethodElements;
static GtkWidget *elementsComboBox;
static GtkWidget *checkRendered;
static GtkWidget *checkMasked;
GtkWidget *widgetVisuElementColor;
gint sortVisuElements(gpointer a, gpointer b);

/* Structure to store the hooks that are required for
   the specific zone for elements. */
struct GtkRenderingInterface_struct
{
/*   setSensitiveFunc sensitiveInterface; */
  panelElements_changeFunc newElementInterface;
  VisuGtkWidgetFunc createInterface;
};
typedef struct GtkRenderingInterface_struct GtkRenderingInterface;
GHashTable *listOfRenderingInterfaces;
static panelElements_changeFunc updateRenderingMethodResources;

/* Local methods. */
static void createInteriorElements(GtkWidget *toolPanel);
static void updateRenderingSpecificPart(VisuRendering *method);
void createCallBacksElements();
void setInitialValuesElements();
char *getDefaultLabelOfElement(VisuElement *element);
/* It adds the interface widget in the subPanel. The variable
   renderingMethodElements pints then to this widget. If
   renderingMethodElements has already a target, this target is
   freed via gtk_widget_destroy. */
static void changeRenderingMethodInterface(GtkWidget* interface);
/* It allows the client to set the method use to update
   user defined values. */
static void setElementChangedFunc(panelElements_changeFunc func);
/* It reads the value of the interface methods. */
static void getRenderingInterface(VisuRendering *method,
				  panelElements_changeFunc *change,
				  VisuGtkWidgetFunc *create);

/* Callbacks */
static void onElementSelected(ElementComboBox *combo, GList *elements, gpointer data);
static gboolean onElementChanged(GSignalInvocationHint *ihint, guint n_param_values,
                                 const GValue *param_values, gpointer data);
static void resourcesChanged(GObject *object, VisuData *dataObj, gpointer data);
static void colorVisuElementChanged(ColorComboBox *colorWd, ToolColor *color, gpointer userData);
static void checkRenderedChanged(GtkToggleButton *button, gpointer data);
static void checkMaskedChanged(GtkToggleButton *button, gpointer data);
static void refreshComboElements(GObject *obj, VisuData *dataObj, gpointer userData);
static void onRenderingMethodChanged(GObject *obj, VisuRendering *method,
				     gpointer userData);
static void onColorChanged(ColorComboBox *colorComboBox, guint colorId, gpointer data);
static void onMaterialChanged(ColorComboBox *colorComboBox, Material mat, gpointer data);

/**
 * panelElementsInit: (skip)
 *
 * Should be used in the list declared in externalModules.h to be loaded by
 * V_Sim on start-up. This routine will create the #ToolPanel where the element
 * stuff can be done, such as choosing a colour, setting the radius of
 * spheres...
 *
 * Returns: a newly created #ToolPanel object.
 */
ToolPanel* panelElementsInit()
{
  gchar *cl = _("Set elements caracteristics");
  gchar *tl = _("Elements");
  int i;

  panelElements = toolPanelNew_withIconFromPath("Panel_elements", cl, tl,
						"stock-elements_20.png");

  /* Create the list of the available rendering method
     interfaces. */
  listOfRenderingInterfaces = g_hash_table_new_full(g_direct_hash,
						    g_direct_equal,
						    NULL, NULL);
  for (i = 0; listInitRendenringGtkPanelFunc[i]; i++)
    listInitRendenringGtkPanelFunc[i]();

  createInteriorElements(panelElements);
  toolPanelSet_dockable(TOOL_PANEL(panelElements), TRUE);

  if (!panelElements)
    return (ToolPanel*)0;

  /* Create the callbacks of all the sensitive widgets. */
  createCallBacksElements();

  return TOOL_PANEL(panelElements);
}

static void createInteriorElements(GtkWidget *toolPanel)
{
  GtkWidget *label, *expand, *image;
  GtkWidget *hbox;
  GtkWidget *align;
  GtkWidget *scrollView;
#if GTK_MAJOR_VERSION == 2 && GTK_MINOR_VERSION < 12
  GtkTooltips *tooltips;

  tooltips = gtk_tooltips_new ();
#endif

  scrollView = gtk_scrolled_window_new((GtkAdjustment*)0,
				       (GtkAdjustment*)0);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrollView),
				  GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (scrollView), GTK_SHADOW_NONE);
  vBoxElements = gtk_vbox_new(FALSE, 0);
  gtk_widget_set_sensitive(vBoxElements, FALSE);
  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scrollView), vBoxElements);


  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vBoxElements), hbox, FALSE, FALSE, 5);
  label = gtk_label_new(_("<b>Set caracteristics of: </b>"));
  gtk_widget_set_name(label, "label_head");
  gtk_misc_set_alignment(GTK_MISC(label), 0., 0.5);
  gtk_label_set_use_markup(GTK_LABEL(label), TRUE);
  gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 2);
  /* We create the tree widget that show the methods. */
  elementsComboBox = elementComboBox_new(TRUE, FALSE, _("Element '%s'"));
  gtk_box_pack_start(GTK_BOX(hbox), elementsComboBox, TRUE, TRUE, 2);

  label = gtk_label_new("");
  gtk_label_set_markup(GTK_LABEL(label), _("<b>Standard resources</b>"));
  gtk_misc_set_alignment(GTK_MISC(label), 0., 0.5);
  gtk_widget_set_name(label, "label_head_2");
  gtk_box_pack_start(GTK_BOX(vBoxElements), label, FALSE, FALSE, 5);

  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vBoxElements), hbox, FALSE, FALSE, 0);
/*   label = gtk_label_new(_("Color: ")); */
/*   gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 2); */
  widgetVisuElementColor = colorComboBox_newWithRanges(TRUE);
  gtk_box_pack_start(GTK_BOX(hbox), widgetVisuElementColor, FALSE, FALSE, 2);
  align = gtk_alignment_new(1., 0.5, 0., 0.);
  gtk_box_pack_start(GTK_BOX(hbox), align, TRUE, TRUE, 2);
  checkRendered = gtk_check_button_new_with_label(_("rendered"));
  gtk_container_add(GTK_CONTAINER(align), checkRendered);
  align = gtk_alignment_new(0.5, 0.5, 0., 0.);
  gtk_box_pack_start(GTK_BOX(hbox), align, FALSE, FALSE, 2);
  checkMasked = gtk_check_button_new();
  gtk_container_add(GTK_CONTAINER(align), checkMasked);
  image = create_pixmap((GtkWidget*)0, "stock-masking.png");
  gtk_container_add(GTK_CONTAINER(checkMasked), image);
  gtk_widget_set_tooltip_text(checkMasked,
		              _("Make nodes sensitive to the masking effect of planes."));

  expand = colorComboBoxGet_rangesWidgets(COLOR_COMBOX(widgetVisuElementColor));
  gtk_box_pack_start(GTK_BOX(vBoxElements), expand, FALSE, FALSE, 0);

  label = gtk_label_new(_("<b>Rendering specific resources</b>"));
  gtk_widget_set_name(label, "label_head_2");
  gtk_misc_set_alignment(GTK_MISC(label), 0., 0.5);
  gtk_label_set_use_markup(GTK_LABEL(label), TRUE);
  gtk_box_pack_start(GTK_BOX(vBoxElements), label, FALSE, FALSE, 5);

  DBG_fprintf(stderr, "Panel Elements: create the specific widgets.\n");
  renderingMethodElements = (GtkWidget*)0;
  updateRenderingSpecificPart(visu_object_getRendering(VISU_INSTANCE));

  gtk_widget_show_all(scrollView);

  gtk_container_add(GTK_CONTAINER(toolPanel), scrollView);
}

void createCallBacksElements()
{
  g_signal_connect(G_OBJECT(elementsComboBox), "element-selected",
                   G_CALLBACK(onElementSelected), (gpointer)0);
  g_signal_connect(G_OBJECT(widgetVisuElementColor), "color-selected",
		   G_CALLBACK(colorVisuElementChanged), (gpointer)0);
  g_signal_connect(G_OBJECT(widgetVisuElementColor), "material-value-changed",
		   G_CALLBACK(onMaterialChanged), (gpointer)0);
  g_signal_connect(G_OBJECT(widgetVisuElementColor), "color-value-changed",
		   G_CALLBACK(onColorChanged), (gpointer)0);
  g_signal_connect(G_OBJECT(checkRendered), "toggled",
		   G_CALLBACK(checkRenderedChanged), (gpointer)0);
  g_signal_connect(G_OBJECT(checkMasked), "toggled",
		   G_CALLBACK(checkMaskedChanged), (gpointer)0);
  /* Connect the signal emitted by visu when a new file is loaded. */
  g_signal_connect(VISU_INSTANCE, "dataReadyForRendering",
		   G_CALLBACK(refreshComboElements), (gpointer)0);
  /* Connect the signal emitted by visu when
     a new rendering method is selected. */
  g_signal_connect(VISU_INSTANCE, "renderingChanged",
		   G_CALLBACK(onRenderingMethodChanged), (gpointer)0);
  /* Connect the signal emitted by visu when
     a new rendering method is selected. */
  g_signal_connect(VISU_INSTANCE, "resourcesLoaded",
		   G_CALLBACK(resourcesChanged), (gpointer)0);
  g_signal_add_emission_hook(g_signal_lookup("ElementRenderedChanged", VISU_ELEMENT_TYPE),
                             0, onElementChanged, (gpointer)0, (GDestroyNotify)0);
  g_signal_add_emission_hook(g_signal_lookup("ElementMaterialChanged", VISU_ELEMENT_TYPE),
                             0, onElementChanged, (gpointer)0, (GDestroyNotify)0);
  g_signal_add_emission_hook(g_signal_lookup("ElementPlaneChanged", VISU_ELEMENT_TYPE),
                             0, onElementChanged, (gpointer)0, (GDestroyNotify)0);
}

/*****************/
/* Miscellaneous */
/*****************/
gint sortVisuElements(gpointer a, gpointer b)
{
  return strcmp(((VisuElement*)a)->name, ((VisuElement*)b)->name);
}

/*************/
/* Callbacks */
/*************/
static void refreshComboElements(GObject *obj _U_, VisuData *dataObj, gpointer userData _U_)
{
  if (dataObj)
    {
      DBG_fprintf(stderr,"Panel Element : creating %d labels for the"
		  " combobox of VisuElements.\n", dataObj->ntype);
      gtk_widget_set_sensitive(vBoxElements, TRUE);
    }
  else
    {
      DBG_fprintf(stderr, "Panel Element : there is no element available.\n");
      gtk_widget_set_sensitive(vBoxElements, FALSE);
    }
}
static void updateRenderingSpecificPart(VisuRendering *method)
{
  panelElements_changeFunc change;
  VisuGtkWidgetFunc create;

  change = (panelElements_changeFunc)0;
  create = (VisuGtkWidgetFunc)0;
  if (method)
    getRenderingInterface(method, &change, &create);

  DBG_fprintf(stderr,"Panel Element: set the gtk interface for"
	      " the method '%s'.\n", visu_rendering_getName(method, FALSE));
  setElementChangedFunc(change);
  if (create)
    changeRenderingMethodInterface(create());
  else
    changeRenderingMethodInterface((GtkWidget*)0);
}
static void onRenderingMethodChanged(GObject *obj _U_, VisuRendering *method,
				     gpointer userData _U_)
{
  updateRenderingSpecificPart(method);
}
static gboolean onElementChanged(GSignalInvocationHint *ihint _U_, guint n_param_values _U_,
                                 const GValue *param_values, gpointer data _U_)
{
  GList *elements;
  VisuElement *ele;

  elements = elementComboBoxGet_selectedElement(ELEMENT_COMBOX(elementsComboBox));
  if (!elements)
    return TRUE;

  ele = VISU_ELEMENT(g_value_get_object(param_values));
  DBG_fprintf(stderr, "Panel Elements: element '%s' has been modified.\n", ele->name);
  if (elements->data != (gpointer)ele)
    {
      g_list_free(elements);
      return TRUE;
    }
  g_list_free(elements);

  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkRendered),
                               visu_element_getRendered(ele));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkMasked),
                               visu_element_getSensitiveToPlanes(ele));
  colorComboBoxSet_color(COLOR_COMBOX(widgetVisuElementColor), ele->rgb, FALSE);
  colorComboBoxSet_material(COLOR_COMBOX(widgetVisuElementColor),
                            ele->material, FALSE);
  return TRUE;
}
static void onElementSelected(ElementComboBox *combo _U_, GList *elements,
                              gpointer data _U_)
{
  VisuElement *ele;

  DBG_fprintf(stderr, "Panel Elements: set new list of selected elements to %p.\n",
              (gpointer)elements);
  if (elements)
    {
      ele = (VisuElement*)elements->data;
      DBG_fprintf(stderr, "Panel Element: set the selected VisuElement to '%s'.\n",
                  ele->name);
      colorComboBoxSet_color(COLOR_COMBOX(widgetVisuElementColor), ele->rgb, FALSE);
      colorComboBoxSet_material(COLOR_COMBOX(widgetVisuElementColor),
                                ele->material, FALSE);
      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkRendered),
				   visu_element_getRendered(ele));
      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkMasked),
				   visu_element_getSensitiveToPlanes(ele));
    }
  /* Force refresh on the specific area. */
  DBG_fprintf(stderr, "Panel Element: update the render widgets for %p.\n",
              (gpointer)elements);
  if (updateRenderingMethodResources)
    updateRenderingMethodResources(elements);
}
static void colorVisuElementChanged(ColorComboBox *colorWd _U_, ToolColor *color,
			     gpointer userData _U_)
{
  int res;
  gboolean refresh;
  VisuData *dataObj;
  GList *elements, *tmp;

  DBG_fprintf(stderr, "Panel Elements: Catch 'color-selected' signal (%p).\n",
              (gpointer) color);

  /* Get the selected elements. */
  elements = elementComboBoxGet_selectedElement(ELEMENT_COMBOX(elementsComboBox));
  g_return_if_fail(elements);
    
  /* If currentSelectedElement is NULL we apply the changes
     on all elements. */
  dataObj = toolPanelGet_visuData(TOOL_PANEL(panelElements));

  refresh = FALSE;
  for (tmp = elements; tmp; tmp = g_list_next(tmp))
    {
      res = visu_element_setAllRGBValues((VisuElement*)tmp->data, color->rgba);
      if (res > 0)
        visu_data_createNodes(dataObj, (VisuElement*)tmp->data);
      refresh = refresh || (res != 0);
    }
  g_list_free(elements);
  if (refresh)
    {
      DBG_fprintf(stderr, "Panel Elements: color change, refresh.\n");
      VISU_ADD_REDRAW;
    }
}
static void onColorChanged(ColorComboBox *colorComboBox, guint colorId _U_,
			   gpointer data _U_)
{
  int res;
  gboolean refresh;
  VisuData *dataObj;
  float *rgbVal;
  GList *elements, *tmp;

  /* Get the selected elements. */
  elements = elementComboBoxGet_selectedElement(ELEMENT_COMBOX(elementsComboBox));
  g_return_if_fail(elements);
    
  /* Get the values. */
  rgbVal = colorComboBoxGet_color(colorComboBox);

  /* If currentSelectedElement is NULL we apply the changes
     on all elements. */
  dataObj = toolPanelGet_visuData(TOOL_PANEL(panelElements));

  refresh = FALSE;
  for (tmp = elements; tmp; tmp = g_list_next(tmp))
    {
      res = visu_element_setAllRGBValues((VisuElement*)tmp->data, rgbVal);
      if (res > 0)
        visu_data_createNodes(dataObj, (VisuElement*)tmp->data);
      refresh = refresh || (res != 0);
    }
  g_list_free(elements);
  if (refresh)
    {
      DBG_fprintf(stderr, "Panel Elements: RGB change, refresh.\n");
      VISU_ADD_REDRAW;
    }
  g_free(rgbVal);
}
static void onMaterialChanged(ColorComboBox *colorComboBox, Material mat _U_,
			      gpointer data _U_)
{
  int res;
  int refresh;
  VisuData *dataObj;
  float *matVal;
  GList *elements, *tmp;

  /* Get the selected elements. */
  elements = elementComboBoxGet_selectedElement(ELEMENT_COMBOX(elementsComboBox));
  g_return_if_fail(elements);
    
  /* Get the values. */
  matVal = colorComboBoxGet_material(colorComboBox);

  /* If currentSelectedElement is NULL we apply the changes
     on all elements. */
  dataObj = toolPanelGet_visuData(TOOL_PANEL(panelElements));

  refresh = FALSE;
  for (tmp = elements; tmp; tmp = g_list_next(tmp))
    {
      res = visu_element_setAllMaterialValues((VisuElement*)tmp->data, matVal);
      if (res > 0)
        visu_data_createNodes(dataObj, (VisuElement*)tmp->data);
      refresh = refresh || (res != 0);
    }
  g_list_free(elements);
  if (refresh)
    {
      DBG_fprintf(stderr, "Panel Elements: material change, refresh.\n");
      VISU_ADD_REDRAW;
    }
  g_free(matVal);
}
static void checkMaskedChanged(GtkToggleButton *button, gpointer data _U_)
{
  gboolean refresh;
  VisuData *dataObj;
  GList *elements, *tmp;

  /* Get the selected elements. */
  elements = elementComboBoxGet_selectedElement(ELEMENT_COMBOX(elementsComboBox));
  g_return_if_fail(elements);

  refresh = FALSE;
  for (tmp = elements; tmp; tmp = g_list_next(tmp))
    refresh =
      visu_element_setSensitiveToPlanes((VisuElement*)tmp->data,
                                        gtk_toggle_button_get_active(button)) ||
      refresh;
  g_list_free(elements);
  if (refresh)
    {
      dataObj = toolPanelGet_visuData(TOOL_PANEL(panelElements));
      refresh = FALSE;
      visu_data_emitAskForShowHideNodes(dataObj, &refresh);
      if (refresh)
	{
	  visu_data_emitNodeRenderedChange(dataObj);
	  visu_data_createAllNodes(dataObj);
	  VISU_ADD_REDRAW;
	}
    }
}
static void checkRenderedChanged(GtkToggleButton *button, gpointer data _U_)
{
  int res;
  gboolean refresh;
  VisuData *dataObj;
  GList *elements, *tmp;

  /* Get the selected elements. */
  elements = elementComboBoxGet_selectedElement(ELEMENT_COMBOX(elementsComboBox));
  g_return_if_fail(elements);

  dataObj = toolPanelGet_visuData(TOOL_PANEL(panelElements));

  refresh = FALSE;
  for (tmp = elements; tmp; tmp = g_list_next(tmp))
    {
      res = visu_element_setRendered((VisuElement*)tmp->data,
                                     gtk_toggle_button_get_active(button));
      if (res)
	{
	  refresh = TRUE;
	  visu_data_createNodes(dataObj, (VisuElement*)tmp->data);
	}
    }
  g_list_free(elements);
  if (refresh)
    VISU_ADD_REDRAW;
}
static void resourcesChanged(GObject *object _U_, VisuData *dataObj _U_,
			     gpointer data _U_)
{
  GList *elements;

  /* The resources loading could have changed ToolColor of the element,
     so we check all the list. */
  DBG_fprintf(stderr, "Panel Elements: catch the 'resourcesChanged'"
	      " signal, checking color pointers.\n");

  /* Get the selected iter. */
  elements = elementComboBoxGet_selectedElement(ELEMENT_COMBOX(elementsComboBox));
  if (!elements)
    return;

  /* Force refresh on the specific area. */
  if (updateRenderingMethodResources)
    updateRenderingMethodResources(elements);
  
  g_list_free(elements);
}
/**
 * panelElementsGet_selected:
 *
 * This method is used to get a list of selected #VisuElement from the
 * element selector of this panel.
 *
 * Since: 3.6
 *
 * Returns: a list of #VisuElement, the list should be freed after use
 * by g_list_free().
 */
GList* panelElementsGet_selected()
{
  return elementComboBoxGet_selectedElement(ELEMENT_COMBOX(elementsComboBox));
}


/* It allows the client to set the method use to update
   user defined values. */
static void setElementChangedFunc(panelElements_changeFunc func)
{
  updateRenderingMethodResources = func;
}
/* It adds the interface widget in the subPanel. The variable
   renderingMethodElements pints then to this widget. If
   renderingMethodElements has already a target, this target is
   freed via gtk_widget_destroy. If interface is a null pointer
   the previous renderingMethodElements is removed, but
   nothing replaces it. */
static void changeRenderingMethodInterface(GtkWidget* interface)
{
  GList *elements;

  DBG_fprintf(stderr, "Panel Element: caught 'resourcesLoaded' signal.\n");
  if (renderingMethodElements)
    {
      /* note that gtk_destroy automatically remove
	 renderingMethodElements for its container, so
	 a call to gtk_container_remove is not necessary. */
      DBG_fprintf(stderr, "Panel Element: removing old rendering specific widget.\n");
      gtk_widget_destroy(renderingMethodElements);
    }

  renderingMethodElements = interface;

  if (interface)
    {
      gtk_box_pack_start(GTK_BOX(vBoxElements), interface,
			 FALSE, FALSE, 5);
      gtk_box_reorder_child(GTK_BOX(vBoxElements), interface, 6);
      gtk_widget_show(interface);
    }

  /* Force refresh on the specific area. */
  if (updateRenderingMethodResources)
    {
      elements = elementComboBoxGet_selectedElement(ELEMENT_COMBOX(elementsComboBox));
      updateRenderingMethodResources(elements);
      g_list_free(elements);
    }
}


/* RenderingMethod gtk interface. */

/**
 * panelElements_setInterfaceMethods:
 * @method: a pointer on the method this interface is associated to ;
 * @change: the method to be called whenever the element selection is cahnged ;
 * @create: the method to be called when the interface must be built.
 *
 * This method allows to initiate the methods to deal with the rendering
 * specific interfaces.
 */
void panelElements_setInterfaceMethods(VisuRendering *method,
				       panelElements_changeFunc change,
				       VisuGtkWidgetFunc create)
{
  GtkRenderingInterface *interface;

  if (!method)
    return;

  /* Search for an already existing GtkRenderingInterface
     for this name. */
  interface = g_hash_table_lookup(listOfRenderingInterfaces,
				  (gpointer)method);
  if (interface)
    {
      interface->newElementInterface = change;
      interface->createInterface = create;
    }
  else
    {
      interface = g_malloc(sizeof(GtkRenderingInterface));
      interface->newElementInterface = change;
      interface->createInterface = create;
      g_hash_table_insert(listOfRenderingInterfaces,
			  (gpointer)method,
			  (gpointer)interface);
    }
}
/* It reads the value of the interface methods. */
static void getRenderingInterface(VisuRendering *method,
				  panelElements_changeFunc *change,
				  VisuGtkWidgetFunc *create)
{
  GtkRenderingInterface *interface;

  *change = (panelElements_changeFunc)0;
  *create = (VisuGtkWidgetFunc)0;
  if (!method)
    return;

  interface = g_hash_table_lookup(listOfRenderingInterfaces,
				  (gpointer)method);
  if (interface)
    {
      *change = interface->newElementInterface;
      *create = interface->createInterface;
    }
}
/**
 * panelElementsGet_static:
 *
 * Retrives a pointer on this #ToolPanel.
 *
 * Returns: a pointer owned by V_Sim.
 */
GtkWidget* panelElementsGet_static()
{
  return panelElements;
}
