/*
  scalc_test_suite.cc, copyright (c) 2006 by Vincent Fourmond: 
  A test suite for the SCalc library
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details (in the COPYING file).
  
*/

#include <scalc.hh>
#include <math.h>
using namespace SCalc;
using namespace std;

int failures = 0;
int tests = 0;

/// The main session for running the tests
Session mainSession;

/// Runs a basic evaluation test:
void assert_eval_equals(const char * expression,
		   double expected = 0.0, 
		   Session * s = &mainSession)
{
  tests += 1; 			// Increase the number of tests:
  cerr << "." ;
  ParserResult * r = s->eval(expression);
  if(r->is_expression() && r->to_expression()->is_valid())
    {
      Expression *e = r->to_expression();
      if(e->evaluable())
	{
	  double value = e->evaluate();
	  if(value != expected)
	    {
	      cerr << "Evaluation of '" << expression 
		   << "' returned <" << value 
		   << "> and should have returned <" 
		   << expected << ">" << endl; 
	      failures ++;
	    }
	}
      else
	{
	  cerr << "Evaluation of '" << expression 
	       << "' not possible " << endl; 
	  failures ++;
	}
      delete e;
    }
  else
    {
      failures ++;
      cerr << "Evaluation of '" << expression 
	   << "' failed to return a valid expression" << endl; 
      delete r;
    }
}


// Asserts that the given expression is evaluable in the given
// context.
void assert_evaluable(const char * expression,
		      Session * s = &mainSession)
{
  tests += 1; 			// Increase the number of tests:
  cerr << "." ;
  ParserResult * r = s->eval(expression);

  if(r->is_expression() && r->to_expression()->is_valid())
    {
      Expression *e = r->to_expression();
      if(! e->evaluable())
	{
	  cerr << "Evaluation of '" << expression 
	       << "' not possible " << endl; 
	  failures ++;
	}
      delete e;
    }
  else
    {
      failures ++;
      cerr << "Evaluation of '" << expression 
	   << "' failed to return a valid expression" << endl; 
      delete r;
    }
}

void assert_not_evaluable(const char * expression,
			  Session * s = &mainSession)
{
  tests += 1; 			// Increase the number of tests:
  cerr << "." ;
  ParserResult * r = s->eval(expression);

  if(r->is_expression() && r->to_expression()->is_valid())
    {
      Expression *e = r->to_expression();
      if(e->evaluable())
	{
	  cerr << "Evaluation of '" << expression 
	       << "' possible when it should not have been" << endl; 
	  failures ++;
	}
      delete e;
    }
  else
    {
      failures ++;
      cerr << "Evaluation of '" << expression 
	   << "' failed to return a valid expression" << endl; 
      delete r;
    }
}

void assert_number_parameters(const char * expression, int nb,
			      Session * s = &mainSession)
{
  tests += 1; 			// Increase the number of tests:
  cerr << "." ;
  ParserResult * r = s->eval(expression);

  if(r->is_expression() && r->to_expression()->is_valid()) {
    Expression *e = r->to_expression();
    if(e->used_variables().size() != nb) {
      std::set<int> vars = e->used_variables();
      cerr << "\nExpression '" << expression << "' holds "
	   << vars.size() 
	   << " variables when it should have held " << nb << endl;
      cerr << "Expression '" << expression << "' was interpreted as '"
	   << e->pretty_print() << "'" << endl;
      for(std::set<int>::iterator i = vars.begin(); i != vars.end(); i++)
	cerr << "  Variable " << *i << " = " << s->varname(*i) << endl;
      failures++;
    }
    delete e;
  }
  else {
    failures ++;
    cerr << "Evaluation of '" << expression 
	 << "' failed to return a valid expression" << endl; 
    delete r;
  }
}

void assert_test(bool val, const char * what) 
{
  tests ++;
  if(! val) {
    cerr << "Test failed : " << what << endl;
    failures++;
  }
}

double biniou(double a) 
{
  return a * a;
}

double bidule(void * a, double x)
{
  return ((long) a) * x;
}

int main()
{
  CFunc * def;

  cout << "Running the test suite for SCalc version " 
       << SCalc::version() << endl;

  // Simple arithmetic operations:
  assert_eval_equals("1", 1);
  assert_eval_equals("2 + 1", 3);
  assert_eval_equals("2 * 7", 14);
  assert_eval_equals("7/2", 3.5);
  assert_eval_equals("1 - 4", -3);
  assert_eval_equals("2 ** 8", 256);

  // Grouping:
  assert_eval_equals("1 - 4 * 4", -15);
  assert_eval_equals("(1 - 4) * 4", -12);
  assert_eval_equals("(1 - 4) ** 4", 81);
  assert_eval_equals("(1 - 4) ** 4 *2", 162);
  assert_eval_equals("(1 - 4) ** (4 *2) ", 6561);

  // Standard functions:
  // Trigonometric functions:
  assert_eval_equals("sin(1)", sin(1.0));
  assert_eval_equals("cos(1)", cos(1.0));
  assert_eval_equals("tan(1)", tan(1.0));
  // Hyperbolic ones:
  assert_eval_equals("exp(1)", exp(1.0));
  assert_eval_equals("cosh(1)", cosh(1.0));
  assert_eval_equals("sinh(1)", sinh(1.0));
  assert_eval_equals("tanh(1)", tanh(1.0));
  // Some othe ones...
  assert_eval_equals("ln(2)", log(2.0));
  assert_eval_equals("sqrt(2)", sqrt(2.0));
  assert_eval_equals("erf(7)", erf(7.0));

  
  // Variables stuff
  mainSession.eval_and_free("x = 1");
  assert_eval_equals("x", 1);
  assert_eval_equals("2 * x", 2);
  mainSession.eval_and_free("x = 2");
  assert_eval_equals("2 * x", 4);
  assert_eval_equals("2 ** (2 *x)", 16);

  // User-defined Functions:
  mainSession.eval_and_free("f: x -> x**2");
  assert_eval_equals("f(1)", 1);
  assert_eval_equals("f(2*2)", 16);
  assert_eval_equals("f(f(x))", 16);

  mainSession.eval_and_free("f: x -> x**x");
  assert_eval_equals("f(1)", 1);
  assert_eval_equals("f(2 + 2)", 256);
  assert_eval_equals("f(f(x))", 256);

  mainSession.eval_and_free("f: x -> sin(x - 1)");
  assert_eval_equals("f(1)", 0);

  // Functions with several paremeters
  mainSession.eval_and_free("g: x,y -> x + 2*y");
  assert_eval_equals("g(1,2)", 5 );
  assert_eval_equals("g(3,2)", 7 );

  // User-defined C functions
  def = new CFunc(&mainSession, "bid", biniou);
  assert_eval_equals("bid(1)", 1);
  assert_eval_equals("bid(2)", 4);
  assert_eval_equals("bid(-2)", 4);

  def = new CFuncParam(&mainSession, "biniou", bidule, (void * ) 3);
  assert_eval_equals("biniou(1)", 3);
  assert_eval_equals("biniou(2)", 6);
  assert_eval_equals("biniou(-2)", -6);

  // Various checks with variables
  assert_evaluable("x");
  assert_not_evaluable("x + y");
  mainSession.eval_and_free("y = 3");
  assert_evaluable("x + y");
  assert_not_evaluable("x + y + z");
  mainSession.unset_var("y");
  assert_not_evaluable("x + y");

  // Now checking that the expressions are finding the right number of
  // parameters.
  assert_number_parameters("a + b + c", 3);
  assert_number_parameters("a + b * c", 3);
  assert_number_parameters("a + b * c**d", 4);
  assert_number_parameters("a + b * (-c**d)", 4);
  assert_number_parameters("a + b * exp(-c**d)", 4);
  assert_number_parameters("a*exp(b*x)", 3);
  assert_number_parameters("a*exp(-b*x)", 3);
  // Same thing, but with underscore in variable names
  assert_number_parameters("a_0*exp(-b_1*x)", 3);
  
  int ab_id = mainSession.register_varname("ab");
  int bc_id = mainSession.register_varname("bc");

  assert_test(ab_id != bc_id, "ab and bc should have different IDs");

  cerr << endl << tests << " tests, " << failures << " failures" << endl;
  return (failures != 0);
}
