# -*- coding: utf-8 -*-

"""
Copyright(C) 2007-2008 INL
Written by Romain Bignon <romain AT inl.fr>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, version 3 of the License.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

$Id: sessions.py 12147 2008-01-11 17:05:31Z romain $
"""

from nevow import inevow
from zope.interface import Interface, implements
from twisted.cred import checkers, error, credentials, portal
from nucentral.client import Service
from twisted.internet import defer
from tools import getBacktrace, trans
from nevow.i18n import _
import re
from nevow import guard
from copy import copy, deepcopy

class ISession(Interface):
    """ Session interface to store it in Twisted Session.
        It ISN'T nevow.inevow.ISession!!!!!!
    """
    pass

def getSession(ctx):
    """ We get a session from an user. If he has already a session object, we get it, in
        other case we copy 'default' session object.
    """
    return inevow.ISession(ctx).getComponent(ISession) or getUserSession(ctx, None)

def getUserSession(ctx, username):
    """ Get the stored session of an user and put it on Twisted session.
        @param ctx the context object.
        @param username [string] name of this user
        @return [Session] a session object
    """

    try:
        # We try to get params from "auth" module.
        session_dict = Service.callService_s("auth", "getParams", username, "nulog")

        if session_dict or username is None:
            session = Session(ctx, session_dict)
        else:
            # If there isn't a dict returned, it is because this user isn't registered.
            # So we use getSession() to copy the default session.
            session = getSession(ctx)

        session.store()
        session.username = username

    except Exception, e:
        # If there is an error (typically, if Nulog is a stand-alone process (without nucentral), or
        # if there isn't any auth module on nucentral).
        # So we return the normal session of this user.
        print getBacktrace()
        session = inevow.ISession(ctx).getComponent(ISession)
        if not session:
            session = Session(ctx).store()

    return session

class Session:
    """ A Session object is attached to an user.
        It contains all pages and fragments parameters.
    """

    ################
    #     Page     #
    ################
    class Page:
        """ A page (useless comment) """

        def __init__(self, ctx, name):

            self.ctx = ctx
            self.name = name
            self.link = str()
            self.frags = [[],[]]
            self.title = str()
            self.args = {}
            self.is_index = False
            self.showstate = False

        def copy(self):

            new = Session.Page(self.ctx, self.name)
            new.frags = copy(self.frags)
            new.title = copy(self.title)
            new.link = copy(self.link)
            new.args = copy(self.args)
            new.is_index = self.is_index
            new.showstate = self.showstate

            return new

        def parse(self, attr, value):

            if attr == 'title':
                self.title = trans(self.ctx, _(value))
            elif attr == 'link':
                self.link = value
            elif attr == 'left':
                if value:
                    self.frags[0] = value.split()
            elif attr == 'right':
                if value:
                    self.frags[1] = value.split()
            elif attr == 'index':
                self.is_index = True
            elif attr == 'showstate':
                self.showstate = True
            else:
                m = re.match('([a-zA-Z]+).([a-zA-Z~_]+)', attr)

                if m is None:
                    return

                attr, subattr = m.groups()
                if attr == 'arg':
                    self.args[subattr] = value

    ################
    #   Fragment   #
    ################
    class Fragment:
        """ A fragment """

        class Link:

            # TODO: change this fucking magic numbers to
            # real constants!
            CellLabel = -1111
            CellValue = -2222

            def __init__(self, line=None):

                self.page = str()
                self.args = {}

                if line:
                    self.parse(line)

            def copy(self):

                link = Session.Fragment.Link()
                link.page = self.page
                link.args = copy(self.args)
                return link

            def line(self):

                s = self.page

                for key, value in self.args.items():
                    if value == self.CellValue:
                        value = '$V$'
                    elif value == self.CellLabel:
                        value = '$L$'
                    s += ' %s=%s' % (key, value)

                return s

            def parse(self, line):

                args = line.split(' ')

                try:
                    self.page = args.pop(0)
                except:
                    return

                for tuple in args:
                    tuple = tuple.split('=')

                    if len(tuple) >= 2:
                        if tuple[1] == '$V$':
                            tuple[1] = self.CellValue
                        elif tuple[1] == '$L$':
                            tuple[1] = self.CellLabel

                        self.args[tuple[0]] = tuple[1]

        def __init__(self, ctx, function, args={}, links={}):
            """ Create a fragment.
                @param function [string] Function name (i.e. PacketTable)
                @param args [dict] arguments
            """
            self.function = function
            self.ctx = ctx
            self.args = copy(args)

            self.links = {}
            for label, link in links.items():
                self.links[label] = link.copy()

        def copy(self):

            return Session.Fragment(self.ctx, copy(self.function), self.args, self.links)

        def getLink(self, column):

            try:
                return self.links[column]
            except:
                return None

        def setArgs(self, args):
            """ Set args (I know this is an useless comment) """
            self.args = args

        def parse(self, attr, value):

            if attr == 'function':
                # function of this fragment
                self.function = value
            else:
                # If attr isn't matched, we check if this is
                # in form attr.subattr
                # for example, with args, it will arg.<argname>.
                m = re.match('([a-zA-Z]+).([a-zA-Z~_]+)', attr)

                if m is None:
                    return

                attr, subattr = m.groups()

                if attr == 'arg':
                    # This is an argument.

                    if subattr == 'title' and value:
                        value = trans(self.ctx, _(value))
                    self.args[subattr] = value

                if attr == 'link':

                    self.links[subattr] = self.Link(value)


    ################

    def __init__(self, ctx=None, dict=None):
        """ Create the Session object that store all informations about
            parameters of user.
            @param ctx [context] This is the context object, used to store Session object
                                 to Twisted Session.
            @param dict [dict] we can build Session object from a dictionnary.
        """

        if ctx:
            self.session = inevow.ISession(ctx)
        else:
            self.session = None

        self.context = ctx

        self.pages = {}
        self.frags = {}
        self.username = str()

        if dict:
            self.__load_user(dict)

    def copy(self):

        new = Session(self.ctx)
        for label, page in self.pages:
            new.pages[label] = page.copy()

        for label, frag in self.frags:
            new.frags[label] = frag.copy()

        new.username = self.username

        return new

    def isLogged(self):
        return bool(self.username)

    def getIndexPage(self):
        for name, page in self.pages.items():
            if page.is_index:
                return page

        return None

    def __load_user(self, sdict):
        """ Load a dictionnary to store it in objets. """

        for key, value in sdict.items():

            # A key is in form type_objname_attr
            m = re.match('([a-zA-Z]+).([a-zA-Z0-9_]+).([a-zA-Z._~]+)', key)

            if m is None:
                # Not in good form, we ignore it.
                continue

            typename, objname, attr = m.groups()

            types = {'page': (self.pages, Session.Page),
                     'frag': (self.frags, Session.Fragment)
                    }

            try:
                obj = types[typename][0][objname]
            except:
                obj = types[typename][1](self.context, objname)
                types[typename][0][objname] = obj

            obj.parse(attr, value)

    def __store_user(self):
        """ Create a dict from this object and store it in user parameters. """

        sdict = dict()

        for name, page in self.pages.items():
            if page.title:
                sdict['page.%s.title' % name] = page.title
            if page.link:
                sdict['page.%s.link' % name] = page.link
            if page.is_index:
                sdict['page.%s.index' % name] = 'true'
            if page.showstate:
                sdict['page.%s.index' % name] = 'true'
            sdict['page.%s.left' % name] = ' '.join(page.frags[0])
            sdict['page.%s.right' % name] = ' '.join(page.frags[1])

            for key, value in page.args.items():
                sdict['page.%s.%s' % (name, key)] = value

        for name, fragment in self.frags.items():
            sdict['frag.%s.function' % name] = fragment.function

            for key, link in fragment.links.items():
                sdict['frag.%s.link.%s' % (name, key)] = link.line()

            for key, value in fragment.args.items():
                sdict['frag.%s.arg.%s' % (name, key)] = value

        r = Service.callService_s("auth", "setParams", self.username, "nulog", sdict)
        if r:
            print r

    def store(self, ctx=None):
        """ Store object in Twisted Session.
            If user is logged, we store also in parameters of this user.
        """

        if not self.session and ctx:
            self.session = inevow.ISession(ctx)

        if self.session:
            self.session.setComponent(ISession, self)

        if self.isLogged():
            self.__store_user()

        return self

    def pageFragments(self, page):
        """ Get fragments of the page.
            If it doesn't exist, to don't get an exception, we return an empty list.
            @param page [string] page name
            @return [list] list of frag names
        """

        try:
            return self.pages[page].frags
        except:
            return []

    def fragArgs(self, name):
        """ Get args of this fragment.
            We return an empty dictionnary if this fragment doesn't exist.
            @param name [string] fragment name
            @return [dict] arg dict.
        """

        try:
            return self.frags[name].args.copy()
        except:
            return {}

    def setFragArgs(self, name, args):
        """ Update args of this fragment. """

        if not self.frags.has_key(name):
            return

        self.frags[name].args.update(args)
        self.store()

def authentification(username, password):

    try:
        logged = Service.callService_s("auth", "login", username, password)
        if isinstance(logged, bool):
            return logged
        else:
            print logged
            return False
    except Exception, e:
            print getBacktrace()
            return False

class Checker:

    implements(checkers.ICredentialsChecker)

    credentialInterfaces = (credentials.IUsernamePassword,)

    def _cbPasswordMatch(self, matched, username):
        if matched:
            return username
        else:
            return failure.Failure(error.UnauthorizedLogin())

    def requestAvatarId(self, credentials):

        matched = authentification(credentials.username, credentials.password)
        if matched:
            return defer.succeed(credentials.username)
        else:
            return defer.fail(error.UnauthorizedLogin())

def logout(*args, **kwargs):
    return None

class MyRealm(object):
    implements(portal.IRealm)

    def __init__(self, site, loginpage, config, langconfig):
        self.site = site
        self.loginpage = loginpage
        self.config = config
        self.langconfig = langconfig

    def requestAvatar(self, avatarId, mind, *interfaces):
        for iface in interfaces:
            if iface is inevow.IResource:
                if avatarId is checkers.ANONYMOUS:
                    if self.config.get('Sessions', 'anonymous') == 'yes':
                        resc = self.site(self.config, self.langconfig)
                    else:
                        resc = self.loginpage(self.config, self.langconfig)
                else:
                    resc = self.site(self.config, self.langconfig, avatarId)

                resc.realm = self
                return (inevow.IResource, resc, logout)

        raise NotImplementedError("Do not support any of the interfaces %s"%(interfaces,))

def createAuthResource(nulog, loginpage, config, langconfig):

    realm = MyRealm(nulog, loginpage, config, langconfig)
    myPortal = portal.Portal(realm)

    myPortal.registerChecker(checkers.AllowAnonymousAccess(),
                             credentials.IAnonymous)
    myPortal.registerChecker(Checker())

    res = guard.SessionWrapper(myPortal)
    res.persistentCookies = True

    return res
