/* hash.c
 
   Calculation of packet hashes

   The base algorithm of the "send_expect" Network Expect command is
   based on the algorithm of the "sr" command of Scapy, the packet
   manipulation program written in Python by Philippe Biondi. There are
   some differences, specially that Network Expect's implementation
   stores sent and received packets in raw (binary) form, but the basics
   are the same.

   Copyright (C) 2007, 2008, 2009, 2010 Eloy Paris

   This is part of Network Expect (nexp)

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
    
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
    
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdint.h>
#include <string.h>
#include <tcl.h>
#include <dnet.h>

#include "packets-priv.h"
#include "packets.h"
#include "tcl_ws-ftypes.h"

typedef void (*hash_func_t)(Tcl_Interp *, GByteArray *, int);

static void hash_eth(Tcl_Interp *, GByteArray *, int);
static void hash_ip(Tcl_Interp *, GByteArray *, int);
static void hash_icmp(Tcl_Interp *, GByteArray *, int);
static void hash_tcp(Tcl_Interp *, GByteArray *, int);

/*
 * Protocols (identified by name) we can calculate a hash for, and the
 * respective functions that calculates the hash.
 */
struct {
    char *proto_name;
    hash_func_t hash_func;
} hash_funcs[] = {
    {"eth", &hash_eth},
    {"ip", &hash_ip},
    {"icmp", &hash_icmp},
    {"tcp", &hash_tcp},

    {NULL, NULL}
};

/*
 * Calculates the hash of a packet
 *
 * packet: pointer to packet to dissect
 * fdata: frame_data structure (initialized by caller)
 */
GByteArray *
pkt_hash(Tcl_Interp *interp, const u_char *packet, frame_data *fdata)
{
    char *proto_name;
    GByteArray *hash;
    Tcl_Obj *listObj, *obj;
    int i, layer, llength;
    Tcl_Namespace *nsPtr;
#ifdef DEBUG
    const char *protocols;
#endif

    if ( (nsPtr = Tcl_FindNamespace(interp, "h", NULL, TCL_GLOBAL_ONLY) ) )
	Tcl_DeleteNamespace(nsPtr);

#ifdef DEBUG
    printf("===== Begin dissection for hash calculation =====\n");
#endif

    /* Dissect the packets; dissection results stored in Tcl vars */
    pkt_dissect_tcl(interp, packet, fdata, NULL /* no display filter */,
		    "h", 1 /* create non-ws vars */);

#ifdef DEBUG
    printf("===== End dissection for hash calculation =====\n");

    protocols = Tcl_GetVar(interp, "h::frame.protocols", 0);

    printf("frame.protocols = %s\n", protocols);
#endif

    hash = g_byte_array_new(); /* This is where we will store the packet hash */

    listObj = Tcl_GetVar2Ex(interp, "h::pdu(list)", NULL, 0);
    Tcl_ListObjLength(interp, listObj, &llength);

    /* Calculate hash for each layer of the packet */
    for (layer = 1 /* Skip layer 0 (frame) */;
	 layer < llength;
	 layer++) {
	Tcl_ListObjIndex(interp, listObj, layer, &obj);
	proto_name = Tcl_GetStringFromObj(obj, NULL);

	/* Lookup pointer to hash function and call hash function if found */
	for (i = 0; hash_funcs[i].proto_name; i++)
	    if (!strcmp(hash_funcs[i].proto_name, proto_name) ) {
		hash_funcs[i].hash_func(interp, hash, layer);
		break;
	    }
    }

    return hash;
}

static void
hash_eth(Tcl_Interp *interp, GByteArray *hash, int curr_layer _U_)
{
    uint16_t ether_type;

    _pkt_get_uint16(interp, "h::eth.type", &ether_type);

    g_byte_array_append(hash, (const guint8 *) &ether_type,
			sizeof(ether_type) );
}

static void
hash_ip(Tcl_Interp *interp, GByteArray *hash, int curr_layer)
{
    ip_addr_t src, dst, tmp;
    uint8_t ip_proto, icmp_type;
    char *next_layer;
    int icmp_in_ip; /* true if IP packet carries ICMP message */
    int ip_in_icmp; /* true if ICMP message carries IP packet */

    next_layer = _pkt_layername(interp, "h", curr_layer + 1);

    icmp_in_ip = next_layer && !strcmp(next_layer, "icmp");

    ip_in_icmp = curr_layer - 1 >= 0
		 && !strcmp("icmp",
			    _pkt_layername(interp, "h", curr_layer - 1) );

    if (icmp_in_ip && !ip_in_icmp) {
	/*
	 * We are looking at the outer header of an IP packet that is
	 * carrying an ICMP payload./
	 */

	_pkt_get_uint8(interp, "h::icmp.type", &icmp_type);

	if (   icmp_type == ICMP_UNREACH
	    || icmp_type == ICMP_SRCQUENCH
	    || icmp_type == ICMP_REDIRECT
	    || icmp_type == ICMP_TIMEXCEED
	    || icmp_type == ICMP_PARAMPROB)
	    /*
	     * The received packet is an ICMP error. In this case the hash
	     * of the packet will be that of the IP packet embedded in
	     * the ICMP message, so there's nothing to do here.
	     */
	    return;
    }

    if (ip_in_icmp) {
	/* Looking at IP header embedded in ICMP error message */
	_pkt_get_uint8(interp, "h::icmp.ip.proto", &ip_proto);
	_pkt_get_ipaddr(interp, "h::icmp.ip.src", &src);
	_pkt_get_ipaddr(interp, "h::icmp.ip.dst", &dst);
    } else {
	/* Looking at outer IP header */
	_pkt_get_uint8(interp, "h::ip.proto", &ip_proto);
	_pkt_get_ipaddr(interp, "h::ip.src", &src);
	_pkt_get_ipaddr(interp, "h::ip.dst", &dst);
    }

    tmp = src ^ dst;
    g_byte_array_append(hash, (const guint8 *) &tmp, sizeof(tmp) );

    g_byte_array_append(hash, (const guint8 *) &ip_proto, sizeof(ip_proto) );
}

static void
hash_icmp(Tcl_Interp *interp, GByteArray *hash, int curr_layer _U_)
{
    uint8_t icmp_type;
    uint16_t icmp_id, icmp_seq;
    int icmp_in_icmp;

    icmp_in_icmp = curr_layer - 2 >= 0
		   && !strcmp("icmp",
			      _pkt_layername(interp, "h",  curr_layer - 2) );

    if (!icmp_in_icmp) {
	_pkt_get_uint8(interp, "h::icmp.type", &icmp_type);

	if (   icmp_type == ICMP_UNREACH
	    || icmp_type == ICMP_SRCQUENCH
	    || icmp_type == ICMP_REDIRECT
	    || icmp_type == ICMP_TIMEXCEED
	    || icmp_type == ICMP_PARAMPROB)
	    /*
	     * The received packet is an ICMP error. In this case the hash
	     * of the packet will be that of the IP packet embedded in
	     * the ICMP message, so there's nothing to do here.
	     */
	    return;

	_pkt_get_uint16(interp, "h::icmp.ident", &icmp_id);
	_pkt_get_uint16(interp, "h::icmp.seq", &icmp_seq);
    } else {
	_pkt_get_uint16(interp, "h::icmp.icmp.ident", &icmp_id);
	_pkt_get_uint16(interp, "h::icmp.icmp.seq", &icmp_seq);
    }

    g_byte_array_append(hash, (const guint8 *) &icmp_id, sizeof(icmp_id) );
    g_byte_array_append(hash, (const guint8 *) &icmp_seq, sizeof(icmp_seq) );
}

static void
hash_tcp(Tcl_Interp *interp, GByteArray *hash, int curr_layer)
{
    uint16_t src_port, dst_port, tmp;
    int tcp_in_icmp;

    tcp_in_icmp = curr_layer - 2 >= 0
		  && !strcmp("icmp",
			     _pkt_layername(interp, "h",  curr_layer - 2) );

    if (tcp_in_icmp) {
	_pkt_get_uint16(interp, "h::icmp.tcp.srcport", &src_port);
	_pkt_get_uint16(interp, "h::icmp.tcp.dstport", &dst_port);
    } else {
	_pkt_get_uint16(interp, "h::tcp.srcport", &src_port);
	_pkt_get_uint16(interp, "h::tcp.dstport", &dst_port);
    }

    tmp = src_port ^ dst_port;

    g_byte_array_append(hash, (const guint8 *) &tmp, sizeof(tmp) );
}
