/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: t; c-basic-offset: 4 -*- */
/* libwps
 * Version: MPL 2.0 / LGPLv2.1+
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * Major Contributor(s):
 * Copyright (C) 2009, 2011 Alonso Laurent (alonso@loria.fr)
 * Copyright (C) 2006, 2007 Andrew Ziem
 * Copyright (C) 2004-2006 Fridrich Strba (fridrich.strba@bluewin.ch)
 * Copyright (C) 2004 Marc Maurer (uwog@uwog.net)
 * Copyright (C) 2003-2005 William Lachance (william.lachance@sympatico.ca)
 *
 * For minor contributions see the git repository.
 *
 * Alternatively, the contents of this file may be used under the terms
 * of the GNU Lesser General Public License Version 2.1 or later
 * (LGPLv2.1+), in which case the provisions of the LGPLv2.1+ are
 * applicable instead of those above.
 *
 * For further information visit http://libwps.sourceforge.net
 */

#include <string>
#include <sstream>

#include "libwps_tools_win.h"

namespace libwps_tools_win
{
////////////////////////////////////////////////////////////
//
// FONT
//
////////////////////////////////////////////////////////////

Font::Type Font::getFontType(std::string &fName)
{
	size_t len = fName.length();
	if (len && fName[len-1] != ')')
	{
		if (fName == "Baltica" || fName == "Pragmatica")
			return WIN3_CYRILLIC;
		if (len > 4 && (fName.find(" CYR", len-4) != std::string::npos ||
		                fName.find(" Cyr", len-4) != std::string::npos ||
		                fName.find(" cyr", len-4) != std::string::npos))
		{
			fName.resize(len-4);
			return WIN3_CYRILLIC;
		}
		if (len > 3 && (fName.find(" CE", len-3) != std::string::npos ||
		                fName.find(" Ce", len-3) != std::string::npos ||
		                fName.find(" ce", len-3) != std::string::npos))
		{
			fName.resize(len-3);
			return WIN3_CEUROPE;
		}
		if (len > 6 && (fName.find(" GREEK", len-6) != std::string::npos ||
		                fName.find(" Greek", len-6) != std::string::npos ||
		                fName.find(" greek", len-6) != std::string::npos))
		{
			fName.resize(len-6);
			return WIN3_GREEK;
		}
		if (len > 4 && (fName.find(" TUR", len-4) != std::string::npos ||
		                fName.find(" Tur", len-4) != std::string::npos ||
		                fName.find(" tur", len-4) != std::string::npos))
		{
			fName.resize(len-4);
			return WIN3_TURKISH;
		}
		if (len > 7 && (fName.find(" BALTIC", len-7) != std::string::npos ||
		                fName.find(" Baltic", len-7) != std::string::npos ||
		                fName.find(" baltic", len-7) != std::string::npos))
		{
			fName.resize(len-7);
			return WIN3_BALTIC;
		}
	}
	else if (len > 9)
	{
		if (fName.find(" (HEBREW)", len-9) != std::string::npos ||
		        fName.find(" (Hebrew)", len-9) != std::string::npos ||
		        fName.find(" (hebrew)", len-9) != std::string::npos)
		{
			fName.resize(len-9);
			return WIN3_HEBREW;
		}
		if (fName.find(" (ARABIC)", len-9) != std::string::npos ||
		        fName.find(" (Arabic)", len-9) != std::string::npos ||
		        fName.find(" (arabic)", len-9) != std::string::npos)
		{
			fName.resize(len-9);
			return WIN3_ARABIC;
		}
		if (len > 13 && (fName.find(" (VIETNAMESE)", len-13) != std::string::npos ||
		                 fName.find(" (Vietnamese)", len-13) != std::string::npos ||
		                 fName.find(" (vietnamese)", len-13) != std::string::npos))
		{
			fName.resize(len-13);
			return WIN3_VIETNAMESE;
		}
	}
	return UNKNOWN;
}

Font::Type Font::getTypeForOEM(int oem)
{
	switch (oem)
	{
	case 437:
		return CP_437;
	case 737:
		return CP_737;
	case 775:
		return CP_775;
	case 850:
		return DOS_850;
	case 852:
		return CP_852;
	case 855:
		return CP_855;
	case 856: // checkme
		return CP_856;
	case 857:
		return CP_857;
	case 858: // changme: dos850+Euro symbol
		return DOS_850;
	case 860:
		return CP_860;
	case 861:
		return CP_861;
	case 862:
		return CP_862;
	case 863:
		return CP_863;
	case 864:
		return CP_864;
	case 865:
		return CP_865;
	case 866:
		return CP_866;
	case 869:
		return CP_869;
	case 874:
		return CP_874;
	case 1006:
		return CP_1006;
	default:
		WPS_DEBUG_MSG(("libwps_tools_win::Font::getTypeForOEM: find unknown oem %d\n", oem));
	case 0:
		return UNKNOWN;
	}
}

std::string Font::getTypeName(Type type)
{
	switch (type)
	{
	case WIN3_CYRILLIC:
		return "Cyr";
	case WIN3_CEUROPE:
		return "Ce";
	case WIN3_GREEK:
		return "Greek";
	case WIN3_TURKISH:
		return "Tur";
	case WIN3_BALTIC:
		return "Baltic";
	case WIN3_ARABIC:
		return "Arabic";
	case WIN3_HEBREW:
		return "Hebrew";
	case WIN3_VIETNAMESE:
		return "Vietnamese";
	case WIN3_WEUROPE:
		return "We";
	case CP_424:
		return "CP424";
	case CP_437:
		return "CP437";
	case CP_737:
		return "CP737";
	case CP_775:
		return "CP775";
	case DOS_850:
		return "dos";
	case CP_852:
		return "CP852";
	case CP_855:
		return "CP855";
	case CP_856:
		return "CP856";
	case CP_857:
		return "CP857";
	case CP_860:
		return "CP860";
	case CP_861:
		return "CP861";
	case CP_862:
		return "CP862";
	case CP_863:
		return "CP863";
	case CP_864:
		return "CP864";
	case CP_865:
		return "CP865";
	case CP_866:
		return "CP866";
	case CP_869:
		return "CP869";
	case CP_874:
		return "CP874";
	case CP_1006:
		return "CP1006";
	case UNKNOWN:
	default:
		return "Unknown";
	}
}
#ifdef UNDEF
#  undef UNDEF
#endif
#define UNDEF 0xfffd

// Courtesy of unicode.org: http://unicode.org/Public/MAPPINGS/VENDORS/MICSFT/DOS|IBM/

static unsigned long unicodeFromCP424(unsigned char c)
{
	static const unsigned int cp424[256] =
	{
		0x0,0x1,0x2,0x3,0x9c,0x9,0x86,0x7f,0x97,0x8d,0x8e,0xb,0xc,0xd,0xe,0xf,
		0x10,0x11,0x12,0x13,0x9d,0x85,0x8,0x87,0x18,0x19,0x92,0x8f,0x1c,0x1d,0x1e,0x1f,
		0x80,0x81,0x82,0x83,0x84,0xa,0x17,0x1b,0x88,0x89,0x8a,0x8b,0x8c,0x5,0x6,0x7,
		0x90,0x91,0x16,0x93,0x94,0x95,0x96,0x4,0x98,0x99,0x9a,0x9b,0x14,0x15,0x9e,0x1a,
		0x20,0x5d0,0x5d1,0x5d2,0x5d3,0x5d4,0x5d5,0x5d6,0x5d7,0x5d8,0xa2,0x2e,0x3c,0x28,0x2b,0x7c,
		0x26,0x5d9,0x5da,0x5db,0x5dc,0x5dd,0x5de,0x5df,0x5e0,0x5e1,0x21,0x24,0x2a,0x29,0x3b,0xac,
		0x2d,0x2f,0x5e2,0x5e3,0x5e4,0x5e5,0x5e6,0x5e7,0x5e8,0x5e9,0xa6,0x2c,0x25,0x5f,0x3e,0x3f,
		UNDEF,0x5ea,UNDEF,UNDEF,0xa0,UNDEF,UNDEF,UNDEF,0x2017,0x60,0x3a,0x23,0x40,0x27,0x3d,0x22,
		UNDEF,0x61,0x62,0x63,0x64,0x65,0x66,0x67,0x68,0x69,0xab,0xbb,UNDEF,UNDEF,UNDEF,0xb1,
		0xb0,0x6a,0x6b,0x6c,0x6d,0x6e,0x6f,0x70,0x71,0x72,UNDEF,UNDEF,UNDEF,0xb8,UNDEF,0xa4,
		0xb5,0x7e,0x73,0x74,0x75,0x76,0x77,0x78,0x79,0x7a,UNDEF,UNDEF,UNDEF,UNDEF,UNDEF,0xae,
		0x5e,0xa3,0xa5,0xb7,0xa9,0xa7,0xb6,0xbc,0xbd,0xbe,0x5b,0x5d,0xaf,0xa8,0xb4,0xd7,
		0x7b,0x41,0x42,0x43,0x44,0x45,0x46,0x47,0x48,0x49,0xad,UNDEF,UNDEF,UNDEF,UNDEF,UNDEF,
		0x7d,0x4a,0x4b,0x4c,0x4d,0x4e,0x4f,0x50,0x51,0x52,0xb9,UNDEF,UNDEF,UNDEF,UNDEF,UNDEF,
		0x5c,0xf7,0x53,0x54,0x55,0x56,0x57,0x58,0x59,0x5a,0xb2,UNDEF,UNDEF,UNDEF,UNDEF,UNDEF,
		0x30,0x31,0x32,0x33,0x34,0x35,0x36,0x37,0x38,0x39,0xb3,UNDEF,UNDEF,UNDEF,UNDEF,0x9f,
	};

	return cp424[c];
}

static unsigned long unicodeFromCP437(unsigned char c)
{
	static const unsigned int cp437[128] =
	{
		0xc7,0xfc,0xe9,0xe2,0xe4,0xe0,0xe5,0xe7,0xea,0xeb,0xe8,0xef,0xee,0xec,0xc4,0xc5,
		0xc9,0xe6,0xc6,0xf4,0xf6,0xf2,0xfb,0xf9,0xff,0xd6,0xdc,0xa2,0xa3,0xa5,0x20a7,0x192,
		0xe1,0xed,0xf3,0xfa,0xf1,0xd1,0xaa,0xba,0xbf,0x2310,0xac,0xbd,0xbc,0xa1,0xab,0xbb,
		0x2591,0x2592,0x2593,0x2502,0x2524,0x2561,0x2562,0x2556,0x2555,0x2563,0x2551,0x2557,0x255d,0x255c,0x255b,0x2510,
		0x2514,0x2534,0x252c,0x251c,0x2500,0x253c,0x255e,0x255f,0x255a,0x2554,0x2569,0x2566,0x2560,0x2550,0x256c,0x2567,
		0x2568,0x2564,0x2565,0x2559,0x2558,0x2552,0x2553,0x256b,0x256a,0x2518,0x250c,0x2588,0x2584,0x258c,0x2590,0x2580,
		0x3b1,0xdf,0x393,0x3c0,0x3a3,0x3c3,0xb5,0x3c4,0x3a6,0x398,0x3a9,0x3b4,0x221e,0x3c6,0x3b5,0x2229,
		0x2261,0xb1,0x2265,0x2264,0x2320,0x2321,0xf7,0x2248,0xb0,0x2219,0xb7,0x221a,0x207f,0xb2,0x25a0,0xa0,
	};

	if (c < 0x80) return c;
	return cp437[c - 0x80];
}

static unsigned long unicodeFromCP737(unsigned char c)
{
	static const unsigned int cp737[128] =
	{
		0x391,0x392,0x393,0x394,0x395,0x396,0x397,0x398,0x399,0x39a,0x39b,0x39c,0x39d,0x39e,0x39f,0x3a0,
		0x3a1,0x3a3,0x3a4,0x3a5,0x3a6,0x3a7,0x3a8,0x3a9,0x3b1,0x3b2,0x3b3,0x3b4,0x3b5,0x3b6,0x3b7,0x3b8,
		0x3b9,0x3ba,0x3bb,0x3bc,0x3bd,0x3be,0x3bf,0x3c0,0x3c1,0x3c3,0x3c2,0x3c4,0x3c5,0x3c6,0x3c7,0x3c8,
		0x2591,0x2592,0x2593,0x2502,0x2524,0x2561,0x2562,0x2556,0x2555,0x2563,0x2551,0x2557,0x255d,0x255c,0x255b,0x2510,
		0x2514,0x2534,0x252c,0x251c,0x2500,0x253c,0x255e,0x255f,0x255a,0x2554,0x2569,0x2566,0x2560,0x2550,0x256c,0x2567,
		0x2568,0x2564,0x2565,0x2559,0x2558,0x2552,0x2553,0x256b,0x256a,0x2518,0x250c,0x2588,0x2584,0x258c,0x2590,0x2580,
		0x3c9,0x3ac,0x3ad,0x3ae,0x3ca,0x3af,0x3cc,0x3cd,0x3cb,0x3ce,0x386,0x388,0x389,0x38a,0x38c,0x38e,
		0x38f,0xb1,0x2265,0x2264,0x3aa,0x3ab,0xf7,0x2248,0xb0,0x2219,0xb7,0x221a,0x207f,0xb2,0x25a0,0xa0,
	};

	if (c < 0x80) return c;
	return cp737[c - 0x80];
}

static unsigned long unicodeFromCP775(unsigned char c)
{
	static const unsigned int cp775[128] =
	{
		0x106,0xfc,0xe9,0x101,0xe4,0x123,0xe5,0x107,0x142,0x113,0x156,0x157,0x12b,0x179,0xc4,0xc5,
		0xc9,0xe6,0xc6,0x14d,0xf6,0x122,0xa2,0x15a,0x15b,0xd6,0xdc,0xf8,0xa3,0xd8,0xd7,0xa4,
		0x100,0x12a,0xf3,0x17b,0x17c,0x17a,0x201d,0xa6,0xa9,0xae,0xac,0xbd,0xbc,0x141,0xab,0xbb,
		0x2591,0x2592,0x2593,0x2502,0x2524,0x104,0x10c,0x118,0x116,0x2563,0x2551,0x2557,0x255d,0x12e,0x160,0x2510,
		0x2514,0x2534,0x252c,0x251c,0x2500,0x253c,0x172,0x16a,0x255a,0x2554,0x2569,0x2566,0x2560,0x2550,0x256c,0x17d,
		0x105,0x10d,0x119,0x117,0x12f,0x161,0x173,0x16b,0x17e,0x2518,0x250c,0x2588,0x2584,0x258c,0x2590,0x2580,
		0xd3,0xdf,0x14c,0x143,0xf5,0xd5,0xb5,0x144,0x136,0x137,0x13b,0x13c,0x146,0x112,0x145,0x2019,
		0xad,0xb1,0x201c,0xbe,0xb6,0xa7,0xf7,0x201e,0xb0,0x2219,0xb7,0xb9,0xb3,0xb2,0x25a0,0xa0,
	};

	if (c < 0x80) return c;
	return cp775[c - 0x80];
}

/**
 * Take a character in CP850 encoding, convert it
 * Courtesy of glib2 and iconv
 **/
static unsigned long unicodeFromCP850(unsigned char c)
{
	// Fridrich: I see some MS Works files with IBM850 encoding ???
	static const unsigned int cp850[128] =
	{
		0x00c7, 0x00fc, 0x00e9, 0x00e2, 0x00e4, 0x00e0, 0x00e5, 0x00e7,
		0x00ea, 0x00eb, 0x00e8, 0x00ef, 0x00ee, 0x00ec, 0x00c4, 0x00c5,
		0x00c9, 0x00e6, 0x00c6, 0x00f4, 0x00f6, 0x00f2, 0x00fb, 0x00f9,
		0x00ff, 0x00d6, 0x00dc, 0x00f8, 0x00a3, 0x00d8, 0x00d7, 0x0192,
		0x00e1, 0x00ed, 0x00f3, 0x00fa, 0x00f1, 0x00d1, 0x00aa, 0x00ba,
		0x00bf, 0x00ae, 0x00ac, 0x00bd, 0x00bc, 0x00a1, 0x00ab, 0x00bb,
		0x2591, 0x2592, 0x2593, 0x2502, 0x2524, 0x00c1, 0x00c2, 0x00c0,
		0x00a9, 0x2563, 0x2551, 0x2557, 0x255d, 0x00a2, 0x00a5, 0x2510,
		0x2514, 0x2534, 0x252c, 0x251c, 0x2500, 0x253c, 0x00e3, 0x00c3,
		0x255a, 0x2554, 0x2569, 0x2566, 0x2560, 0x2550, 0x256c, 0x00a4,
		0x00f0, 0x00d0, 0x00ca, 0x00cb, 0x00c8, 0x0131, 0x00cd, 0x00ce,
		0x00cf, 0x2518, 0x250c, 0x2588, 0x2584, 0x00a6, 0x00cc, 0x2580,
		0x00d3, 0x00df, 0x00d4, 0x00d2, 0x00f5, 0x00d5, 0x00b5, 0x00fe,
		0x00de, 0x00da, 0x00db, 0x00d9, 0x00fd, 0x00dd, 0x00af, 0x00b4,
		0x00ad, 0x00b1, 0x2017, 0x00be, 0x00b6, 0x00a7, 0x00f7, 0x00b8,
		0x00b0, 0x00a8, 0x00b7, 0x00b9, 0x00b3, 0x00b2, 0x25a0, 0x00a0
	};

	if (c < 0x80) return c;
	return cp850[c - 0x80];
}

// Courtesy of unicode.org: http://unicode.org/Public/MAPPINGS/VENDORS/MICSFT/DOS|IBM/
static unsigned long unicodeFromCP852(unsigned char c)
{
	static const unsigned int cp852[128] =
	{
		0xc7,0xfc,0xe9,0xe2,0xe4,0x16f,0x107,0xe7,0x142,0xeb,0x150,0x151,0xee,0x179,0xc4,0x106,
		0xc9,0x139,0x13a,0xf4,0xf6,0x13d,0x13e,0x15a,0x15b,0xd6,0xdc,0x164,0x165,0x141,0xd7,0x10d,
		0xe1,0xed,0xf3,0xfa,0x104,0x105,0x17d,0x17e,0x118,0x119,0xac,0x17a,0x10c,0x15f,0xab,0xbb,
		0x2591,0x2592,0x2593,0x2502,0x2524,0xc1,0xc2,0x11a,0x15e,0x2563,0x2551,0x2557,0x255d,0x17b,0x17c,0x2510,
		0x2514,0x2534,0x252c,0x251c,0x2500,0x253c,0x102,0x103,0x255a,0x2554,0x2569,0x2566,0x2560,0x2550,0x256c,0xa4,
		0x111,0x110,0x10e,0xcb,0x10f,0x147,0xcd,0xce,0x11b,0x2518,0x250c,0x2588,0x2584,0x162,0x16e,0x2580,
		0xd3,0xdf,0xd4,0x143,0x144,0x148,0x160,0x161,0x154,0xda,0x155,0x170,0xfd,0xdd,0x163,0xb4,
		0xad,0x2dd,0x2db,0x2c7,0x2d8,0xa7,0xf7,0xb8,0xb0,0xa8,0x2d9,0x171,0x158,0x159,0x25a0,0xa0,
	};

	if (c < 0x80) return c;
	return cp852[c - 0x80];
}

static unsigned long unicodeFromCP855(unsigned char c)
{
	static const unsigned int cp855[128] =
	{
		0x452,0x402,0x453,0x403,0x451,0x401,0x454,0x404,0x455,0x405,0x456,0x406,0x457,0x407,0x458,0x408,
		0x459,0x409,0x45a,0x40a,0x45b,0x40b,0x45c,0x40c,0x45e,0x40e,0x45f,0x40f,0x44e,0x42e,0x44a,0x42a,
		0x430,0x410,0x431,0x411,0x446,0x426,0x434,0x414,0x435,0x415,0x444,0x424,0x433,0x413,0xab,0xbb,
		0x2591,0x2592,0x2593,0x2502,0x2524,0x445,0x425,0x438,0x418,0x2563,0x2551,0x2557,0x255d,0x439,0x419,0x2510,
		0x2514,0x2534,0x252c,0x251c,0x2500,0x253c,0x43a,0x41a,0x255a,0x2554,0x2569,0x2566,0x2560,0x2550,0x256c,0xa4,
		0x43b,0x41b,0x43c,0x41c,0x43d,0x41d,0x43e,0x41e,0x43f,0x2518,0x250c,0x2588,0x2584,0x41f,0x44f,0x2580,
		0x42f,0x440,0x420,0x441,0x421,0x442,0x422,0x443,0x423,0x436,0x416,0x432,0x412,0x44c,0x42c,0x2116,
		0xad,0x44b,0x42b,0x437,0x417,0x448,0x428,0x44d,0x42d,0x449,0x429,0x447,0x427,0xa7,0x25a0,0xa0,
	};

	if (c < 0x80) return c;
	return cp855[c - 0x80];
}

static unsigned long unicodeFromCP856(unsigned char c)
{
	static const unsigned int cp856[128] =
	{
		0x5d0,0x5d1,0x5d2,0x5d3,0x5d4,0x5d5,0x5d6,0x5d7,0x5d8,0x5d9,0x5da,0x5db,0x5dc,0x5dd,0x5de,0x5df,
		0x5e0,0x5e1,0x5e2,0x5e3,0x5e4,0x5e5,0x5e6,0x5e7,0x5e8,0x5e9,0x5ea,UNDEF,0xa3,UNDEF,0xd7,UNDEF,
		UNDEF,UNDEF,UNDEF,UNDEF,UNDEF,UNDEF,UNDEF,UNDEF,UNDEF,0xae,0xac,0xbd,0xbc,UNDEF,0xab,0xbb,
		0x2591,0x2592,0x2593,0x2502,0x2524,UNDEF,UNDEF,UNDEF,0xa9,0x2563,0x2551,0x2557,0x255d,0xa2,0xa5,0x2510,
		0x2514,0x2534,0x252c,0x251c,0x2500,0x253c,UNDEF,UNDEF,0x255a,0x2554,0x2569,0x2566,0x2560,0x2550,0x256c,0xa4,
		UNDEF,UNDEF,UNDEF,UNDEF,UNDEF,UNDEF,UNDEF,UNDEF,UNDEF,0x2518,0x250c,0x2588,0x2584,0xa6,UNDEF,0x2580,
		UNDEF,UNDEF,UNDEF,UNDEF,UNDEF,UNDEF,0xb5,UNDEF,UNDEF,UNDEF,UNDEF,UNDEF,UNDEF,UNDEF,0xaf,0xb4,
		0xad,0xb1,0x2017,0xbe,0xb6,0xa7,0xf7,0xb8,0xb0,0xa8,0xb7,0xb9,0xb3,0xb2,0x25a0,0xa0,
	};

	if (c < 0x80) return c;
	return cp856[c - 0x80];
}

static unsigned long unicodeFromCP857(unsigned char c)
{
	static const unsigned int cp857[128] =
	{
		0xc7,0xfc,0xe9,0xe2,0xe4,0xe0,0xe5,0xe7,0xea,0xeb,0xe8,0xef,0xee,0x131,0xc4,0xc5,
		0xc9,0xe6,0xc6,0xf4,0xf6,0xf2,0xfb,0xf9,0x130,0xd6,0xdc,0xf8,0xa3,0xd8,0x15e,0x15f,
		0xe1,0xed,0xf3,0xfa,0xf1,0xd1,0x11e,0x11f,0xbf,0xae,0xac,0xbd,0xbc,0xa1,0xab,0xbb,
		0x2591,0x2592,0x2593,0x2502,0x2524,0xc1,0xc2,0xc0,0xa9,0x2563,0x2551,0x2557,0x255d,0xa2,0xa5,0x2510,
		0x2514,0x2534,0x252c,0x251c,0x2500,0x253c,0xe3,0xc3,0x255a,0x2554,0x2569,0x2566,0x2560,0x2550,0x256c,0xa4,
		0xba,0xaa,0xca,0xcb,0xc8,UNDEF,0xcd,0xce,0xcf,0x2518,0x250c,0x2588,0x2584,0xa6,0xcc,0x2580,
		0xd3,0xdf,0xd4,0xd2,0xf5,0xd5,0xb5,UNDEF,0xd7,0xda,0xdb,0xd9,0xec,0xff,0xaf,0xb4,
		0xad,0xb1,UNDEF,0xbe,0xb6,0xa7,0xf7,0xb8,0xb0,0xa8,0xb7,0xb9,0xb3,0xb2,0x25a0,0xa0,
	};

	if (c < 0x80) return c;
	return cp857[c - 0x80];
}

static unsigned long unicodeFromCP860(unsigned char c)
{
	static const unsigned int cp860[128] =
	{
		0xc7,0xfc,0xe9,0xe2,0xe3,0xe0,0xc1,0xe7,0xea,0xca,0xe8,0xcd,0xd4,0xec,0xc3,0xc2,
		0xc9,0xc0,0xc8,0xf4,0xf5,0xf2,0xda,0xf9,0xcc,0xd5,0xdc,0xa2,0xa3,0xd9,0x20a7,0xd3,
		0xe1,0xed,0xf3,0xfa,0xf1,0xd1,0xaa,0xba,0xbf,0xd2,0xac,0xbd,0xbc,0xa1,0xab,0xbb,
		0x2591,0x2592,0x2593,0x2502,0x2524,0x2561,0x2562,0x2556,0x2555,0x2563,0x2551,0x2557,0x255d,0x255c,0x255b,0x2510,
		0x2514,0x2534,0x252c,0x251c,0x2500,0x253c,0x255e,0x255f,0x255a,0x2554,0x2569,0x2566,0x2560,0x2550,0x256c,0x2567,
		0x2568,0x2564,0x2565,0x2559,0x2558,0x2552,0x2553,0x256b,0x256a,0x2518,0x250c,0x2588,0x2584,0x258c,0x2590,0x2580,
		0x3b1,0xdf,0x393,0x3c0,0x3a3,0x3c3,0xb5,0x3c4,0x3a6,0x398,0x3a9,0x3b4,0x221e,0x3c6,0x3b5,0x2229,
		0x2261,0xb1,0x2265,0x2264,0x2320,0x2321,0xf7,0x2248,0xb0,0x2219,0xb7,0x221a,0x207f,0xb2,0x25a0,0xa0,
	};

	if (c < 0x80) return c;
	return cp860[c - 0x80];
}

static unsigned long unicodeFromCP861(unsigned char c)
{
	static const unsigned int cp861[128] =
	{
		0xc7,0xfc,0xe9,0xe2,0xe4,0xe0,0xe5,0xe7,0xea,0xeb,0xe8,0xd0,0xf0,0xde,0xc4,0xc5,
		0xc9,0xe6,0xc6,0xf4,0xf6,0xfe,0xfb,0xdd,0xfd,0xd6,0xdc,0xf8,0xa3,0xd8,0x20a7,0x192,
		0xe1,0xed,0xf3,0xfa,0xc1,0xcd,0xd3,0xda,0xbf,0x2310,0xac,0xbd,0xbc,0xa1,0xab,0xbb,
		0x2591,0x2592,0x2593,0x2502,0x2524,0x2561,0x2562,0x2556,0x2555,0x2563,0x2551,0x2557,0x255d,0x255c,0x255b,0x2510,
		0x2514,0x2534,0x252c,0x251c,0x2500,0x253c,0x255e,0x255f,0x255a,0x2554,0x2569,0x2566,0x2560,0x2550,0x256c,0x2567,
		0x2568,0x2564,0x2565,0x2559,0x2558,0x2552,0x2553,0x256b,0x256a,0x2518,0x250c,0x2588,0x2584,0x258c,0x2590,0x2580,
		0x3b1,0xdf,0x393,0x3c0,0x3a3,0x3c3,0xb5,0x3c4,0x3a6,0x398,0x3a9,0x3b4,0x221e,0x3c6,0x3b5,0x2229,
		0x2261,0xb1,0x2265,0x2264,0x2320,0x2321,0xf7,0x2248,0xb0,0x2219,0xb7,0x221a,0x207f,0xb2,0x25a0,0xa0,
	};

	if (c < 0x80) return c;
	return cp861[c - 0x80];
}

static unsigned long unicodeFromCP862(unsigned char c)
{
	static const unsigned int cp862[128] =
	{
		0x5d0,0x5d1,0x5d2,0x5d3,0x5d4,0x5d5,0x5d6,0x5d7,0x5d8,0x5d9,0x5da,0x5db,0x5dc,0x5dd,0x5de,0x5df,
		0x5e0,0x5e1,0x5e2,0x5e3,0x5e4,0x5e5,0x5e6,0x5e7,0x5e8,0x5e9,0x5ea,0xa2,0xa3,0xa5,0x20a7,0x192,
		0xe1,0xed,0xf3,0xfa,0xf1,0xd1,0xaa,0xba,0xbf,0x2310,0xac,0xbd,0xbc,0xa1,0xab,0xbb,
		0x2591,0x2592,0x2593,0x2502,0x2524,0x2561,0x2562,0x2556,0x2555,0x2563,0x2551,0x2557,0x255d,0x255c,0x255b,0x2510,
		0x2514,0x2534,0x252c,0x251c,0x2500,0x253c,0x255e,0x255f,0x255a,0x2554,0x2569,0x2566,0x2560,0x2550,0x256c,0x2567,
		0x2568,0x2564,0x2565,0x2559,0x2558,0x2552,0x2553,0x256b,0x256a,0x2518,0x250c,0x2588,0x2584,0x258c,0x2590,0x2580,
		0x3b1,0xdf,0x393,0x3c0,0x3a3,0x3c3,0xb5,0x3c4,0x3a6,0x398,0x3a9,0x3b4,0x221e,0x3c6,0x3b5,0x2229,
		0x2261,0xb1,0x2265,0x2264,0x2320,0x2321,0xf7,0x2248,0xb0,0x2219,0xb7,0x221a,0x207f,0xb2,0x25a0,0xa0,
	};

	if (c < 0x80) return c;
	return cp862[c - 0x80];
}

static unsigned long unicodeFromCP863(unsigned char c)
{
	static const unsigned int cp863[128] =
	{
		0xc7,0xfc,0xe9,0xe2,0xc2,0xe0,0xb6,0xe7,0xea,0xeb,0xe8,0xef,0xee,0x2017,0xc0,0xa7,
		0xc9,0xc8,0xca,0xf4,0xcb,0xcf,0xfb,0xf9,0xa4,0xd4,0xdc,0xa2,0xa3,0xd9,0xdb,0x192,
		0xa6,0xb4,0xf3,0xfa,0xa8,0xb8,0xb3,0xaf,0xce,0x2310,0xac,0xbd,0xbc,0xbe,0xab,0xbb,
		0x2591,0x2592,0x2593,0x2502,0x2524,0x2561,0x2562,0x2556,0x2555,0x2563,0x2551,0x2557,0x255d,0x255c,0x255b,0x2510,
		0x2514,0x2534,0x252c,0x251c,0x2500,0x253c,0x255e,0x255f,0x255a,0x2554,0x2569,0x2566,0x2560,0x2550,0x256c,0x2567,
		0x2568,0x2564,0x2565,0x2559,0x2558,0x2552,0x2553,0x256b,0x256a,0x2518,0x250c,0x2588,0x2584,0x258c,0x2590,0x2580,
		0x3b1,0xdf,0x393,0x3c0,0x3a3,0x3c3,0xb5,0x3c4,0x3a6,0x398,0x3a9,0x3b4,0x221e,0x3c6,0x3b5,0x2229,
		0x2261,0xb1,0x2265,0x2264,0x2320,0x2321,0xf7,0x2248,0xb0,0x2219,0xb7,0x221a,0x207f,0xb2,0x25a0,0xa0,
	};

	if (c < 0x80) return c;
	return cp863[c - 0x80];
}

static unsigned long unicodeFromCP864(unsigned char c)
{
	static const unsigned int cp864[128] =
	{
		0xb0,0xb7,0x2219,0x221a,0x2592,0x2500,0x2502,0x253c,0x2524,0x252c,0x251c,0x2534,0x2510,0x250c,0x2514,0x2518,
		0x3b2,0x221e,0x3c6,0xb1,0xbd,0xbc,0x2248,0xab,0xbb,0xfef7,0xfef8,UNDEF,UNDEF,0xfefb,0xfefc,UNDEF,
		0xa0,0xad,0xfe82,0xa3,0xa4,0xfe84,UNDEF,UNDEF,0xfe8e,0xfe8f,0xfe95,0xfe99,0x60c,0xfe9d,0xfea1,0xfea5,
		0x660,0x661,0x662,0x663,0x664,0x665,0x666,0x667,0x668,0x669,0xfed1,0x61b,0xfeb1,0xfeb5,0xfeb9,0x61f,
		0xa2,0xfe80,0xfe81,0xfe83,0xfe85,0xfeca,0xfe8b,0xfe8d,0xfe91,0xfe93,0xfe97,0xfe9b,0xfe9f,0xfea3,0xfea7,0xfea9,
		0xfeab,0xfead,0xfeaf,0xfeb3,0xfeb7,0xfebb,0xfebf,0xfec1,0xfec5,0xfecb,0xfecf,0xa6,0xac,0xf7,0xd7,0xfec9,
		0x640,0xfed3,0xfed7,0xfedb,0xfedf,0xfee3,0xfee7,0xfeeb,0xfeed,0xfeef,0xfef3,0xfebd,0xfecc,0xfece,0xfecd,0xfee1,
		0xfe7d,0x651,0xfee5,0xfee9,0xfeec,0xfef0,0xfef2,0xfed0,0xfed5,0xfef5,0xfef6,0xfedd,0xfed9,0xfef1,0x25a0,UNDEF,
	};

	if (c < 0x80) return c;
	return cp864[c - 0x80];
}

static unsigned long unicodeFromCP865(unsigned char c)
{
	static const unsigned int cp865[128] =
	{
		0xc7,0xfc,0xe9,0xe2,0xe4,0xe0,0xe5,0xe7,0xea,0xeb,0xe8,0xef,0xee,0xec,0xc4,0xc5,
		0xc9,0xe6,0xc6,0xf4,0xf6,0xf2,0xfb,0xf9,0xff,0xd6,0xdc,0xf8,0xa3,0xd8,0x20a7,0x192,
		0xe1,0xed,0xf3,0xfa,0xf1,0xd1,0xaa,0xba,0xbf,0x2310,0xac,0xbd,0xbc,0xa1,0xab,0xa4,
		0x2591,0x2592,0x2593,0x2502,0x2524,0x2561,0x2562,0x2556,0x2555,0x2563,0x2551,0x2557,0x255d,0x255c,0x255b,0x2510,
		0x2514,0x2534,0x252c,0x251c,0x2500,0x253c,0x255e,0x255f,0x255a,0x2554,0x2569,0x2566,0x2560,0x2550,0x256c,0x2567,
		0x2568,0x2564,0x2565,0x2559,0x2558,0x2552,0x2553,0x256b,0x256a,0x2518,0x250c,0x2588,0x2584,0x258c,0x2590,0x2580,
		0x3b1,0xdf,0x393,0x3c0,0x3a3,0x3c3,0xb5,0x3c4,0x3a6,0x398,0x3a9,0x3b4,0x221e,0x3c6,0x3b5,0x2229,
		0x2261,0xb1,0x2265,0x2264,0x2320,0x2321,0xf7,0x2248,0xb0,0x2219,0xb7,0x221a,0x207f,0xb2,0x25a0,0xa0,
	};

	if (c < 0x80) return c;
	return cp865[c - 0x80];
}

static unsigned long unicodeFromCP866(unsigned char c)
{
	static const unsigned int cp866[128] =
	{
		0x410,0x411,0x412,0x413,0x414,0x415,0x416,0x417,0x418,0x419,0x41a,0x41b,0x41c,0x41d,0x41e,0x41f,
		0x420,0x421,0x422,0x423,0x424,0x425,0x426,0x427,0x428,0x429,0x42a,0x42b,0x42c,0x42d,0x42e,0x42f,
		0x430,0x431,0x432,0x433,0x434,0x435,0x436,0x437,0x438,0x439,0x43a,0x43b,0x43c,0x43d,0x43e,0x43f,
		0x2591,0x2592,0x2593,0x2502,0x2524,0x2561,0x2562,0x2556,0x2555,0x2563,0x2551,0x2557,0x255d,0x255c,0x255b,0x2510,
		0x2514,0x2534,0x252c,0x251c,0x2500,0x253c,0x255e,0x255f,0x255a,0x2554,0x2569,0x2566,0x2560,0x2550,0x256c,0x2567,
		0x2568,0x2564,0x2565,0x2559,0x2558,0x2552,0x2553,0x256b,0x256a,0x2518,0x250c,0x2588,0x2584,0x258c,0x2590,0x2580,
		0x440,0x441,0x442,0x443,0x444,0x445,0x446,0x447,0x448,0x449,0x44a,0x44b,0x44c,0x44d,0x44e,0x44f,
		0x401,0x451,0x404,0x454,0x407,0x457,0x40e,0x45e,0xb0,0x2219,0xb7,0x221a,0x2116,0xa4,0x25a0,0xa0,
	};

	if (c < 0x80) return c;
	return cp866[c - 0x80];
}

static unsigned long unicodeFromCP869(unsigned char c)
{
	static const unsigned int cp869[128] =
	{
		UNDEF,UNDEF,UNDEF,UNDEF,UNDEF,UNDEF,0x386,UNDEF,0xb7,0xac,0xa6,0x2018,0x2019,0x388,0x2015,0x389,
		0x38a,0x3aa,0x38c,UNDEF,UNDEF,0x38e,0x3ab,0xa9,0x38f,0xb2,0xb3,0x3ac,0xa3,0x3ad,0x3ae,0x3af,
		0x3ca,0x390,0x3cc,0x3cd,0x391,0x392,0x393,0x394,0x395,0x396,0x397,0xbd,0x398,0x399,0xab,0xbb,
		0x2591,0x2592,0x2593,0x2502,0x2524,0x39a,0x39b,0x39c,0x39d,0x2563,0x2551,0x2557,0x255d,0x39e,0x39f,0x2510,
		0x2514,0x2534,0x252c,0x251c,0x2500,0x253c,0x3a0,0x3a1,0x255a,0x2554,0x2569,0x2566,0x2560,0x2550,0x256c,0x3a3,
		0x3a4,0x3a5,0x3a6,0x3a7,0x3a8,0x3a9,0x3b1,0x3b2,0x3b3,0x2518,0x250c,0x2588,0x2584,0x3b4,0x3b5,0x2580,
		0x3b6,0x3b7,0x3b8,0x3b9,0x3ba,0x3bb,0x3bc,0x3bd,0x3be,0x3bf,0x3c0,0x3c1,0x3c3,0x3c2,0x3c4,0x384,
		0xad,0xb1,0x3c5,0x3c6,0x3c7,0xa7,0x3c8,0x385,0xb0,0xa8,0x3c9,0x3cb,0x3b0,0x3ce,0x25a0,0xa0,
	};

	if (c < 0x80) return c;
	return cp869[c - 0x80];
}

static unsigned long unicodeFromCP874(unsigned char c)
{
	static const unsigned int cp874[128] =
	{
		0x20ac,UNDEF,UNDEF,UNDEF,UNDEF,0x2026,UNDEF,UNDEF,UNDEF,UNDEF,UNDEF,UNDEF,UNDEF,UNDEF,UNDEF,UNDEF,
		UNDEF,0x2018,0x2019,0x201c,0x201d,0x2022,0x2013,0x2014,UNDEF,UNDEF,UNDEF,UNDEF,UNDEF,UNDEF,UNDEF,UNDEF,
		0xa0,0xe01,0xe02,0xe03,0xe04,0xe05,0xe06,0xe07,0xe08,0xe09,0xe0a,0xe0b,0xe0c,0xe0d,0xe0e,0xe0f,
		0xe10,0xe11,0xe12,0xe13,0xe14,0xe15,0xe16,0xe17,0xe18,0xe19,0xe1a,0xe1b,0xe1c,0xe1d,0xe1e,0xe1f,
		0xe20,0xe21,0xe22,0xe23,0xe24,0xe25,0xe26,0xe27,0xe28,0xe29,0xe2a,0xe2b,0xe2c,0xe2d,0xe2e,0xe2f,
		0xe30,0xe31,0xe32,0xe33,0xe34,0xe35,0xe36,0xe37,0xe38,0xe39,0xe3a,UNDEF,UNDEF,UNDEF,UNDEF,0xe3f,
		0xe40,0xe41,0xe42,0xe43,0xe44,0xe45,0xe46,0xe47,0xe48,0xe49,0xe4a,0xe4b,0xe4c,0xe4d,0xe4e,0xe4f,
		0xe50,0xe51,0xe52,0xe53,0xe54,0xe55,0xe56,0xe57,0xe58,0xe59,0xe5a,0xe5b,UNDEF,UNDEF,UNDEF,UNDEF,
	};

	if (c < 0x80) return c;
	return cp874[c - 0x80];
}

static unsigned long unicodeFromCP1006(unsigned char c)
{
	static const unsigned int cp1006[128] =
	{
		0x80,0x81,0x82,0x83,0x84,0x85,0x86,0x87,0x88,0x89,0x8a,0x8b,0x8c,0x8d,0x8e,0x8f,
		0x90,0x91,0x92,0x93,0x94,0x95,0x96,0x97,0x98,0x99,0x9a,0x9b,0x9c,0x9d,0x9e,0x9f,
		0xa0,0x6f0,0x6f1,0x6f2,0x6f3,0x6f4,0x6f5,0x6f6,0x6f7,0x6f8,0x6f9,0x60c,0x61b,0xad,0x61f,0xfe81,
		0xfe8d,0xfe8e,0xfe8e,0xfe8f,0xfe91,0xfb56,0xfb58,0xfe93,0xfe95,0xfe97,0xfb66,0xfb68,0xfe99,0xfe9b,0xfe9d,0xfe9f,
		0xfb7a,0xfb7c,0xfea1,0xfea3,0xfea5,0xfea7,0xfea9,0xfb84,0xfeab,0xfead,0xfb8c,0xfeaf,0xfb8a,0xfeb1,0xfeb3,0xfeb5,
		0xfeb7,0xfeb9,0xfebb,0xfebd,0xfebf,0xfec1,0xfec5,0xfec9,0xfeca,0xfecb,0xfecc,0xfecd,0xfece,0xfecf,0xfed0,0xfed1,
		0xfed3,0xfed5,0xfed7,0xfed9,0xfedb,0xfb92,0xfb94,0xfedd,0xfedf,0xfee0,0xfee1,0xfee3,0xfb9e,0xfee5,0xfee7,0xfe85,
		0xfeed,0xfba6,0xfba8,0xfba9,0xfbaa,0xfe80,0xfe89,0xfe8a,0xfe8b,0xfef1,0xfef2,0xfef3,0xfbb0,0xfbae,0xfe7c,0xfe7d,
	};

	if (c < 0x80) return c;
	return cp1006[c - 0x80];
}

/**
 * Take a character in CP1250 encoding, convert it
 * Courtesy of unicode.org: http://unicode.org/Public/MAPPINGS/VENDORS/MICSFT/WINDOWS/
 **/
static unsigned long unicodeFromCP1250(unsigned char c)
{
	static unsigned int const cp1250[] =
	{
		0x20AC, UNDEF /* 0x81 */, 0x201A, UNDEF /* 0x83 */, 0x201E, 0x2026, 0x2020, 0x2021,
		UNDEF /* 0x88 */, 0x2030, 0x0160, 0x2039, 0x015A, 0x0164, 0x017D, 0x0179,
		UNDEF /* 0x90 */, 0x2018, 0x2019, 0x201C, 0x201D, 0x2022, 0x2013, 0x2014,
		UNDEF /* 0x98 */, 0x2122, 0x0161, 0x203A, 0x015B, 0x0165, 0x017E, 0x017A,
		0x00A0, 0x02C7, 0x02D8, 0x0141, 0x00A4, 0x0104, 0x00A6, 0x00A7,
		0x00A8, 0x00A9, 0x015E, 0x00AB, 0x00AC, 0x00AD, 0x00AE, 0x017B,
		0x00B0, 0x00B1, 0x02DB, 0x0142, 0x00B4, 0x00B5, 0x00B6, 0x00B7,
		0x00B8, 0x0105, 0x015F, 0x00BB, 0x013D, 0x02DD, 0x013E, 0x017C,
		0x0154, 0x00C1, 0x00C2, 0x0102, 0x00C4, 0x0139, 0x0106, 0x00C7,
		0x010C, 0x00C9, 0x0118, 0x00CB, 0x011A, 0x00CD, 0x00CE, 0x010E,
		0x0110, 0x0143, 0x0147, 0x00D3, 0x00D4, 0x0150, 0x00D6, 0x00D7,
		0x0158, 0x016E, 0x00DA, 0x0170, 0x00DC, 0x00DD, 0x0162, 0x00DF,
		0x0155, 0x00E1, 0x00E2, 0x0103, 0x00E4, 0x013A, 0x0107, 0x00E7,
		0x010D, 0x00E9, 0x0119, 0x00EB, 0x011B, 0x00ED, 0x00EE, 0x010F,
		0x0111, 0x0144, 0x0148, 0x00F3, 0x00F4, 0x0151, 0x00F6, 0x00F7,
		0x0159, 0x016F, 0x00FA, 0x0171, 0x00FC, 0x00FD, 0x0163, 0x02D9
	};
	if (c < 0x80) return c;
	return cp1250[c - 0x80];
}

/**
 * Take a character in CP1251 encoding, convert it
 * Courtesy of unicode.org: http://unicode.org/Public/MAPPINGS/VENDORS/MICSFT/WINDOWS/
 **/
static unsigned long unicodeFromCP1251(unsigned char c)
{
	static unsigned int const cp1251[] =
	{
		0x0402, 0x0403, 0x201A, 0x0453, 0x201E, 0x2026, 0x2020, 0x2021,
		0x20AC, 0x2030, 0x0409, 0x2039, 0x040A, 0x040C, 0x040B, 0x040F,
		0x0452, 0x2018, 0x2019, 0x201C, 0x201D, 0x2022, 0x2013, 0x2014,
		UNDEF /* 0x98 */, 0x2122, 0x0459, 0x203A, 0x045A, 0x045C, 0x045B, 0x045F,
		0x00A0, 0x040E, 0x045E, 0x0408, 0x00A4, 0x0490, 0x00A6, 0x00A7,
		0x0401, 0x00A9, 0x0404, 0x00AB, 0x00AC, 0x00AD, 0x00AE, 0x0407,
		0x00B0, 0x00B1, 0x0406, 0x0456, 0x0491, 0x00B5, 0x00B6, 0x00B7,
		0x0451, 0x2116, 0x0454, 0x00BB, 0x0458, 0x0405, 0x0455, 0x0457,
		0x0410, 0x0411, 0x0412, 0x0413, 0x0414, 0x0415, 0x0416, 0x0417,
		0x0418, 0x0419, 0x041A, 0x041B, 0x041C, 0x041D, 0x041E, 0x041F,
		0x0420, 0x0421, 0x0422, 0x0423, 0x0424, 0x0425, 0x0426, 0x0427,
		0x0428, 0x0429, 0x042A, 0x042B, 0x042C, 0x042D, 0x042E, 0x042F,
		0x0430, 0x0431, 0x0432, 0x0433, 0x0434, 0x0435, 0x0436, 0x0437,
		0x0438, 0x0439, 0x043A, 0x043B, 0x043C, 0x043D, 0x043E, 0x043F,
		0x0440, 0x0441, 0x0442, 0x0443, 0x0444, 0x0445, 0x0446, 0x0447,
		0x0448, 0x0449, 0x044A, 0x044B, 0x044C, 0x044D, 0x044E, 0x044F,
	};
	if (c < 0x80) return c;
	return cp1251[c - 0x80];
}

/**
 * Take a character in CP1252 encoding, convert it
 * Courtesy of glib2 and iconv
 *
 */
static unsigned long unicodeFromCP1252(unsigned char c)
{
	static const unsigned int cp1252[32] =
	{
		0x20ac, UNDEF, 0x201a, 0x0192, 0x201e, 0x2026, 0x2020, 0x2021,
		0x02c6, 0x2030, 0x0160, 0x2039, 0x0152, UNDEF, 0x017d, UNDEF,
		UNDEF, 0x2018, 0x2019, 0x201c, 0x201d, 0x2022, 0x2013, 0x2014,
		0x02dc, 0x2122, 0x0161, 0x203a, 0x0153, UNDEF, 0x017e, 0x0178
	};

	if (c < 0x80 || c >= 0xa0) return c;
	return cp1252[c - 0x80];
}

/**
 * Take a character in CP1253 encoding, convert it
 * Courtesy of unicode.org: http://unicode.org/Public/MAPPINGS/VENDORS/MICSFT/WINDOWS/
 *
 */
static unsigned long unicodeFromCP1253(unsigned char c)
{
	static unsigned int const cp1253[] =
	{
		0x20AC, UNDEF /* 0x81 */, 0x201A, 0x0192, 0x201E, 0x2026, 0x2020, 0x2021,
		UNDEF /* 0x88 */, 0x2030, UNDEF /* 0x8a */, 0x2039, UNDEF /* 0x8c */, UNDEF /* 0x8d */, UNDEF /* 0x8e */, UNDEF /* 0x8f */,
		UNDEF /* 0x90 */, 0x2018, 0x2019, 0x201C, 0x201D, 0x2022, 0x2013, 0x2014,
		UNDEF /* 0x98 */, 0x2122, UNDEF /* 0x9a */, 0x203A, UNDEF /* 0x9c */, UNDEF /* 0x9d */, UNDEF /* 0x9e */, UNDEF /* 0x9f */,
		0x00A0, 0x0385, 0x0386, 0x00A3, 0x00A4, 0x00A5, 0x00A6, 0x00A7,
		0x00A8, 0x00A9, UNDEF /* 0xaa */, 0x00AB, 0x00AC, 0x00AD, 0x00AE, 0x2015,
		0x00B0, 0x00B1, 0x00B2, 0x00B3, 0x0384, 0x00B5, 0x00B6, 0x00B7,
		0x0388, 0x0389, 0x038A, 0x00BB, 0x038C, 0x00BD, 0x038E, 0x038F,
		0x0390, 0x0391, 0x0392, 0x0393, 0x0394, 0x0395, 0x0396, 0x0397,
		0x0398, 0x0399, 0x039A, 0x039B, 0x039C, 0x039D, 0x039E, 0x039F,
		0x03A0, 0x03A1, UNDEF /* 0xd2 */, 0x03A3, 0x03A4, 0x03A5, 0x03A6, 0x03A7,
		0x03A8, 0x03A9, 0x03AA, 0x03AB, 0x03AC, 0x03AD, 0x03AE, 0x03AF,
		0x03B0, 0x03B1, 0x03B2, 0x03B3, 0x03B4, 0x03B5, 0x03B6, 0x03B7,
		0x03B8, 0x03B9, 0x03BA, 0x03BB, 0x03BC, 0x03BD, 0x03BE, 0x03BF,
		0x03C0, 0x03C1, 0x03C2, 0x03C3, 0x03C4, 0x03C5, 0x03C6, 0x03C7,
		0x03C8, 0x03C9, 0x03CA, 0x03CB, 0x03CC, 0x03CD, 0x03CE, UNDEF /* 0xff */,
	};
	if (c < 0x80) return c;
	return cp1253[c - 0x80];
}

/**
 * Take a character in CP1254 encoding, convert it
 * Courtesy of unicode.org: http://unicode.org/Public/MAPPINGS/VENDORS/MICSFT/WINDOWS/
 *
 */
static unsigned long unicodeFromCP1254(unsigned char c)
{
	static unsigned int const cp1254[] =
	{
		0x20AC, UNDEF /* 0x81 */, 0x201A, 0x0192, 0x201E, 0x2026, 0x2020, 0x2021,
		0x02C6, 0x2030, 0x0160, 0x2039, 0x0152, UNDEF /* 0x8d */, UNDEF /* 0x8e */, UNDEF /* 0x8f */,
		UNDEF /* 0x90 */, 0x2018, 0x2019, 0x201C, 0x201D, 0x2022, 0x2013, 0x2014,
		0x02DC, 0x2122, 0x0161, 0x203A, 0x0153, UNDEF /* 0x9d */, UNDEF /* 0x9e */, 0x0178,
		0x00A0, 0x00A1, 0x00A2, 0x00A3, 0x00A4, 0x00A5, 0x00A6, 0x00A7,
		0x00A8, 0x00A9, 0x00AA, 0x00AB, 0x00AC, 0x00AD, 0x00AE, 0x00AF,
		0x00B0, 0x00B1, 0x00B2, 0x00B3, 0x00B4, 0x00B5, 0x00B6, 0x00B7,
		0x00B8, 0x00B9, 0x00BA, 0x00BB, 0x00BC, 0x00BD, 0x00BE, 0x00BF,
		0x00C0, 0x00C1, 0x00C2, 0x00C3, 0x00C4, 0x00C5, 0x00C6, 0x00C7,
		0x00C8, 0x00C9, 0x00CA, 0x00CB, 0x00CC, 0x00CD, 0x00CE, 0x00CF,
		0x011E, 0x00D1, 0x00D2, 0x00D3, 0x00D4, 0x00D5, 0x00D6, 0x00D7,
		0x00D8, 0x00D9, 0x00DA, 0x00DB, 0x00DC, 0x0130, 0x015E, 0x00DF,
		0x00E0, 0x00E1, 0x00E2, 0x00E3, 0x00E4, 0x00E5, 0x00E6, 0x00E7,
		0x00E8, 0x00E9, 0x00EA, 0x00EB, 0x00EC, 0x00ED, 0x00EE, 0x00EF,
		0x011F, 0x00F1, 0x00F2, 0x00F3, 0x00F4, 0x00F5, 0x00F6, 0x00F7,
		0x00F8, 0x00F9, 0x00FA, 0x00FB, 0x00FC, 0x0131, 0x015F, 0x00FF,
	};
	if (c < 0x80) return c;
	return cp1254[c - 0x80];
}

/**
 * Take a character in CP1255 encoding, convert it
 * Courtesy of unicode.org: http://unicode.org/Public/MAPPINGS/VENDORS/MICSFT/WINDOWS/
 **/
static unsigned long unicodeFromCP1255(unsigned char c)
{
	static int const cp1255[] =
	{
		0x20AC, UNDEF /* 0x81 */, 0x201A, 0x0192, 0x201E, 0x2026, 0x2020, 0x2021,
		0x02C6, 0x2030, UNDEF /* 0x8a */, 0x2039, UNDEF /* 0x8c */, UNDEF /* 0x8d */, UNDEF /* 0x8e */, UNDEF /* 0x8f */,
		UNDEF /* 0x90 */, 0x2018, 0x2019, 0x201C, 0x201D, 0x2022, 0x2013, 0x2014,
		0x02DC, 0x2122, UNDEF /* 0x9a */, 0x203A, UNDEF /* 0x9c */, UNDEF /* 0x9d */, UNDEF /* 0x9e */, UNDEF /* 0x9f */,
		0x00A0, 0x00A1, 0x00A2, 0x00A3, 0x20AA, 0x00A5, 0x00A6, 0x00A7,
		0x00A8, 0x00A9, 0x00D7, 0x00AB, 0x00AC, 0x00AD, 0x00AE, 0x00AF,
		0x00B0, 0x00B1, 0x00B2, 0x00B3, 0x00B4, 0x00B5, 0x00B6, 0x00B7,
		0x00B8, 0x00B9, 0x00F7, 0x00BB, 0x00BC, 0x00BD, 0x00BE, 0x00BF,
		0x05B0, 0x05B1, 0x05B2, 0x05B3, 0x05B4, 0x05B5, 0x05B6, 0x05B7,
		0x05B8, 0x05B9, UNDEF /* 0xca */, 0x05BB, 0x05BC, 0x05BD, 0x05BE, 0x05BF,
		0x05C0, 0x05C1, 0x05C2, 0x05C3, 0x05F0, 0x05F1, 0x05F2, 0x05F3,
		0x05F4, UNDEF /* 0xd9 */, UNDEF /* 0xda */, UNDEF /* 0xdb */, UNDEF /* 0xdc */, UNDEF /* 0xdd */, UNDEF /* 0xde */, UNDEF /* 0xdf */,
		0x05D0, 0x05D1, 0x05D2, 0x05D3, 0x05D4, 0x05D5, 0x05D6, 0x05D7,
		0x05D8, 0x05D9, 0x05DA, 0x05DB, 0x05DC, 0x05DD, 0x05DE, 0x05DF,
		0x05E0, 0x05E1, 0x05E2, 0x05E3, 0x05E4, 0x05E5, 0x05E6, 0x05E7,
		0x05E8, 0x05E9, 0x05EA, UNDEF /* 0xfb */, UNDEF /* 0xfc */, 0x200E, 0x200F, UNDEF /* 0xff */,
	};
	if (c < 0x80) return c;
	return (unsigned long)cp1255[c - 0x80];
}

/**
 * Take a character in CP1256 encoding, convert it
 * Courtesy of unicode.org: http://unicode.org/Public/MAPPINGS/VENDORS/MICSFT/WINDOWS/
 **/
static unsigned long unicodeFromCP1256(unsigned char c)
{
	static int const cp1256[] =
	{
		0x20AC, 0x067E, 0x201A, 0x0192, 0x201E, 0x2026, 0x2020, 0x2021,
		0x02C6, 0x2030, 0x0679, 0x2039, 0x0152, 0x0686, 0x0698, 0x0688,
		0x06AF, 0x2018, 0x2019, 0x201C, 0x201D, 0x2022, 0x2013, 0x2014,
		0x06A9, 0x2122, 0x0691, 0x203A, 0x0153, 0x200C, 0x200D, 0x06BA,
		0x00A0, 0x060C, 0x00A2, 0x00A3, 0x00A4, 0x00A5, 0x00A6, 0x00A7,
		0x00A8, 0x00A9, 0x06BE, 0x00AB, 0x00AC, 0x00AD, 0x00AE, 0x00AF,
		0x00B0, 0x00B1, 0x00B2, 0x00B3, 0x00B4, 0x00B5, 0x00B6, 0x00B7,
		0x00B8, 0x00B9, 0x061B, 0x00BB, 0x00BC, 0x00BD, 0x00BE, 0x061F,
		0x06C1, 0x0621, 0x0622, 0x0623, 0x0624, 0x0625, 0x0626, 0x0627,
		0x0628, 0x0629, 0x062A, 0x062B, 0x062C, 0x062D, 0x062E, 0x062F,
		0x0630, 0x0631, 0x0632, 0x0633, 0x0634, 0x0635, 0x0636, 0x00D7,
		0x0637, 0x0638, 0x0639, 0x063A, 0x0640, 0x0641, 0x0642, 0x0643,
		0x00E0, 0x0644, 0x00E2, 0x0645, 0x0646, 0x0647, 0x0648, 0x00E7,
		0x00E8, 0x00E9, 0x00EA, 0x00EB, 0x0649, 0x064A, 0x00EE, 0x00EF,
		0x064B, 0x064C, 0x064D, 0x064E, 0x00F4, 0x064F, 0x0650, 0x00F7,
		0x0651, 0x00F9, 0x0652, 0x00FB, 0x00FC, 0x200E, 0x200F, 0x06D2,
	};
	if (c < 0x80) return c;
	return (unsigned long) cp1256[c - 0x80];
}


/**
 * Take a character in CP1257 encoding, convert it
 * Courtesy of unicode.org: http://unicode.org/Public/MAPPINGS/VENDORS/MICSFT/WINDOWS/
 **/
static unsigned long unicodeFromCP1257(unsigned char c)
{
	static unsigned int const cp1257[] =
	{
		0x20AC, UNDEF /* 0x81 */, 0x201A, UNDEF /* 0x83 */, 0x201E, 0x2026, 0x2020, 0x2021,
		UNDEF /* 0x88 */, 0x2030, UNDEF /* 0x8a */, 0x2039, UNDEF /* 0x8c */, 0x00A8, 0x02C7, 0x00B8,
		UNDEF /* 0x90 */, 0x2018, 0x2019, 0x201C, 0x201D, 0x2022, 0x2013, 0x2014,
		UNDEF /* 0x98 */, 0x2122, UNDEF /* 0x9a */, 0x203A, UNDEF /* 0x9c */, 0x00AF, 0x02DB, UNDEF /* 0x9f */,
		0x00A0, UNDEF /* 0xa1 */, 0x00A2, 0x00A3, 0x00A4, UNDEF /* 0xa5 */, 0x00A6, 0x00A7,
		0x00D8, 0x00A9, 0x0156, 0x00AB, 0x00AC, 0x00AD, 0x00AE, 0x00C6,
		0x00B0, 0x00B1, 0x00B2, 0x00B3, 0x00B4, 0x00B5, 0x00B6, 0x00B7,
		0x00F8, 0x00B9, 0x0157, 0x00BB, 0x00BC, 0x00BD, 0x00BE, 0x00E6,
		0x0104, 0x012E, 0x0100, 0x0106, 0x00C4, 0x00C5, 0x0118, 0x0112,
		0x010C, 0x00C9, 0x0179, 0x0116, 0x0122, 0x0136, 0x012A, 0x013B,
		0x0160, 0x0143, 0x0145, 0x00D3, 0x014C, 0x00D5, 0x00D6, 0x00D7,
		0x0172, 0x0141, 0x015A, 0x016A, 0x00DC, 0x017B, 0x017D, 0x00DF,
		0x0105, 0x012F, 0x0101, 0x0107, 0x00E4, 0x00E5, 0x0119, 0x0113,
		0x010D, 0x00E9, 0x017A, 0x0117, 0x0123, 0x0137, 0x012B, 0x013C,
		0x0161, 0x0144, 0x0146, 0x00F3, 0x014D, 0x00F5, 0x00F6, 0x00F7,
		0x0173, 0x0142, 0x015B, 0x016B, 0x00FC, 0x017C, 0x017E, 0x02D9,
	};
	if (c < 0x80) return c;
	return  cp1257[c - 0x80];
}

/**
 * Take a character in CP1258 encoding, convert it
 * Courtesy of unicode.org: http://unicode.org/Public/MAPPINGS/VENDORS/MICSFT/WINDOWS/
 **/
static unsigned long unicodeFromCP1258(unsigned char c)
{
	static unsigned int const cp1258[] =
	{
		0x20AC, UNDEF /* 0x81 */, 0x201A, 0x0192, 0x201E, 0x2026, 0x2020, 0x2021,
		0x02C6, 0x2030, UNDEF /* 0x8a */, 0x2039, 0x0152, UNDEF /* 0x8d */, UNDEF /* 0x8e */, UNDEF /* 0x8f */,
		UNDEF /* 0x90 */, 0x2018, 0x2019, 0x201C, 0x201D, 0x2022, 0x2013, 0x2014,
		0x02DC, 0x2122, UNDEF /* 0x9a */, 0x203A, 0x0153, UNDEF /* 0x9d */, UNDEF /* 0x9e */, 0x0178,
		0x00A0, 0x00A1, 0x00A2, 0x00A3, 0x00A4, 0x00A5, 0x00A6, 0x00A7,
		0x00A8, 0x00A9, 0x00AA, 0x00AB, 0x00AC, 0x00AD, 0x00AE, 0x00AF,
		0x00B0, 0x00B1, 0x00B2, 0x00B3, 0x00B4, 0x00B5, 0x00B6, 0x00B7,
		0x00B8, 0x00B9, 0x00BA, 0x00BB, 0x00BC, 0x00BD, 0x00BE, 0x00BF,
		0x00C0, 0x00C1, 0x00C2, 0x0102, 0x00C4, 0x00C5, 0x00C6, 0x00C7,
		0x00C8, 0x00C9, 0x00CA, 0x00CB, 0x0300, 0x00CD, 0x00CE, 0x00CF,
		0x0110, 0x00D1, 0x0309, 0x00D3, 0x00D4, 0x01A0, 0x00D6, 0x00D7,
		0x00D8, 0x00D9, 0x00DA, 0x00DB, 0x00DC, 0x01AF, 0x0303, 0x00DF,
		0x00E0, 0x00E1, 0x00E2, 0x0103, 0x00E4, 0x00E5, 0x00E6, 0x00E7,
		0x00E8, 0x00E9, 0x00EA, 0x00EB, 0x0301, 0x00ED, 0x00EE, 0x00EF,
		0x0111, 0x00F1, 0x0323, 0x00F3, 0x00F4, 0x01A1, 0x00F6, 0x00F7,
		0x00F8, 0x00F9, 0x00FA, 0x00FB, 0x00FC, 0x01B0, 0x20AB, 0x00FF,
	};
	if (c < 0x80) return c;
	return cp1258[c - 0x80];
}

unsigned long Font::unicode(unsigned char c, Font::Type type)
{
	switch (type)
	{
	case CP_424:
		return unicodeFromCP424(c);
	case CP_437:
		return unicodeFromCP437(c);
	case CP_737:
		return unicodeFromCP737(c);
	case CP_775:
		return unicodeFromCP775(c);
	case DOS_850:
		return unicodeFromCP850(c);
	case CP_852:
		return unicodeFromCP852(c);
	case CP_855:
		return unicodeFromCP855(c);
	case CP_856:
		return unicodeFromCP856(c);
	case CP_857:
		return unicodeFromCP857(c);
	case CP_860:
		return unicodeFromCP860(c);
	case CP_861:
		return unicodeFromCP861(c);
	case CP_862:
		return unicodeFromCP862(c);
	case CP_863:
		return unicodeFromCP863(c);
	case CP_864:
		return unicodeFromCP864(c);
	case CP_865:
		return unicodeFromCP865(c);
	case CP_866:
		return unicodeFromCP866(c);
	case CP_869:
		return unicodeFromCP869(c);
	case CP_874:
		return unicodeFromCP874(c);
	case CP_1006:
		return unicodeFromCP1006(c);
	case WIN3_ARABIC:
		return unicodeFromCP1256(c);
	case WIN3_BALTIC:
		return unicodeFromCP1257(c);
	case WIN3_CEUROPE:
		return unicodeFromCP1250(c);
	case WIN3_CYRILLIC:
		return unicodeFromCP1251(c);
	case WIN3_GREEK:
		return unicodeFromCP1253(c);
	case WIN3_HEBREW:
		return unicodeFromCP1255(c);
	case WIN3_TURKISH:
		return unicodeFromCP1254(c);
	case WIN3_VIETNAMESE:
		return unicodeFromCP1258(c);
	case WIN3_WEUROPE:
		return unicodeFromCP1252(c);
	case UNKNOWN:
	default:
		assert(0);
		return c;
	}
}

////////////////////////////////////////////////////////////
//
// Other
//
////////////////////////////////////////////////////////////

long Language::getDefault()
{
	return 0x409;
}

std::string Language::name(long id)
{
	switch (id)
	{
	case 0x400:
		return "none";
	case 0x401:
		return "arabic";
	case 0x402:
		return "bulgarian";
	case 0x403:
		return "catalan";
	case 0x404:
		return "chinese(Trad)";
	case 0x405:
		return "czech";
	case 0x406:
		return "danish";
	case 0x407:
		return "german";
	case 0x408:
		return "greek";
	case 0x409:
		return "english(US)";
	case 0x40A:
		return "spanish";
	case 0x40B:
		return "finish";
	case 0x40C:
		return "french";
	case 0x40D:
		return "hebrew";
	case 0x40E:
		return "hungarian";
	case 0x40F:
		return "islandic";
	case 0x410:
		return "italian";
	case 0x411:
		return "japonese";
	case 0x412:
		return "korean";
	case 0x413:
		return "dutch";
	case 0x414:
		return "norvegian";
	case 0x415:
		return "polish";
	case 0x416:
		return "portuguese(Brazil)";
	case 0x417:
		return "rhaeto(Romanic)";
	case 0x418:
		return "romania";
	case 0x419:
		return "russian";
	case 0x41D:
		return "swedish";
	case 0x420:
		return "croatian";
	case 0x809:
		return "english(UK)";
	case 0x80a:
		return "spanish(Mexican)";
	case 0x816:
		return "portuguese";
	case 0xC09:
		return "englAUS";
	case 0xC0A:
		return "spanish(Modern)";
	case 0xC0C:
		return "french(Canadian)";
	case 0x1009:
		return "englCan";
	case 0x100c:
		return "french(Swiss)";
	case 0x2C0A:
		return "spanish(Argentina)";
	case 0x3409:
		return "english(Philippines)";
	case 0x480A:
		return "spanish(Honduras)";
	default:
		break;
	}

	std::stringstream f;
	f << "###unkn=" << std::hex << id;
	return f.str();
}

std::string Language::localeName(long id)
{
	switch (id)
	{
	case 0x400:
		return "";
	case 0x401:
		return "ar_DZ"; // checkme
	case 0x402:
		return "bg_BG";
	case 0x403:
		return "ca_ES";
	case 0x404:
		return "zh_TW";
	case 0x405:
		return "cs_CZ";
	case 0x406:
		return "da_DK";
	case 0x407:
		return "de_DE";
	case 0x408:
		return "el_GR";
	case 0x409:
		return "en_US";
	case 0x40A:
		return "es_ES";
	case 0x40B:
		return "fi_FI";
	case 0x40C:
		return "fr_FR";
	case 0x40D:
		return "iw_IL";
	case 0x40E:
		return "hu_HU";
	case 0x40F:
		return "is_IS";
	case 0x410:
		return "it_IT";
	case 0x411:
		return "ja_JP";
	case 0x412:
		return "ko_KR";
	case 0x413:
		return "nl_NL";
	case 0x414:
		return "no_NO";
	case 0x415:
		return "pl_PL";
	case 0x416:
		return "pt_BR";
	case 0x417:
		return "rm_CH";
	case 0x418:
		return "ro_RO";
	case 0x419:
		return "ru_RU";
	case 0x41d:
		return "sv_SE";
	case 0x420:
		return "hr_HR";
	case 0x809:
		return "en_GB";
	case 0x80a:
		return "es_MX";
	case 0x816:
		return "pt_PT";
	case 0xC09:
		return "en_AU";
	case 0xC0a:
		return "es_ES";
	case 0xC0c:
		return "fr_CA";
	case 0x1009:
		return "en_CA";
	case 0x100c:
		return "fr_CH";
	case 0x2c0a:
		return "es_AR";
	case 0x3409:
		return "en_PH"; // english Philippines
	case 0x480A:
		return "es_HN";
	default:
		return "";
	}
}

void Language::addLocaleName(long id, librevenge::RVNGPropertyList &propList)
{
	if (id<0) return;
	std::string lang = libwps_tools_win::Language::localeName(id);
	if (lang.length())
	{
		std::string language(lang);
		std::string country("none");
		if (lang.length() > 3 && lang[2]=='_')
		{
			country=lang.substr(3);
			language=lang.substr(0,2);
		}
		propList.insert("fo:language", language.c_str());
		propList.insert("fo:country", country.c_str());
	}
	else
	{
		propList.insert("fo:language", "none");
		propList.insert("fo:country", "none");
	}
}

}
/* vim:set shiftwidth=4 softtabstop=4 noexpandtab: */
