/*
 * Library store_block type testing program
 *
 * Copyright (C) 2011-2016, Joachim Metz <joachim.metz@gmail.com>
 *
 * Refer to AUTHORS for acknowledgements.
 *
 * This software is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <common.h>
#include <file_stream.h>
#include <types.h>

#if defined( HAVE_STDLIB_H ) || defined( WINAPI )
#include <stdlib.h>
#endif

#include "vshadow_test_libcerror.h"
#include "vshadow_test_libvshadow.h"
#include "vshadow_test_macros.h"
#include "vshadow_test_memory.h"
#include "vshadow_test_unused.h"

#include "../libvshadow/libvshadow_store_block.h"

uint8_t vshadow_test_store_block_header_data[ 128 ] = {
	0x6b, 0x87, 0x08, 0x38, 0x76, 0xc1, 0x48, 0x4e, 0xb7, 0xae, 0x04, 0x04, 0x6e, 0x6c, 0xc7, 0x52,
	0x01, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x28, 0x7e, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x64, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

uint8_t vshadow_test_store_block_error_header_data[ 128 ] = {
	0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	0x01, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x28, 0x7e, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x64, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

#if defined( __GNUC__ )

/* Tests the libvshadow_store_block_initialize function
 * Returns 1 if successful or 0 if not
 */
int vshadow_test_store_block_initialize(
     void )
{
	libcerror_error_t *error              = NULL;
	libvshadow_store_block_t *store_block = NULL;
	int result                            = 0;

#if defined( HAVE_VSHADOW_TEST_MEMORY )
	int number_of_malloc_fail_tests       = 2;
	int number_of_memset_fail_tests       = 1;
	int test_number                       = 0;
#endif

	/* Test store_block initialization
	 */
	result = libvshadow_store_block_initialize(
	          &store_block,
	          4096,
	          &error );

	VSHADOW_TEST_ASSERT_EQUAL_INT(
	 "result",
	 result,
	 1 );

        VSHADOW_TEST_ASSERT_IS_NOT_NULL(
         "store_block",
         store_block );

        VSHADOW_TEST_ASSERT_IS_NULL(
         "error",
         error );

	result = libvshadow_store_block_free(
	          &store_block,
	          &error );

	VSHADOW_TEST_ASSERT_EQUAL_INT(
	 "result",
	 result,
	 1 );

        VSHADOW_TEST_ASSERT_IS_NULL(
         "store_block",
         store_block );

        VSHADOW_TEST_ASSERT_IS_NULL(
         "error",
         error );

	/* Test error cases
	 */
	result = libvshadow_store_block_initialize(
	          NULL,
	          4096,
	          &error );

	VSHADOW_TEST_ASSERT_EQUAL_INT(
	 "result",
	 result,
	 -1 );

        VSHADOW_TEST_ASSERT_IS_NOT_NULL(
         "error",
         error );

	libcerror_error_free(
	 &error );

	store_block = (libvshadow_store_block_t *) 0x12345678UL;

	result = libvshadow_store_block_initialize(
	          &store_block,
	          4096,
	          &error );

	VSHADOW_TEST_ASSERT_EQUAL_INT(
	 "result",
	 result,
	 -1 );

        VSHADOW_TEST_ASSERT_IS_NOT_NULL(
         "error",
         error );

	libcerror_error_free(
	 &error );

	store_block = NULL;

	result = libvshadow_store_block_initialize(
	          &store_block,
	          (size_t) SSIZE_MAX + 1,
	          &error );

	VSHADOW_TEST_ASSERT_EQUAL_INT(
	 "result",
	 result,
	 -1 );

        VSHADOW_TEST_ASSERT_IS_NOT_NULL(
         "error",
         error );

	libcerror_error_free(
	 &error );

#if defined( HAVE_VSHADOW_TEST_MEMORY )

	for( test_number = 0;
	     test_number < number_of_malloc_fail_tests;
	     test_number++ )
	{
		/* Test libvshadow_store_block_initialize with malloc failing
		 */
		vshadow_test_malloc_attempts_before_fail = test_number;

		result = libvshadow_store_block_initialize(
		          &store_block,
		          4096,
		          &error );

		if( vshadow_test_malloc_attempts_before_fail != -1 )
		{
			vshadow_test_malloc_attempts_before_fail = -1;

			if( store_block != NULL )
			{
				libvshadow_store_block_free(
				 &store_block,
				 NULL );
			}
		}
		else
		{
			VSHADOW_TEST_ASSERT_EQUAL_INT(
			 "result",
			 result,
			 -1 );

			VSHADOW_TEST_ASSERT_IS_NULL(
			 "store_block",
			 store_block );

			VSHADOW_TEST_ASSERT_IS_NOT_NULL(
			 "error",
			 error );

			libcerror_error_free(
			 &error );
		}
	}
	for( test_number = 0;
	     test_number < number_of_memset_fail_tests;
	     test_number++ )
	{
		/* Test libvshadow_store_block_initialize with memset failing
		 */
		vshadow_test_memset_attempts_before_fail = test_number;

		result = libvshadow_store_block_initialize(
		          &store_block,
		          4096,
		          &error );

		if( vshadow_test_memset_attempts_before_fail != -1 )
		{
			vshadow_test_memset_attempts_before_fail = -1;

			if( store_block != NULL )
			{
				libvshadow_store_block_free(
				 &store_block,
				 NULL );
			}
		}
		else
		{
			VSHADOW_TEST_ASSERT_EQUAL_INT(
			 "result",
			 result,
			 -1 );

			VSHADOW_TEST_ASSERT_IS_NULL(
			 "store_block",
			 store_block );

			VSHADOW_TEST_ASSERT_IS_NOT_NULL(
			 "error",
			 error );

			libcerror_error_free(
			 &error );
		}
	}
#endif /* defined( HAVE_VSHADOW_TEST_MEMORY ) */

	return( 1 );

on_error:
	if( error != NULL )
	{
		libcerror_error_free(
		 &error );
	}
	if( store_block != NULL )
	{
		libvshadow_store_block_free(
		 &store_block,
		 NULL );
	}
	return( 0 );
}

/* Tests the libvshadow_store_block_free function
 * Returns 1 if successful or 0 if not
 */
int vshadow_test_store_block_free(
     void )
{
	libcerror_error_t *error = NULL;
	int result               = 0;

	/* Test error cases
	 */
	result = libvshadow_store_block_free(
	          NULL,
	          &error );

	VSHADOW_TEST_ASSERT_EQUAL_INT(
	 "result",
	 result,
	 -1 );

        VSHADOW_TEST_ASSERT_IS_NOT_NULL(
         "error",
         error );

	libcerror_error_free(
	 &error );

	return( 1 );

on_error:
	if( error != NULL )
	{
		libcerror_error_free(
		 &error );
	}
	return( 0 );
}

/* Tests the libvshadow_store_block_read function
 * Returns 1 if successful or 0 if not
 */
int vshadow_test_store_block_read(
     void )
{
	libbfio_handle_t *file_io_handle      = NULL;
	libcerror_error_t *error              = NULL;
	libvshadow_store_block_t *store_block = NULL;
	int result                            = 0;

	/* Initialize test
	 */
	result = libbfio_memory_range_initialize(
	          &file_io_handle,
	          &error );

	VSHADOW_TEST_ASSERT_EQUAL_INT(
	 "result",
	 result,
	 1 );

        VSHADOW_TEST_ASSERT_IS_NOT_NULL(
         "file_io_handle",
         file_io_handle );

        VSHADOW_TEST_ASSERT_IS_NULL(
         "error",
         error );

/* TODO allocate a 16k buffer and copy the header data */
	result = libbfio_memory_range_set(
	          file_io_handle,
	          vshadow_test_store_block_header_data,
	          128,
	          &error );

	VSHADOW_TEST_ASSERT_EQUAL_INT(
	 "result",
	 result,
	 1 );

        VSHADOW_TEST_ASSERT_IS_NULL(
         "error",
         error );

	result = libbfio_handle_open(
	          file_io_handle,
	          LIBBFIO_OPEN_READ,
	          &error );

	VSHADOW_TEST_ASSERT_EQUAL_INT(
	 "result",
	 result,
	 1 );

        VSHADOW_TEST_ASSERT_IS_NULL(
         "error",
         error );

	result = libvshadow_store_block_initialize(
	          &store_block,
	          128,
	          &error );

	VSHADOW_TEST_ASSERT_EQUAL_INT(
	 "result",
	 result,
	 1 );

        VSHADOW_TEST_ASSERT_IS_NOT_NULL(
         "store_block",
         store_block );

        VSHADOW_TEST_ASSERT_IS_NULL(
         "error",
         error );

	/* Test regular cases
	 */
	result = libvshadow_store_block_read(
	          store_block,
	          file_io_handle,
	          0,
	          &error );

	VSHADOW_TEST_FPRINT_ERROR( error );

	VSHADOW_TEST_ASSERT_EQUAL_INT(
	 "result",
	 result,
	 1 );

        VSHADOW_TEST_ASSERT_IS_NULL(
         "error",
         error );

	/* Test error cases
	 */
	result = libvshadow_store_block_read(
	          NULL,
	          file_io_handle,
	          0,
	          &error );

	VSHADOW_TEST_ASSERT_EQUAL_INT(
	 "result",
	 result,
	 -1 );

        VSHADOW_TEST_ASSERT_IS_NOT_NULL(
         "error",
         error );

	libcerror_error_free(
	 &error );

	result = libvshadow_store_block_read(
	          store_block,
	          NULL,
	          0,
	          &error );

	VSHADOW_TEST_ASSERT_EQUAL_INT(
	 "result",
	 result,
	 -1 );

        VSHADOW_TEST_ASSERT_IS_NOT_NULL(
         "error",
         error );

	libcerror_error_free(
	 &error );

	result = libvshadow_store_block_read(
	          store_block,
	          file_io_handle,
	          -1,
	          &error );

	VSHADOW_TEST_ASSERT_EQUAL_INT(
	 "result",
	 result,
	 -1 );

        VSHADOW_TEST_ASSERT_IS_NOT_NULL(
         "error",
         error );

	libcerror_error_free(
	 &error );

	/* Clean up
	 */
	result = libvshadow_store_block_free(
	          &store_block,
	          &error );

	VSHADOW_TEST_ASSERT_EQUAL_INT(
	 "result",
	 result,
	 1 );

        VSHADOW_TEST_ASSERT_IS_NULL(
         "store_block",
         store_block );

        VSHADOW_TEST_ASSERT_IS_NULL(
         "error",
         error );

	result = libbfio_handle_close(
	          file_io_handle,
	          &error );

	VSHADOW_TEST_ASSERT_EQUAL_INT(
	 "result",
	 result,
	 0 );

        VSHADOW_TEST_ASSERT_IS_NULL(
         "error",
         error );

	result = libbfio_handle_free(
	          &file_io_handle,
	          &error );

	VSHADOW_TEST_ASSERT_EQUAL_INT(
	 "result",
	 result,
	 1 );

        VSHADOW_TEST_ASSERT_IS_NULL(
         "file_io_handle",
         file_io_handle );

        VSHADOW_TEST_ASSERT_IS_NULL(
         "error",
         error );

	return( 1 );

on_error:
	if( error != NULL )
	{
		libcerror_error_free(
		 &error );
	}
	if( store_block != NULL )
	{
		libvshadow_store_block_free(
		 &store_block,
		 NULL );
	}
	if( file_io_handle != NULL )
	{
		libbfio_handle_free(
		 &file_io_handle,
		 NULL );
	}
	return( 0 );
}

/* Tests the libvshadow_store_block_read_header_data function
 * Returns 1 if successful or 0 if not
 */
int vshadow_test_store_block_read_header_data(
     void )
{
	libcerror_error_t *error              = NULL;
	libvshadow_store_block_t *store_block = NULL;
	int result                            = 0;

	/* Initialize test
	 */
	result = libvshadow_store_block_initialize(
	          &store_block,
	          4096,
	          &error );

	VSHADOW_TEST_ASSERT_EQUAL_INT(
	 "result",
	 result,
	 1 );

        VSHADOW_TEST_ASSERT_IS_NOT_NULL(
         "store_block",
         store_block );

        VSHADOW_TEST_ASSERT_IS_NULL(
         "error",
         error );

	/* Test regular cases
	 */
	result = libvshadow_store_block_read_header_data(
	          store_block,
	          vshadow_test_store_block_header_data,
	          128,
	          &error );

	VSHADOW_TEST_ASSERT_EQUAL_INT(
	 "result",
	 result,
	 1 );

        VSHADOW_TEST_ASSERT_IS_NULL(
         "error",
         error );

	/* Test error cases
	 */
	result = libvshadow_store_block_read_header_data(
	          NULL,
	          vshadow_test_store_block_header_data,
	          128,
	          &error );

	VSHADOW_TEST_ASSERT_EQUAL_INT(
	 "result",
	 result,
	 -1 );

        VSHADOW_TEST_ASSERT_IS_NOT_NULL(
         "error",
         error );

	libcerror_error_free(
	 &error );

	result = libvshadow_store_block_read_header_data(
	          store_block,
	          NULL,
	          128,
	          &error );

	VSHADOW_TEST_ASSERT_EQUAL_INT(
	 "result",
	 result,
	 -1 );

        VSHADOW_TEST_ASSERT_IS_NOT_NULL(
         "error",
         error );

	libcerror_error_free(
	 &error );

	result = libvshadow_store_block_read_header_data(
	          store_block,
	          vshadow_test_store_block_header_data,
	          127,
	          &error );

	VSHADOW_TEST_ASSERT_EQUAL_INT(
	 "result",
	 result,
	 -1 );

        VSHADOW_TEST_ASSERT_IS_NOT_NULL(
         "error",
         error );

	libcerror_error_free(
	 &error );

	result = libvshadow_store_block_read_header_data(
	          store_block,
	          vshadow_test_store_block_header_data,
	          (size_t) SSIZE_MAX + 1,
	          &error );

	VSHADOW_TEST_ASSERT_EQUAL_INT(
	 "result",
	 result,
	 -1 );

        VSHADOW_TEST_ASSERT_IS_NOT_NULL(
         "error",
         error );

	libcerror_error_free(
	 &error );

	/* Test error case where the signature does not match
	 */
	result = libvshadow_store_block_read_header_data(
	          store_block,
	          vshadow_test_store_block_error_header_data,
	          128,
	          &error );

	VSHADOW_TEST_ASSERT_EQUAL_INT(
	 "result",
	 result,
	 -1 );

        VSHADOW_TEST_ASSERT_IS_NOT_NULL(
         "error",
         error );

	libcerror_error_free(
	 &error );

/* TODO test unsupported version */

	/* Clean up
	 */
	result = libvshadow_store_block_free(
	          &store_block,
	          &error );

	VSHADOW_TEST_ASSERT_EQUAL_INT(
	 "result",
	 result,
	 1 );

        VSHADOW_TEST_ASSERT_IS_NULL(
         "store_block",
         store_block );

        VSHADOW_TEST_ASSERT_IS_NULL(
         "error",
         error );

	return( 1 );

on_error:
	if( error != NULL )
	{
		libcerror_error_free(
		 &error );
	}
	if( store_block != NULL )
	{
		libvshadow_store_block_free(
		 &store_block,
		 NULL );
	}
	return( 0 );
}

#endif /* defined( __GNUC__ ) */

/* The main program
 */
#if defined( HAVE_WIDE_SYSTEM_CHARACTER )
int wmain(
     int argc VSHADOW_TEST_ATTRIBUTE_UNUSED,
     wchar_t * const argv[] VSHADOW_TEST_ATTRIBUTE_UNUSED )
#else
int main(
     int argc VSHADOW_TEST_ATTRIBUTE_UNUSED,
     char * const argv[] VSHADOW_TEST_ATTRIBUTE_UNUSED )
#endif
{
	VSHADOW_TEST_UNREFERENCED_PARAMETER( argc )
	VSHADOW_TEST_UNREFERENCED_PARAMETER( argv )

#if defined( __GNUC__ )

	VSHADOW_TEST_RUN(
	 "libvshadow_store_block_initialize",
	 vshadow_test_store_block_initialize );

	VSHADOW_TEST_RUN(
	 "libvshadow_store_block_free",
	 vshadow_test_store_block_free );

	VSHADOW_TEST_RUN(
	 "libvshadow_store_block_read",
	 vshadow_test_store_block_read );

	VSHADOW_TEST_RUN(
	 "libvshadow_store_block_read_header_data",
	 vshadow_test_store_block_read_header_data );

#endif /* defined( __GNUC__ ) */

	return( EXIT_SUCCESS );

on_error:
	return( EXIT_FAILURE );
}

