/*
 * Licensed to the University Corporation for Advanced Internet Development, 
 * Inc. (UCAID) under one or more contributor license agreements.  See the 
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache 
 * License, Version 2.0 (the "License"); you may not use this file except in 
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package edu.internet2.middleware.shibboleth.common.attribute.resolver.provider.dataConnector;

import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Collection;
import java.util.Collections;
import java.util.Map;

import org.opensaml.xml.util.Base64;
import org.opensaml.xml.util.DatatypeHelper;
import org.opensaml.xml.util.LazyMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import edu.internet2.middleware.shibboleth.common.attribute.BaseAttribute;
import edu.internet2.middleware.shibboleth.common.attribute.provider.BasicAttribute;
import edu.internet2.middleware.shibboleth.common.attribute.resolver.AttributeResolutionException;
import edu.internet2.middleware.shibboleth.common.attribute.resolver.provider.ShibbolethResolutionContext;

/**
 * A data connector that generates a unique ID by computing the SHA-1 hash of a given attribute value, the entity ID of
 * the inbound message issuer, and a provided salt.
 *
 */
public class ComputedIDDataConnector extends BaseDataConnector {

    /** Class logger. */
    private final Logger log = LoggerFactory.getLogger(ComputedIDDataConnector.class);

    /** ID of the attribute generated by this data connector. */
    private String generatedAttribute;

    /** ID of the attribute whose first value is used when generating the computed ID. */
    private String sourceAttribute;

    /** Salt used when computing the ID. */
    private byte[] salt;

    /**
     * Constructor.
     * 
     * @param generatedAttributeId ID of the attribute generated by this data connector
     * @param sourceAttributeId ID of the attribute whose first value is used when generating the computed ID
     * @param idSalt salt used when computing the ID
     */
    public ComputedIDDataConnector(String generatedAttributeId, String sourceAttributeId, byte[] idSalt) {
        log.warn("This data connector is deprecated.  The StoredID data connector should be used in its place.");

        if (DatatypeHelper.isEmpty(generatedAttributeId)) {
            throw new IllegalArgumentException("Provided generated attribute ID must not be empty");
        }
        generatedAttribute = generatedAttributeId;

        if (DatatypeHelper.isEmpty(sourceAttributeId)) {
            throw new IllegalArgumentException("Provided source attribute ID must not be empty");
        }
        sourceAttribute = sourceAttributeId;

        if (idSalt.length < 16) {
            throw new IllegalArgumentException("Provided salt must be at least 16 bytes in size.");
        }
        salt = idSalt;
    }

    /**
     * Gets the salt used when computing the ID.
     * 
     * @return salt used when computing the ID
     */
    public byte[] getSalt() {
        return salt;
    }

    /**
     * Gets the ID of the attribute whose first value is used when generating the computed ID.
     * 
     * @return ID of the attribute whose first value is used when generating the computed ID
     */
    public String getSourceAttributeId() {
        return sourceAttribute;
    }

    /**
     * Gets the ID of the attribute generated by this connector.
     * 
     * @return ID of the attribute generated by this connector
     */
    public String getGeneratedAttributeId() {
        return generatedAttribute;
    }

    /** {@inheritDoc} */
    public Map<String, BaseAttribute> resolve(ShibbolethResolutionContext resolutionContext)
            throws AttributeResolutionException {

        String inboundMessageIssuer = resolutionContext.getAttributeRequestContext().getInboundMessageIssuer();
        if (inboundMessageIssuer == null) {
            log.debug("No inbound message issuer identified, unable to compute ID");
            throw new AttributeResolutionException("No inbound message issuer identified");
        }

        Collection<Object> sourceIdValues = getValuesFromAllDependencies(resolutionContext, getSourceAttributeId());
        if (sourceIdValues == null || sourceIdValues.isEmpty()) {
            log.debug("Source attribute {} for connector {} provide no values", getSourceAttributeId(), getId());
            return Collections.EMPTY_MAP;
        }

        if (sourceIdValues.size() > 1) {
            log.warn("Source attribute {} for connector {} has more than one value, only the first value is used",
                    getSourceAttributeId(), getId());
        }
        String sourceId = sourceIdValues.iterator().next().toString();

        BasicAttribute<String> computedIdAttrib = new BasicAttribute<String>();
        computedIdAttrib.setId(getGeneratedAttributeId());

        try {
            MessageDigest md = MessageDigest.getInstance("SHA");
            md.update(inboundMessageIssuer.getBytes());
            md.update((byte) '!');
            md.update(sourceId.getBytes());
            md.update((byte) '!');

            computedIdAttrib.getValues().add(Base64.encodeBytes(md.digest(salt)));

            LazyMap<String, BaseAttribute> attribtues = new LazyMap<String, BaseAttribute>();
            attribtues.put(getGeneratedAttributeId(), computedIdAttrib);
            return attribtues;
        } catch (NoSuchAlgorithmException e) {
            log.error("JVM error, SHA-1 hash is not supported.");
            throw new AttributeResolutionException("SHA-1A is not supported, unable to compute ID");
        }
    }

    /** {@inheritDoc} */
    public void validate() throws AttributeResolutionException {
        if (getDependencyIds() == null || getDependencyIds().size() != 1) {
            log.error("Computed ID " + getId() + " data connectore requires exactly one dependency");
            throw new AttributeResolutionException("Computed ID " + getId()
                    + " data connectore requires exactly one dependency");
        }
    }
}