/**********************************************************************
 * Copyright (c) 2012 Ericsson
 * 
 * All rights reserved. This program and the accompanying materials are
 * made available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 *   Bernd Hufmann - Initial API and implementation
 **********************************************************************/
package org.eclipse.linuxtools.lttng2.core.tests.control.model.impl;

import java.util.LinkedList;
import java.util.List;

import junit.framework.TestCase;

import org.eclipse.linuxtools.internal.lttng2.core.control.model.IChannelInfo;
import org.eclipse.linuxtools.internal.lttng2.core.control.model.IEventInfo;
import org.eclipse.linuxtools.internal.lttng2.core.control.model.TraceEnablement;
import org.eclipse.linuxtools.internal.lttng2.core.control.model.impl.ChannelInfo;
import org.eclipse.linuxtools.internal.lttng2.core.control.model.impl.EventInfo;

/**
 * The class <code>ChannelInfoTest</code> contains tests for the class <code>{@link ChannelInfo}</code>.
 *
 */
@SuppressWarnings("nls")
public class ChannelInfoTest extends TestCase {
    // ------------------------------------------------------------------------
    // Test data
    // ------------------------------------------------------------------------
    private IChannelInfo fChannelInfo1 = null;
    private IChannelInfo fChannelInfo2 = null;
   
    // ------------------------------------------------------------------------
    // Housekeeping
    // ------------------------------------------------------------------------
   
    /**
     * Perform pre-test initialization.
     *
     * @throws Exception
     *         if the initialization fails for some reason
     *
     */
    @Override
    public void setUp() {
        ModelImplFactory factory = new ModelImplFactory();
        fChannelInfo1 = factory.getChannel1();
        fChannelInfo2 = factory.getChannel2();
    }

    /**
     * Perform post-test clean-up.
     *
     * @throws Exception
     *         if the clean-up fails for some reason
     *
     */
    @Override
    public void tearDown() {
    }

    // ------------------------------------------------------------------------
    // Tests
    // ------------------------------------------------------------------------
    
    // ------------------------------------------------------------------------
    // Constructors
    // ------------------------------------------------------------------------

    /**
     * Run the ChannelInfo() constructor test.
     *
     */
    public void testChannelInfo() {
        ChannelInfo result = new ChannelInfo("test");
        assertNotNull(result);
        
        assertEquals("test", result.getName());
        assertEquals(0, result.getNumberOfSubBuffers());
        assertEquals("", result.getOutputType());
        assertEquals(false, result.isOverwriteMode());
        assertEquals(0, result.getReadTimer());
        assertEquals("disabled", result.getState().getInName());
        assertEquals(0, result.getSubBufferSize());
        assertEquals(0, result.getSwitchTimer());
    }

    public void testChannelInfoCopy() {
        ChannelInfo channelInfo = new ChannelInfo((ChannelInfo)fChannelInfo1);
        
        assertEquals(fChannelInfo1.getName(), channelInfo.getName());
        assertEquals(fChannelInfo1.getNumberOfSubBuffers(), channelInfo.getNumberOfSubBuffers());
        assertEquals(fChannelInfo1.getOutputType(), channelInfo.getOutputType());
        assertEquals(fChannelInfo1.isOverwriteMode(), channelInfo.isOverwriteMode());
        assertEquals(fChannelInfo1.getReadTimer(), channelInfo.getReadTimer());
        assertEquals(fChannelInfo1.getState(), channelInfo.getState());
        assertEquals(fChannelInfo1.getSwitchTimer(), channelInfo.getSwitchTimer());
        assertEquals(fChannelInfo1.getEvents().length, channelInfo.getEvents().length);
        
        IEventInfo[] orignalEvents = fChannelInfo1.getEvents();
        IEventInfo[] resultEvents = channelInfo.getEvents();
        for (int i = 0; i < orignalEvents.length; i++) {
            assertEquals(orignalEvents[i], resultEvents[i]);
        }
    }

    public void testChannelCopy2() {
        try {
            ChannelInfo channel = null;
            new ChannelInfo(channel);
            fail("null copy");
        }
        catch (IllegalArgumentException e) {
            // Success
        }
    }

    /**
     * Run the IEventInfo[] getEvents() method test.
     *
     */
    public void testAddAndGetEvents_1() {
        ChannelInfo fixture = new ChannelInfo("test");
        fixture.setSwitchTimer(1L);
        fixture.setOverwriteMode(true);
        fixture.setReadTimer(1L);
        fixture.setState(TraceEnablement.DISABLED);
        fixture.setNumberOfSubBuffers(1);
        fixture.setOutputType("");
        fixture.setSubBufferSize(1L);
        
        // add an event
        IEventInfo event = new EventInfo("event");
        fixture.addEvent(event);

        // Verify the stored events
        IEventInfo[] result = fixture.getEvents();

        assertNotNull(result);
        assertEquals(1, result.length);
        assertNotNull(result[0]);
        assertTrue(event.equals(result[0]));
    }

    /**
     * Run the long getNumberOfSubBuffers() method test.
     *
     */
    public void testGetAndSetters() {
        ChannelInfo fixture = new ChannelInfo("test");
        fixture.setSwitchTimer(2L);
        fixture.setOverwriteMode(true);
        fixture.setReadTimer(3L);
        fixture.setState(TraceEnablement.DISABLED);
        fixture.setNumberOfSubBuffers(4);
        fixture.setOutputType("splice()");
        fixture.setSubBufferSize(1L);
        fixture.addEvent(new EventInfo("event"));

        long switchTimer = fixture.getSwitchTimer();
        assertEquals(2L, switchTimer);
        
        boolean mode = fixture.isOverwriteMode();
        assertTrue(mode);
        
        long readTimer = fixture.getReadTimer();
        assertEquals(3L, readTimer);
        
        TraceEnablement state = fixture.getState();
        assertEquals("disabled", state.getInName());
        
        long numSubBuffers = fixture.getNumberOfSubBuffers();
        assertEquals(4, numSubBuffers);
        
        String outputType = fixture.getOutputType();
        assertEquals("splice()", outputType);
        
        long subBufferSize = fixture.getSubBufferSize();
        assertEquals(1L, subBufferSize);

        fixture.setSwitchTimer(5L);
        fixture.setOverwriteMode(false);
        fixture.setReadTimer(6L);
        fixture.setState(TraceEnablement.ENABLED);
        fixture.setNumberOfSubBuffers(7);
        fixture.setOutputType("mmap()");
        fixture.setSubBufferSize(8L);

        switchTimer = fixture.getSwitchTimer();
        assertEquals(5L, switchTimer);
        
        mode = fixture.isOverwriteMode();
        assertFalse(mode);
        
        readTimer = fixture.getReadTimer();
        assertEquals(6L, readTimer);
        
        state = fixture.getState();
        assertEquals("enabled", state.getInName());
        
        numSubBuffers = fixture.getNumberOfSubBuffers();
        assertEquals(7, numSubBuffers);
        
        outputType = fixture.getOutputType();
        assertEquals("mmap()", outputType);
        
        subBufferSize = fixture.getSubBufferSize();
        assertEquals(8L, subBufferSize);
    }

    /**
     * Run the void setEvents(List<IEventInfo>) method test.
     *
     */
    public void testSetEvents_1() {
        ChannelInfo fixture = new ChannelInfo("test");
        fixture.setSwitchTimer(1L);
        fixture.setOverwriteMode(true);
        fixture.setReadTimer(1L);
        fixture.setState(TraceEnablement.DISABLED);
        fixture.setNumberOfSubBuffers(1);
        fixture.setOutputType("");
        fixture.setSubBufferSize(1L);
        List<IEventInfo> events = new LinkedList<IEventInfo>();
        
        for (int i = 0; i < 2; i++) {
            IEventInfo info = new EventInfo("event" + i);
            info.setEventType("tracepoint");
            info.setState((i % 2 == 0 ? "enabled" : "disabled"));
            events.add(info);
        }

        fixture.setEvents(events);

        IEventInfo[] infos = fixture.getEvents();
        
        assertEquals(events.size(), infos.length);
        
        for (int i = 0; i < infos.length; i++) {
            assertEquals(events.get(i), infos[i]);
        }
    }

    public void testToString_1() {
        ChannelInfo fixture = new ChannelInfo("channel");
        fixture.setSwitchTimer(1L);
        fixture.setOverwriteMode(true);
        fixture.setReadTimer(1L);
        fixture.setState(TraceEnablement.DISABLED);
        fixture.setNumberOfSubBuffers(1);
        fixture.setOutputType("splice()");
        fixture.setSubBufferSize(1L);

        String result = fixture.toString();

        // add additional test code here
        assertEquals("[ChannelInfo([TraceInfo(Name=channel)],State=DISABLED,OverwriteMode=true,SubBuffersSize=1,NumberOfSubBuffers=1,SwitchTimer=1,ReadTimer=1,output=splice(),Events=None)]", result);
    }

    /**
     * Run the String toString() method test.
     *
     */
    public void testToString_2() {
        String result = fChannelInfo1.toString();

        // add additional test code here
        assertEquals("[ChannelInfo([TraceInfo(Name=channel1)],State=DISABLED,OverwriteMode=true,SubBuffersSize=13,NumberOfSubBuffers=12,SwitchTimer=10,ReadTimer=11,output=splice(),Events=[EventInfo([BaseEventInfo([TraceInfo(Name=event1)],type=TRACEPOINT,level=TRACE_DEBUG)],State=ENABLED)])]", result);
    }
    
    // ------------------------------------------------------------------------
    // equals
    // ------------------------------------------------------------------------

    public void testEqualsReflexivity() {
        assertTrue("equals", fChannelInfo1.equals(fChannelInfo1));
        assertTrue("equals", fChannelInfo2.equals(fChannelInfo2));

        assertTrue("equals", !fChannelInfo1.equals(fChannelInfo2));
        assertTrue("equals", !fChannelInfo2.equals(fChannelInfo1));
    }
    
    public void testEqualsSymmetry() {
        ChannelInfo event1 = new ChannelInfo((ChannelInfo)fChannelInfo1);
        ChannelInfo event2 = new ChannelInfo((ChannelInfo)fChannelInfo2);

        assertTrue("equals", event1.equals(fChannelInfo1));
        assertTrue("equals", fChannelInfo1.equals(event1));

        assertTrue("equals", event2.equals(fChannelInfo2));
        assertTrue("equals", fChannelInfo2.equals(event2));
    }
    
    public void testEqualsTransivity() {
        ChannelInfo channel1 = new ChannelInfo((ChannelInfo)fChannelInfo1);
        ChannelInfo channel2 = new ChannelInfo((ChannelInfo)fChannelInfo1);
        ChannelInfo channel3 = new ChannelInfo((ChannelInfo)fChannelInfo1);

        assertTrue("equals", channel1.equals(channel2));
        assertTrue("equals", channel2.equals(channel3));
        assertTrue("equals", channel1.equals(channel3));
    }
    
    public void testEqualsNull() throws Exception {
        assertTrue("equals", !fChannelInfo1.equals(null));
        assertTrue("equals", !fChannelInfo2.equals(null));
    }
    
    // ------------------------------------------------------------------------
    // hashCode
    // ------------------------------------------------------------------------

    public void testHashCode() {
        ChannelInfo channel1 = new ChannelInfo((ChannelInfo)fChannelInfo1);
        ChannelInfo channel2 = new ChannelInfo((ChannelInfo)fChannelInfo2);

        assertTrue("hashCode", fChannelInfo1.hashCode() == channel1.hashCode());
        assertTrue("hashCode", fChannelInfo2.hashCode() == channel2.hashCode());

        assertTrue("hashCode", fChannelInfo1.hashCode() != channel2.hashCode());
        assertTrue("hashCode", fChannelInfo2.hashCode() != channel1.hashCode());
    }
}