/*
 * OWL (OpenedHand Widget Library) Colour Selector Dialog.
 *
 * Author: Rob Bradford  <rob@openedhand.com>
 *
 * Copyright (C) 2007 OpenedHand Ltd.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "owlcolourselectordialog.h"
#include "owlcolourswatch.h"
#include "owlcolourpalette.h"

#include <gtk/gtk.h>

G_DEFINE_TYPE (OwlColourSelectorDialog, owl_colour_selector_dialog, GTK_TYPE_DIALOG);

#define COLOUR_SELECTOR_DIALOG_PRIVATE(o) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((o), OWL_TYPE_COLOUR_SELECTOR_DIALOG, OwlColourSelectorDialogPrivate))

typedef struct _OwlColourSelectorDialogPrivate OwlColourSelectorDialogPrivate;

struct _OwlColourSelectorDialogPrivate
{
  GtkWidget *frame;
  GtkWidget *main_box;
  GtkWidget *palette;
  GtkWidget *swatch;
  GtkWidget *swatch_box;
  GtkWidget *alignment;

  gint orientation;
};

enum {
  PROP_ORIENTATION = 1
};

GType
owl_colour_selector_dialog_orientation_get_type (void)
{
  static GType etype = 0;
  if (etype == 0) {
    static const GEnumValue values[] = {
      { OWL_COLOUR_SELECTOR_DIALOG_HORIZONTAL, "OWL_COLOUR_SELECTOR_DIALOG_HORIZONTAL", "horizontal" },
      { OWL_COLOUR_SELECTOR_DIALOG_VERTICAL, "OWL_COLOUR_SELECTOR_DIALOG_VERTICAL", "vertical" },
      { OWL_COLOUR_SELECTOR_DIALOG_AUTOMATIC, "OWL_COLOUR_SELECTOR_DIALOG_AUTOMATIC", "automatic" },
      { 0, NULL, NULL }
    };
    etype = g_enum_register_static (g_intern_static_string ("OwlColourSelectorDialogOrientation"), values);
  }
  return etype;
}

#define OWL_TYPE_COLOUR_SELECTOR_DIALOG_ORIENTATION owl_colour_selector_dialog_orientation_get_type ()

static const guint32 colours[] =
                      {0xfce94f00, 0xedd40000, 0xc4a00000,
                       0xfcaf3e00, 0xf5790000, 0xce5c0000,
                       0xe9b96e00, 0xc17d1100, 0x8f590200,
                       0x8ae23400, 0x73d21600, 0x4e9a0600,
                       0x729fcf00, 0x3465a400, 0x204a8700,
                       0xad7fa800, 0x75507b00, 0x5c356600,
                       0xef292900, 0xcc000000, 0xa4000000,
                       0xffffffff, 0xbabdb600, 0x00000000};

static void colour_selected_cb (GtkWidget *widget, OwlColourSwatch *swatch, gpointer user_data);


static void
owl_colour_selector_dialog_get_property (GObject *object, guint property_id,
                              GValue *value, GParamSpec *pspec)
{
  OwlColourSelectorDialogPrivate *priv = COLOUR_SELECTOR_DIALOG_PRIVATE (object);

  switch (property_id) {
    case PROP_ORIENTATION:
      g_value_set_enum (value, priv->orientation);
      break;
  default:
    G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
  }
}

static void
owl_colour_selector_dialog_set_property (GObject *object, guint property_id,
                              const GValue *value, GParamSpec *pspec)
{
  OwlColourSelectorDialogPrivate *priv = COLOUR_SELECTOR_DIALOG_PRIVATE (object);

  switch (property_id) {
    case PROP_ORIENTATION:
      priv->orientation = g_value_get_enum (value);
      break;
  default:
    G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
  }
}

static void
owl_colour_selector_dialog_dispose (GObject *object)
{
  if (G_OBJECT_CLASS (owl_colour_selector_dialog_parent_class)->dispose)
    G_OBJECT_CLASS (owl_colour_selector_dialog_parent_class)->dispose (object);
}

static void
owl_colour_selector_dialog_finalize (GObject *object)
{
  G_OBJECT_CLASS (owl_colour_selector_dialog_parent_class)->finalize (object);
}

static GObject *
owl_colour_selector_dialog_constructor (GType type,
    guint n_construct_properties, GObjectConstructParam *construct_properties)
{
  GObject *obj;
  OwlColourSelectorDialogPrivate *priv; 

  GtkWidget *main_box = NULL;
  GtkWidget *swatch_box = NULL;
  GtkWidget *alignment = NULL;
  GtkWidget *frame = NULL;
  GtkWidget *palette = NULL;
  OwlColourSelectorDialogOrientation orientation;

  {
    /* Invoke parent constructor. */
    OwlColourSelectorDialogClass *klass;
    GObjectClass *parent_class;
    klass = OWL_COLOUR_SELECTOR_DIALOG_CLASS (
        g_type_class_peek (OWL_TYPE_COLOUR_SELECTOR_DIALOG));

    parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));

    obj = parent_class->constructor (type, n_construct_properties,
        construct_properties);
  }

  priv = COLOUR_SELECTOR_DIALOG_PRIVATE (obj); 

  alignment = priv->alignment;
  frame = priv->frame;
  palette = priv->palette;

  if (priv->orientation == OWL_COLOUR_SELECTOR_DIALOG_AUTOMATIC)
  {
    if (gdk_screen_width () > gdk_screen_height ())
      orientation = OWL_COLOUR_SELECTOR_DIALOG_HORIZONTAL;
    else
      orientation = OWL_COLOUR_SELECTOR_DIALOG_VERTICAL;

  } else {
    orientation = priv->orientation;
  }

  if (orientation == OWL_COLOUR_SELECTOR_DIALOG_VERTICAL)
  {
    main_box = gtk_vbox_new (FALSE, 6);
    swatch_box = gtk_hbox_new (FALSE, 0);
  } else {
    main_box = gtk_hbox_new (FALSE, 6);
    swatch_box = gtk_vbox_new (FALSE, 0);
  }

  gtk_box_pack_start (GTK_BOX (swatch_box), alignment, TRUE, FALSE, 6);
  gtk_container_add (GTK_CONTAINER (frame), main_box);
  gtk_box_pack_start (GTK_BOX (main_box), palette, TRUE, TRUE, 6);
  gtk_box_pack_start (GTK_BOX (main_box), swatch_box, FALSE, FALSE, 6);
  gtk_container_add (GTK_CONTAINER (GTK_DIALOG (obj)->vbox), frame);
  gtk_widget_show_all (frame);

  return obj;
}

static void
owl_colour_selector_dialog_class_init (OwlColourSelectorDialogClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);
  GParamSpec *pspec;

  g_type_class_add_private (klass, sizeof (OwlColourSelectorDialogPrivate));

  object_class->get_property = owl_colour_selector_dialog_get_property;
  object_class->set_property = owl_colour_selector_dialog_set_property;
  object_class->dispose = owl_colour_selector_dialog_dispose;
  object_class->finalize = owl_colour_selector_dialog_finalize;
  object_class->constructor = owl_colour_selector_dialog_constructor;

  pspec = g_param_spec_enum ("orientation",
      "The orientation of the dialog",
      "Set orientation of the dialog",
      OWL_TYPE_COLOUR_SELECTOR_DIALOG_ORIENTATION,
      OWL_COLOUR_SELECTOR_DIALOG_AUTOMATIC,
      G_PARAM_CONSTRUCT_ONLY | G_PARAM_READWRITE);
  
  g_object_class_install_property (object_class,
      PROP_ORIENTATION,
      pspec);
}

static void
colour_selected_cb (GtkWidget *widget, OwlColourSwatch *swatch, gpointer user_data)
{
  OwlColourSelectorDialogPrivate *priv = COLOUR_SELECTOR_DIALOG_PRIVATE (user_data);

  owl_colour_swatch_set_colour (OWL_COLOUR_SWATCH(priv->swatch), owl_colour_swatch_get_colour (swatch));
}

guint32
owl_colour_selector_dialog_get_colour (OwlColourSelectorDialog *self)
{
  OwlColourSelectorDialogPrivate *priv = COLOUR_SELECTOR_DIALOG_PRIVATE (self);
  
  return owl_colour_swatch_get_colour (OWL_COLOUR_SWATCH(priv->swatch));
}

void
owl_colour_selector_dialog_set_colour (OwlColourSelectorDialog *self, guint32 colour)
{
  OwlColourSelectorDialogPrivate *priv = COLOUR_SELECTOR_DIALOG_PRIVATE (self);

  owl_colour_swatch_set_colour (OWL_COLOUR_SWATCH (priv->swatch), colour);
}

static void
owl_colour_selector_dialog_init (OwlColourSelectorDialog *self)
{
  GtkWidget *frame;
  GtkWidget *swatch;
  GtkWidget *alignment;
  GtkWidget *palette;

  OwlColourSelectorDialogPrivate *priv = COLOUR_SELECTOR_DIALOG_PRIVATE (self);

  palette = owl_colour_palette_new (6, 4);
  owl_colour_palette_set_colours (OWL_COLOUR_PALETTE (palette), (guint32 *)colours, 24);
  g_signal_connect (palette, "colour-selected", (GCallback)colour_selected_cb, self);

  frame = gtk_frame_new (NULL);
  swatch = owl_colour_swatch_new ();

  alignment = gtk_alignment_new (0.5, 0.5, 1, 1);
  gtk_container_add (GTK_CONTAINER (alignment), swatch);

  gtk_widget_set_size_request (swatch, 64, 64);

  gtk_dialog_add_buttons (GTK_DIALOG (self),
      GTK_STOCK_CANCEL,
      GTK_RESPONSE_CANCEL,
      GTK_STOCK_OK,
      GTK_RESPONSE_OK,
      NULL);

  gtk_window_set_resizable (GTK_WINDOW (self), FALSE);
  gtk_window_set_title (GTK_WINDOW (self), "Pick a colour");

  priv->swatch = swatch;
  priv->frame = frame;
  priv->alignment = alignment;
  priv->palette = palette;
}

GtkWidget*
owl_colour_selector_dialog_new_with_orientation (OwlColourSelectorDialogOrientation orientation)
{
  return GTK_WIDGET(g_object_new (OWL_TYPE_COLOUR_SELECTOR_DIALOG, "orientation", orientation, NULL));
}

GtkWidget*
owl_colour_selector_dialog_new (void)
{
  return GTK_WIDGET(g_object_new (OWL_TYPE_COLOUR_SELECTOR_DIALOG, NULL));
}
