/*
  seccom-loader: secure computing loader for linux
  Copyright (C) 2004-2006  Andrea Arcangeli <andrea@cpushare.com>
  
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation;
  only version 2.1 of the License.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef _SECCOMP_LOADER_H
#define _SECCOMP_LOADER_H

#if !defined(__i386__) && (!defined(__powerpc__) || defined(__powerpc64__))
#error "only x86 and ppc are supported, not enough resources to port to other archs yet"
#endif

#include <sys/types.h>

#define TEXT_ADDRESS 0x12000000
#define DATA_BSS_ADDRESS 0x20000000
#define STATIC_PAYLOAD_END 0x40000000
#define HEAP_START 0x60000000
#define HEAP_END 0xb8000000

#define MAX_TEXT_SIZE (DATA_BSS_ADDRESS - TEXT_ADDRESS) /* 224M */
#define MAX_DATA_BSS_SIZE (STATIC_PAYLOAD_END - DATA_BSS_ADDRESS) /* 512M */
#define MAX_HEAP_SIZE (HEAP_END-HEAP_START) /* 1408M */
#define MAX_STACK_SIZE (8*1024*1024) /* 8M */

#define PR_GET_SECCOMP	21
#define PR_SET_SECCOMP	22

#define MAGIC_SECCOMP_PROC 209
#define MAGIC_SECCOMP_PRCTL 210

#define MAGIC_ASK_SECCOMP 211
#define MAGIC_GOT_SECCOMP 212

struct obj_header {
	int32_t text_size;
	int32_t data_size;
	int32_t bss_size;
	u_int32_t call_address;
	u_int32_t signal_address;
	int32_t heap_size;
	int32_t stack_size;
	u_int32_t cksum;
} __attribute__((packed));

#include <errno.h>

/* ntoh/hton */
#include <netinet/in.h>

#ifdef __i386__
#include <unistd-i386.h>

extern long inline sys_exit(int error_code)
{
	long ret;
	asm volatile("int $0x80" :
		     "=a" (ret) :
		     "0" (__NR_exit), "b" (error_code));
	__syscall_return(long, ret);
}

extern ssize_t inline sys_read(unsigned int fd, char * buf, size_t count)
{
	ssize_t ret;
	asm volatile("int $0x80" :
		     "=a" (ret) :
		     "0" (__NR_read), "b" (fd),
		     "c" (buf), "d" (count));
	__syscall_return(ssize_t, ret);
}

extern ssize_t inline sys_write(unsigned int fd, char * buf, size_t count)
{
	ssize_t ret;
	asm volatile("int $0x80" :
		     "=a" (ret) :
		     "0" (__NR_write), "b" (fd),
		     "c" (buf), "d" (count));
	__syscall_return(ssize_t, ret);
}

#ifdef MALICIOUS
static inline long sys_getpid(void)
{
	long ret;
	asm volatile("int $0x80" :
		     "=a" (ret) :
		     "0" (__NR_getpid));
	__syscall_return(long, ret);
}
#endif /* MALICIOUS */
#endif

#ifdef __powerpc__
#include <unistd-ppc.h>

static inline long sys_exit(int error_code)
{
	__syscall_nr(1, long, exit, error_code);
}

static inline ssize_t sys_read(unsigned int fd, char * buf, size_t count)
{
	__syscall_nr(3, ssize_t, read, fd, buf, count);
}

static inline ssize_t sys_write(unsigned int fd, char * buf, size_t count)
{
	__syscall_nr(3, ssize_t, write, fd, buf, count);
}

#ifdef MALICIOUS
static inline long sys_getpid(void)
{
	__syscall_nr(0, long, getpid);
}
#endif /* MALICIOUS */
#endif /* __powerpc__ */

#endif /* _SECCOMP_LOADER_H */
