// 
// Copyright (c) 2006-2010, Benjamin Kaufmann
// 
// This file is part of Clasp. See http://www.cs.uni-potsdam.de/clasp/ 
// 
// Clasp is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// Clasp is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with Clasp; if not, write to the Free Software
// Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
//
#include "clasp_options.h"
#include "program_opts/typed_value.h"  
#include "program_opts/composite_value_parser.h"  // pair and vector
#include <clasp/satelite.h>
#include <sstream>
#include <fstream>
#include <algorithm>
using namespace ProgramOptions;
using namespace std;

#ifdef _DEBUG
#define STATE_DEFAULTED Value::value_unassigned 
#else
#define STATE_DEFAULTED Value::value_defaulted
#endif
namespace Clasp { 
/////////////////////////////////////////////////////////////////////////////////////////
// Parseing & Mapping of options
/////////////////////////////////////////////////////////////////////////////////////////
// a little helper to make parsing more comfortable
template <class T>
inline bool parse(const std::string& str, T& out) {
	return ProgramOptions::DefaultParser<T>::parse(str, out);
}

// maps positional options to options number or file
bool parsePositional(const std::string& t, std::string& out) {
	int num;
	if   (parse(t, num)) { out = "number"; }
	else                 { out = "file";   }
	return true;
}

// overload for parsing a clasp schedule - needed
// to make parsing of schedule composable, e.g. in a pair 
ProgramOptions::StringSlice parseValue(const ProgramOptions::StringSlice& in, ScheduleStrategy& sched, int) {
	std::pair<std::string, std::vector<double> > v;
	bool ok = false;
	ScheduleStrategy::Type t = ScheduleStrategy::geometric_schedule;
	if (in.size() > 0) {
		if      (parseValue(in, v.second, 0).ok()) { ok = true; }
		else if (parseValue(in, v, 0).ok())        {
			v.first = toLower(v.first);
			if      (v.first == "no")                      { ok = v.second.empty(); }
			else if (v.first == "+" || v.first == "arith") { ok = v.second.size() >= 2; t = ScheduleStrategy::arithmetic_schedule; }
			else if (v.first == "*" || v.first == "geom")  { ok = v.second.size() >= 2;  }
			else if (v.first == "l" || v.first == "luby")  { ok = v.second.size() >= 1; v.second.resize(1); }
			else if (v.first == "b" || v.first == "biere") { ok = v.second.size() >= 3; v.second.resize(3); }
		}
	}
	if (!ok)                  { return in.parsed(false); }
	if (v.second.size() == 1) { t = ScheduleStrategy::luby_schedule; }
	v.second.resize(3, 0.0);
	sched = ScheduleStrategy(t, uint32(v.second[0]), v.second[1], uint64(v.second[2]));
	return in.parsed(true, in.size());
}
// maps value to the corresponding enum-constant
// theMap must be null-terminated!
bool mapEnumImpl(const EnumMap* theMap, const std::string& value, int& out) {
	std::string temp = toLower(value);
	for (int i = 0; theMap[i].str; ++i) {
		if (value == theMap[i].str) { 
			out = theMap[i].ev; 
			return true; 
		}
	}
	return false;
}
// statically binds theMap so that the resulting function can be
// used as parser in the ProgramOption-library
template <const EnumMap* theMap, class EnumT>
bool mapEnum(const std::string& value, EnumT& out) {
	int temp;
	return mapEnumImpl(theMap, value, temp) && (out = (EnumT)temp, true);
}
/////////////////////////////////////////////////////////////////////////////////////////
// Clasp specific basic options
/////////////////////////////////////////////////////////////////////////////////////////
BasicOptions::BasicOptions() : timeout(-1), quiet(-1,-1), stats(0), outf(0), maxSat(false), ifs(' ') {}
void BasicOptions::initOptions(ProgramOptions::OptionContext& root) {
	OptionGroup basic("Basic Options");
	basic.addOptions()
		("verbose,V"  , storeTo(verbose)->implicit("3")->defaultsTo("1")->arg("<n>"), "Verbosity level")
		("stats"      , storeTo(stats)->implicit("1"), "Print statistics (2=extended stats)")
		("quiet,q"    , storeTo(quiet)->implicit("3,3")->arg("<m,o>"), 
			"Configure printing of models and optimize values\n"
			"      <m>: print models          (0=all, 1=best/last, 2=no)\n"
			"      <o>: print optimize values (0=all, 1=best, 2=no)")
		("opt-sat", flag(maxSat), "Compute MaxSAT (requires DIMACS input)")
		("outf", storeTo(outf), "Output format (0=default, 1=competition, 2=JSON)")
		("time-limit" , storeTo(timeout)->arg("<n>"), "Set time limit to %A seconds")
	;
	ProgramOptions::OptionGroup hidden("Hidden Options");
	hidden.hidden(true);
	hidden.addOptions()
		("file,f", storeTo(input)->composing(), "Input files")
		("ifs", storeTo(ifs, &BasicOptions::mapIfs), "Internal field separator\n")
	;
	root.add(basic);
	root.add(hidden);
}

bool BasicOptions::mapIfs(const std::string& s, char& ifs) {
	if (s.empty() || s.size() > 2) return false;
	ifs = s[0];
	if (ifs == '\\' && s.size() > 1) {
		ifs = s[1];
		switch (ifs) {
			case 't' : ifs = '\t'; break;
			case 'n' : ifs = '\n'; break;
			case '\\': ifs = '\\'; break;
			case 'v' : ifs = '\v'; break;
		}
	}
	return true;
}
/////////////////////////////////////////////////////////////////////////////////////////
// Clasp specific mode options
/////////////////////////////////////////////////////////////////////////////////////////
void GeneralOptions::initOptions(ProgramOptions::OptionContext& root) {
	GlobalOptions* global = config;
	OptionGroup general("Clasp - General Options");
	general.addOptions()
		("enum-mode", storeTo(global->enumerate.mode, &parseEnumMode)->defaultsTo("auto")->state(Value::value_defaulted), 
			"Configure enumeration algorithm\n"
			"      Default: %D\n"
			"      Valid:   auto, bt, record, brave, cautious\n"
			"        auto    : bt for (projected) enumeration, record for optimization\n"
			"        bt      : Backtrack decision literals from solutions\n"
			"        record  : Add nogoods for computed solutions\n"
			"        brave   : Compute brave consequences (union of all models)\n"
			"        cautious: Compute cautious consequences (intersection of all models)")
		("number,n", storeTo(global->enumerate.numModels)->arg("<n>"), 
			"Compute at most %A models (0 for all)\n"
			"      Default: 1 / 0 when optimizing/computing consequences")
		("restart-on-model"  , flag(global->enumerate.restartOnModel), "Restart after each model")
		("project"           , flag(global->enumerate.project)       , "Project models to named atoms in enumeration mode\n")
		("opt-ignore"        , flag(global->opt.no) , "Ignore optimize statements")
		("opt-hierarch"      , storeTo(global->opt.hierarch)->implicit("1"), "Hierarchical opt. and {1=fixed, 2=inc, 3=dec} step")
		("opt-all"           , storeNotify(myVec, this, &GeneralOptions::mapMyVec)->arg( "<opt>"), "Compute only models <= %A")
		("opt-value"         , storeNotify(myVec, this, &GeneralOptions::mapMyVec), 
			"Initialize objective function(s)\n"
			"      Valid:   <n1[,n2,n3,...]>\n")
		
		("sat-prepro", storeNotify(myVec, this, &GeneralOptions::mapMyVec)->implicit("-1"),
		  "Run SatELite-like preprocessing\n"
			"      Valid: <n1[,n2,n3,n4,n5]> / Implicit: %I\n"
			"        <n1>: Run for at most <n1> iterations           (-1=run to fixpoint)\n"
			"        <n2>: Run variable elimination with cutoff <n2> (-1=no cutoff)\n"
			"        <n3>: Run for at most <n3> seconds              (-1=no time limit)\n"
			"        <n4>: Disable if <n4>%% of vars are frozen       (-1=no limit)\n"
			"        <n5>: Run blocked clause elimination (0=no,1=limited,2=full)")
	;
	OptionGroup asp("Clasp - ASP Options");
	asp.addOptions()
		("pre" , flag(global->enumerate.onlyPre), "Run ASP preprocessor and exit")
		("supp-models",flag(global->eq.noSCC), "Compute supported models (no unfounded set check)\n")
		("eq", storeTo(global->eq.iters)->arg("<n>"), 
			"Configure equivalence preprocessing\n"
			"        %A : Run for at most %A iterations (-1=run to fixpoint/0=disable)")
		("backprop",flag(global->eq.backprop), "Enable backpropagation in ASP-preprocessing")
		("trans-ext", storeTo(global->eq.erMode, &mapEnum<extRules>),
			"Configure handling of Lparse-like extended rules\n"
			"      Valid:   all, choice, card, weight, integ, dynamic, no\n"
			"        all    : Transform all extended rules to basic rules\n"
			"        choice : Transform choice rules, but keep cardinality and weight rules\n"
			"        card   : Transform cardinality rules, but keep choice and weight rules\n"
			"        weight : Transform cardinality and weight rules, but keep choice rules\n"
			"        integ  : Transform cardinality integrity constraints\n"
			"        dynamic: Transform \"simple\" extended rules, but keep more complex ones\n"
			"        no     : Do not transform extended rules\n")
	;
	root.add(general);
	root.add(asp);
	ProgramOptions::OptionGroup hidden("Hidden Options");
	hidden.hidden(true);
	hidden.addOptions()
		("brave"   , storeTo(global->enumerate.mode, &parseEnumMode)->flag()->implicit("brave"), "Alias for --enum-mode=%I")
		("cautious"   , storeTo(global->enumerate.mode, &parseEnumMode)->flag()->implicit("cautious") , "Alias for --enum-mode=%I")
		("solution-recording" , storeTo(global->enumerate.mode, &parseEnumMode)->flag()->implicit("record"), "Alias for --enum-mode=%I")
		("project-opt", storeTo(global->enumerate.projectOpts), "Additional options for projection as octal digit")
		("eq-dfs"     , flag(global->eq.dfOrder)    , "Enable df-order in eq-preprocessing")
		("learn-explicit", flag(this, GeneralOptions::mapLearnExplicit), "Don't use Short Implication Graph for learning\n")
	;
	root.add(hidden);
}
const EnumMap GeneralOptions::extRules[] = {
	{"all", ProgramBuilder::mode_transform}, {"yes", ProgramBuilder::mode_transform},
	{"no", ProgramBuilder::mode_native}, {"0", ProgramBuilder::mode_native},
	{"choice", ProgramBuilder::mode_transform_choice}, {"card", ProgramBuilder::mode_transform_card},
	{"weight", ProgramBuilder::mode_transform_weight}, {"integ", ProgramBuilder::mode_transform_integ},
	{"dynamic", ProgramBuilder::mode_transform_dynamic}, {0, 0}
};
const EnumMap GeneralOptions::enumModes[]= {
	{"auto", GlobalOptions::enum_auto}, {"bt", GlobalOptions::enum_bt},
	{"record", GlobalOptions::enum_record}, {"brave", GlobalOptions::enum_brave},
	{"cautious", GlobalOptions::enum_cautious}, {0,0}
};
bool GeneralOptions::parseEnumMode(const std::string& s, GlobalOptions::EnumMode& m) {	
	int x;
	if (mapEnumImpl(enumModes, s, x)) {
		if (m != GlobalOptions::enum_auto && x != m) { throw ValueError("", ValueError::multiple_occurences, "enum-mode", s); }
		m = static_cast<GlobalOptions::EnumMode>(x);
		return true;
	}
	return false;
}
bool GeneralOptions::mapLearnExplicit(GeneralOptions* this_, const std::string&, const bool* v) {
	this_->config->ctx.enableUpdateShortImplications(!*v);
	return false;
}
bool GeneralOptions::mapMyVec(GeneralOptions* this_, const std::string& n, const std::vector<int>* v) {
	GlobalOptions* global = this_->config;
	assert(v == &this_->myVec);
	if (n == "sat-prepro") {
		if (!v->empty() && v->at(0) != 0) {
			SatElite::SatElite* pre = new SatElite::SatElite();
			pre->options.maxIters   = v->size()>0 ? v->at(0) : -1;
			pre->options.maxOcc     = v->size()>1 ? v->at(1) : -1;
			pre->options.maxTime    = v->size()>2 ? v->at(2) : -1;
			pre->options.maxFrozen  = v->size()>3 && v->at(3) > 0 ? (v->at(3)/100.0) : 1.0;
			if (v->size() > 4 && v->at(4) >= 0) {
				pre->options.bce      = (uint32)v->at(4);
			}
			global->ctx.satPrepro.reset(pre);
		}
	}
	else if (n == "opt-all" || !global->opt.all) {
		global->opt.vals.assign(v->begin(), v->end());
		global->opt.all = (n == "opt-all");
	}
	this_->myVec.clear();
	return true; // keep vector on return
}

bool GeneralOptions::validateOptions(const ProgramOptions::ParsedOptions& vm, Messages& m) {
	if (vm.count("opt-value") && vm.count("opt-all")) {
		m.warning.push_back("'opt-all' and 'opt-value' are mutually exclusive!");
	}
	if (config->eq.noSCC && vm.count("eq") == 0) {
		config->eq.noEq();
	}
	return true;
}
/////////////////////////////////////////////////////////////////////////////////////////
// Clasp specific search options
/////////////////////////////////////////////////////////////////////////////////////////
SearchOptions::SearchOptions(LocalOptions* o) : local(o) {}
SolverStrategies* SearchOptions::solverOpts() const { return &local->solver().strategies(); }
void SearchOptions::initOptions(ProgramOptions::OptionContext& root) {
	OptionGroup search("Clasp - Search Options");
	search.addOptions()
		("no-lookback"   , notify(this, &SearchOptions::mapSolverOpts)->flag(), "Disable all lookback strategies\n")
		("lookahead"     , storeTo(local->heuristic().lookahead, &mapEnum<lookTypes>)->implicit("atom"),
			"Configure failed-literal detection\n"
			"      Default: no (atom, if --nolookback)\n"
			"      Valid:   atom, body, hybrid, no / Implicit: %I\n"
			"        atom  : Apply failed-literal detection to atoms\n"
			"        body  : Apply failed-literal detection to bodies\n"
			"        hybrid: Apply Nomore++-like failed-literal detection\n"
			"        no    : Do not apply failed-literal detection")
		("initial-lookahead", storeTo(local->heuristic().lookaheadNum)->arg("<n>"), "Restrict failed-literal detection to %A decisions\n")

		("heuristic", storeTo(local->heuristic().name, &SearchOptions::parseHeuristic)->defaultsTo("Berkmin")->state(STATE_DEFAULTED), 
			"Configure decision heuristic\n"
			"      Default: %D (Unit, if --no-lookback)\n"
			"      Valid:   Berkmin, Vmtf, Vsids, Unit, None\n"
			"        Berkmin: Apply BerkMin-like heuristic\n"
			"        Vmtf   : Apply Siege-like heuristic\n"
			"        Vsids  : Apply Chaff-like heuristic\n"
			"        Unit   : Apply Smodels-like heuristic\n"
			"        None   : Select the first free variable")
		("opt-heuristic" , notify(this, &SearchOptions::mapSolverOpts)->implicit("1")->arg("<n>"), 
		 "Configure optimize heuristic\n"
		  "      Valid: 0 <= <n> <= 3 / Implicit: %I \n"
		  "         1: Apply sign heuristic\n"
		  "         2: Apply model heuristic\n"
		  "         3: Apply sign and model heuristic")
		("rand-freq", notify(this, &SearchOptions::mapRandOpts)->arg("<p>"), 
			"Make random decisions with probability %A\n"
			"      Valid:   [0.0...1.0]\n")

		("rand-prob", notify(this, &SearchOptions::mapRandOpts)->implicit("50,20")->arg("<n1,n2>"),
			"Configure random probing\n"
			"      Valid:   no (0), <n1,n2> (<n1> >= 0, <n2> > 0) / Implicit: %I\n"
			"        no     : Do not run random probing\n"
			"        <n1,n2>: Run <n1> random passes up to at most <n2> conflicts each\n")
		("rand-watches", notify(this, &SearchOptions::mapSolverOpts)->implicit("true")->defaultsTo("true"),
			"Configure watched literal initialization\n"
			"      Default: %D / Implicit: %I\n"
			"      Valid:   true (yes, 1), false (no, 0)\n"
			"        true : Randomly determine watched literals\n"
			"        false: Watch least watched literals in a nogood")
		("seed"    , storeNotify(aNumber, this, &SearchOptions::mapNumber)->arg("<n>"),    "Set random number generator's seed to %A\n")
		("search-limit", notify(this, &SearchOptions::mapPair)->arg("<n,m>"), "Stop search after <n> conflicts or <m> restarts\n")
	;
	
	OptionGroup lookback("Clasp - Lookback Options");
	lookback.addOptions()
		("restarts,r", notify(this, &SearchOptions::mapRestart)->defaultsTo("100,1.5")->state(Value::value_defaulted),
			"Configure restart policy\n"
			"      Default: %D\n"
			"      Valid:   [type,]<n1[,n2,n3]> (<n1> >= 0, <n2>,<n3> > 0), no\n"
			"      [L,]<n1>          : Run Luby et al.'s sequence with unit length <n1>\n"
			"      [*,]<n1>,<n2>     : Run geometric sequence of <n1>*(<n2>^i) conflicts\n"
			"      +,<n1>,<n2>       : Run arithmetic sequence of <n1>+(<n2>*i) conflicts\n"
			"      [+,]<n1>,<n2>,<n3>: Run Biere's inner-outer sequence (<n3>=outer)\n"
			"          <n1> = 0, no  : Disable restarts")
		("local-restarts"  , notify(this, &SearchOptions::mapRestart)->flag(), "Enable Ryvchin et al.'s local restarts")
		("bounded-restarts", flag(local->solve.restart.bounded), "Enable (bounded) restarts during model enumeration")
		("reset-restarts",   flag(local->solve.restart.resetOnModel), "Reset restart strategy during model enumeration")
		("save-progress"   , notify(this, &SearchOptions::mapSolverOpts)->implicit("1")->arg("<n>"), "Enable RSat-like progress saving on backjumps > %A")

		("shuffle,s", notify(this, &SearchOptions::mapPair)->arg("<n1,n2>"),
			"Configure shuffling after restarts\n"
			"      Valid:   <n1,n2> (<n1> >= 0, <n2> >= 0)\n"
			"        <n1> > 0: Shuffle problem after <n1> and re-shuffle every <n2> restarts\n"
			"        <n1> = 0: Do not shuffle problem after restarts\n"
			"        <n2> = 0: Do not re-shuffle problem\n")

		("deletion,d", notify(this, &SearchOptions::mapReduceOpts)->defaultsTo("3.0,1.1,3.0")->state(STATE_DEFAULTED), 
			"Configure size of learnt nogood database\n"
			"      Default: %D\n"
			"      Valid:   <n1[,n2,n3]> (<n3> >= <n1> >= 0, <n2> >= 1.0), no\n"
			"        <n1,n2,n3>: Store at most min(P/<n1>*(<n2>^i),P*<n3>) learnt nogoods,\n"
			"                    P and i being initial problem size and number of restarts\n"
			"        no        : Do not restrict learnt db size")
		("estimate", flag(local->solve.reduce.estimate), "Use estimated problem complexity to init learnt db")
		("dinit",notify(this, &SearchOptions::mapReduceOpts)->arg("<min,max>"), "Limit initial learnt db size to range [min,max]")
		("dsched", storeTo(local->solve.reduce.cflSched, &parseSchedule), "Configure secondary deletion policy (see restarts)")
		("dglue", notify(this, &SearchOptions::mapReduceOpts), "Don't delete nogoods with lbd <= x")
		("update-lbd", notify(this, &SearchOptions::mapSolverOpts)->implicit("1"), "Update lbds of learnt nogoods (1=yes,2=strict)")
		("reduce-on-restart", flag(local->solve.reduce.reduceOnRestart), "Delete some learnt nogoods after every restart\n")

		("strengthen", notify(this, &SearchOptions::mapSolverOpts)->defaultsTo("all")->state(STATE_DEFAULTED),
			"Configure conflict nogood strengthening\n"
			"      Default: %D\n"
			"      Valid:   bin, tern, all, no\n"
			"        bin : Check only binary antecedents for self-subsumption\n"
			"        tern: Check binary and ternary antecedents for self-subsumption\n"
			"        all : Check all antecedents for self-subsumption\n"
			"        no  : Do not check antecedents for self-subsumption")
		("recursive-str", notify(this, &SearchOptions::mapSolverOpts)->flag(), "Enable MiniSAT-like conflict nogood strengthening")
		("otfs",notify(this, &SearchOptions::mapSolverOpts)->implicit("1"), "Enable on-the-fly subsumption (1=partial, 2=full)")
		("reverse-arcs",notify(this, &SearchOptions::mapSolverOpts)->implicit("1"), "Enable ManySAT-like inverse-arc learning\n")
		
		("loops", storeTo(local->loopRep, &mapEnum<loopTypes>)->defaultsTo("common")->state(STATE_DEFAULTED),
			"Configure representation/learning of loop formulas\n"
			"      Default: %D\n"
			"      Valid:   common, distinct, shared, no\n"
			"        common  : Create loop nogoods for atoms in an unfounded set\n"
			"        distinct: Create distinct loop nogood for each atom in an unfounded set\n"
			"        shared  : Create loop formula for a whole unfounded set\n"
			"        no      : Do not learn loop formulas")
		("loops-in-heu", storeTo(local->heuristic().loops), "Consider loop nogoods in heuristics\n")

		("contraction", storeNotify(aNumber, this, &SearchOptions::mapNumber)->arg("<n>")->defaultsTo("250")->state(STATE_DEFAULTED),
			"Configure temporary contraction of learnt nogoods\n"
			"      Default: %D\n"
			"      Valid:\n"
			"        0  : Do not contract learnt nogoods\n"
			"        > 0: Contract learnt nogoods containing more than %A literals\n")
	;
	ProgramOptions::OptionGroup hidden("Hidden Options");
	hidden.hidden(true);
	hidden.addOptions()
		("berk-max", storeTo(local->heuristic().extra.berkMax), "Consider at most <n> nogoods in Berkmin heuristic")
		("berk-moms", storeTo(local->heuristic().berkMoms)->implicit("1"), "Enable/Disable MOMs in Berkmin")
		("berk-huang",flag(local->heuristic().berkHuang), "Enable Huang-scoring in Berkmin")
		("berk-once",flag(local->heuristic().berkOnce), "Score sets (instead of multisets) in Berkmin")
		("vmtf-mtf",storeTo(local->heuristic().extra.vmtfMtf), "In Vmtf move <n> conflict-literals to the front")
		("vsids-decay",storeTo(local->heuristic().extra.vsidsDecay), "In Vsids use 1.0/0.<n> as decay factor")
		("nant",flag(local->heuristic().nant), "In Unit count only atoms in NAnt(P)\n")
		("dfrac", storeNotify(aNumber, this, &SearchOptions::mapNumber), "Set fraction of nogoods to delete on reduction")
		("dalgo", notify(this, &SearchOptions::mapReduceOpts), "Use {0=basic, 1=sort-in-place, 2=sorted} deletion")
		("dscore", notify(this, &SearchOptions::mapReduceOpts), "Use {0=activity, 1=lbd, 2=combined} nogood scores")
		("dgrowS", notify(this, &SearchOptions::mapReduceOpts), "Set deletion grow schedule (see restarts)")
		("digng", notify(this, &SearchOptions::mapReduceOpts)->flag(), "Do not count glue clauses in learnt limit")
		("cir", storeNotify(aNumber, this, &SearchOptions::mapNumber), "Do CI Restart every %A restarts (0=disable)")
		("cir-bump", storeTo(local->solve.restart.cirBump), "Set CIR bump factor to %A")
		("dynamic-restarts", notify(this, &SearchOptions::mapRestart), "Enable restarts based on lbd/cfl progression\n")
	;
	root.add(search);
	root.add(lookback);
	root.add(hidden);
}
const EnumMap SearchOptions::loopTypes[] = {
	{"common", DefaultUnfoundedCheck::common_reason}, {"shared", DefaultUnfoundedCheck::shared_reason},
	{"distinct", DefaultUnfoundedCheck::distinct_reason}, {"no", DefaultUnfoundedCheck::only_reason},
	{0,0}
};
const EnumMap SearchOptions::lookTypes[] = {
	{"atom", Lookahead::atom_lookahead}, {"no", Lookahead::no_lookahead},
	{"body", Lookahead::body_lookahead}, {"hybrid", Lookahead::hybrid_lookahead},
	{0,0}
};
const EnumMap SearchOptions::anteTypes[] = {
	{"all", SolverStrategies::all_antes}, {"no", SolverStrategies::no_antes},
	{"bin", SolverStrategies::binary_antes}, {"tern", SolverStrategies::binary_ternary_antes},
	{0,0}
};

bool SearchOptions::parseHeuristic(const std::string& s, std::string& out) {
	std::string temp = toLower(s);
	if      (temp == "berkmin")   { out = temp; return true; }
	else if (temp == "vmtf")      { out = temp; return true; }
	else if (temp == "vsids")     { out = temp; return true; }
	else if (temp == "unit")      { out = temp; return true; }
	else if (temp == "none")      { out = temp; return true; }
	return false;
}

bool SearchOptions::mapRandOpts(SearchOptions* this_, const std::string& opt, const std::string& value) {
	std::pair<double, double> v(0, 0);
	if (value == "no" || parse(value, v)) {
		return (opt == "rand-freq" && this_->local->solve.setRandomProbability(v.first))
		  ||   (opt == "rand-prob" && this_->local->solve.setRandomizeParams((uint32)v.first, (uint32)v.second));
	}
	return false;
}

bool SearchOptions::parseSchedule(const std::string& s, ScheduleStrategy& sched) {
	return parseValue(StringSlice(s), sched, 0).ok();
}

bool SearchOptions::mapNumber(SearchOptions* this_, const std::string& opt, const double* v) {
	if      (opt == "seed")        { this_->solverOpts()->rng.srand(static_cast<uint32>(*v)); }
	else if (opt == "contraction") { this_->solverOpts()->compress = static_cast<uint32>(*v); }
	else if (opt == "dfrac")       { this_->local->solve.reduce.remFrac = (float)*v; }
	else if (opt == "cir")         { this_->local->solve.restart.cir = (uint8)*v; }
	this_->aNumber = 0.0;
	return true;
}

bool SearchOptions::mapRestart(SearchOptions* this_, const std::string& n, const std::string& value) {
	bool ok = false;
	if (n == "restarts") {
		uint8 t = this_->local->solve.restart.type;
		ok      = true;
		if (t != RestartParams::dynamic_restarts) {
			this_->local->solve.restart.type = (t & RestartParams::local_restarts);
			ok = parseSchedule(value, this_->local->solve.restart.sched);
		}
	}
	if (n == "dynamic-restarts") {
		std::vector<double> v; v.reserve(3);
		ok = parse(value, v) && !v.empty();
		if (ok && v.size() == 1) { v.push_back(0.7); }
		if (ok && v.size() == 2) { v.push_back(0.0); }
		ok = ok && v.size() == 3;
		if (ok) { 
			this_->local->solve.restart.initDynamic((uint32)v[0], v[1], v[2]);
		}
	}
	else if (n == "local-restarts") {
		bool b = false;
		ok     = FlagStr::store_true(value, b);
		uint8 t= this_->local->solve.restart.type;
		if (b && t != RestartParams::dynamic_restarts) t |= RestartParams::local_restarts;
		this_->local->solve.restart.type = t;
	}
	return ok;
}

bool SearchOptions::mapReduceOpts(SearchOptions* this_, const std::string& n, const std::string& value) {
	ReduceParams& p = this_->local->solve.reduce;
	bool ok = false, b; int num;
	if (n == "deletion") {
		std::vector<double> v;
		ok = value == "no" || parse(value, v);
		if (v.empty())     { p.baseFrac = 0; }
		if (v.size() >= 1) { p.baseFrac = std::max(0.0001f, (float)v[0]);    }
		if (v.size() >= 2) { p.dbGrow   = std::max(1.0f   , (float)v[1]);    }
		if (v.size() >= 3) { p.dbMaxGrow= std::max(p.baseFrac, (float)v[2]); }
	}
	else if (n == "dinit") {
		std::pair<int, int> x(-1, -1);
		ok = parse(value, x);
		p.baseMin = (uint32)x.first; 
		p.baseMax = std::max((uint32)x.second, p.baseMin); 
	}
	else if (n == "dgrowS"){ ok = parseSchedule(value, p.growSched); }
	else if (n == "dglue") { ok = parse(value, num) && (p.strategy.glue = (uint32)num) == (uint32)num; }
	else if (n == "dalgo") { ok = parse(value, num) && (p.strategy.algo = (uint32)num) == (uint32)num && num < 3;}
	else if (n == "dscore"){ ok = parse(value, num) && (p.strategy.score= (uint32)num) == (uint32)num && num < 3;}
	else if (n == "digng") { ok = FlagStr::store_true(value, b) && (p.strategy.noGlue= (uint32)b) == (uint32)b;  }
	return ok;
}

bool SearchOptions::mapSolverOpts(SearchOptions* this_, const std::string& n, const std::string& v) {
#define SET(M, N) ((opts->M = (uint32)N) == (uint32)N)
	SolverStrategies* opts = this_->solverOpts();
	int  i; bool b;
	if      (n == "strengthen")    { return mapEnum<anteTypes>(v, i) && SET(cflMinAntes, i); }
	else if (n == "rand-watches")  { return parse(v, b) && SET(randomWatches, b);}
	else if (n == "save-progress") { return parse(v, i) && SET(saveProgress, i); }
	else if (n == "opt-heuristic") { return parse(v, i) && SET(heuOpts, i);      }
	else if (n == "otfs")          { return parse(v, i) && SET(otfs, i);         }
	else if (n == "reverse-arcs")  { return parse(v, i) && SET(reverseArcs, i);  }
	else if (n == "update-lbd")    { return parse(v, i) && SET(updateLbd, i);    }
	else if (n == "recursive-str") { return FlagStr::store_true(v, b) && SET(strRecursive, b); }
	else if (n == "no-lookback")   { return FlagStr::store_true(v, b) && SET(search, b);       }
#undef SET
	return false;
}

bool SearchOptions::mapPair(SearchOptions* this_, const std::string& n, const std::string& value) {
	std::pair<int, int> p(-1, -1);
	bool ok = parse(value, p);
	if (ok) {
		if      (n == "shuffle")      { this_->local->solve.setShuffleParams(p.first, p.second); }
		else if (n == "search-limit") { this_->local->solve.limits = SolveLimits(static_cast<uint64>(p.first), static_cast<uint64>(p.second)); }
		else                          { return false; }
	}
	return ok;
}

bool SearchOptions::validateOptions(const ProgramOptions::ParsedOptions& vm, Messages& m) {
	if (solverOpts()->search == Clasp::SolverStrategies::no_learning) {	
		if (vm.count("heuristic") == 0) { local->heuristic().name = "unit"; }
		if (vm.count("lookahead") == 0) { local->heuristic().lookahead = Lookahead::atom_lookahead; }
		bool warn = local->solve.restart.type != 0 || local->solve.restart.bounded || local->solve.reduce.reduceOnRestart;
		if (warn || vm.count("restarts") || vm.count("deletion") || vm.count("rand-prob") || vm.count("shuffle")) {
			m.warning.push_back("lookback-options ignored because lookback strategy is not used!");     
		}
	}
	if (local->solve.restart.type == RestartParams::dynamic_restarts) {
		if      (vm.count("restarts"))       { m.error = "'restarts' and 'dynamic-restarts' are mutually exclusive!"; }
		else if (vm.count("local-restarts")) { m.error = "'local-restarts' and 'dynamic-restarts' are mutually exclusive!"; }
	}
	return m.error.empty();
}

/////////////////////////////////////////////////////////////////////////////////////////
// clasp option validation
/////////////////////////////////////////////////////////////////////////////////////////
void ClaspOptions::initOptions(ProgramOptions::OptionContext& root, ClaspConfig& config) {
	satPreDefault = true;
	genTemplate   = false;
	this->config  = &config;
	initThreadOptions(root);
	mode.reset(new GeneralOptions(&config));
	search.reset(new SearchOptions(config.master()));
	mode->initOptions(root);
	search->initOptions(root);
	basic.initOptions(root);
}
#ifndef DISABLE_MULTI_THREADING
void ClaspOptions::initThreadOptions(ProgramOptions::OptionContext& root) {
	OptionGroup threadOps("Clasp - Thread Options");
	ThreadOptions* thread = &config->thread;
	threadOps.addOptions() 
		("threads,t", notify(this, &ClaspOptions::mapThreadOpts)->defaultsTo("1")->state(STATE_DEFAULTED)->arg("<n>"), 
			"Set number of threads to use\n"
			"      Default: %D / Valid: 0 <= %A < 64")
		("portfolio,p", storeTo(portfolio)->arg("<file>"), 
		 "Use portfolio to configure threads\n"
		 "      Valid: %A or 'default'")
		("force-gp", flag(thread->forceGP), "Force guiding path scheme in portfolio search")
		("create-template,g" , flag(genTemplate), "Print default portfolio and exit\n")
		("global-restarts", notify(this, &ClaspOptions::mapThreadOpts)->implicit("5,100,1.5"),
		 "Configure global restart policy (0 if not given)\n"
		 "      Valid: <n>[,strat>] / Implicit: %I\n"
		 "        <n> : Maximal number of global restarts (0=disable)\n"
		 "     <strat>: Restart strategy (see --restarts)\n")
		("copy-problem", flag(thread->distribute.copyProblem), "Copy (instead of share) problem between threads")
		("distribute", notify(this, &ClaspOptions::mapThreadOpts)->defaultsTo("no")->state(STATE_DEFAULTED),
			"Configure nogood distribution\n"
			"      Default: %D\n"
			"      Valid:   <type>[,<lbd>]\n"
			"        <type> : Nogoods to distribute (no, all, short, conflict, loop)\n"
			"        <lbd>  : Distribute only if lbd <= <lbd>")
		("integrate", notify(this, &ClaspOptions::mapThreadOpts)->defaultsTo("gp,1024")->state(STATE_DEFAULTED),
			"Configure nogood integration\n"
			"      Default: %D\n"
			"      Valid:   <pick>[,<grace>]\n"
			"        <pick>  : Selector to apply (all, unsat, gp, heuristic)\n"
			"        <grace> : Keep at least last <grace> shared nogoods")
	;
	root.add(threadOps);
}
bool ClaspOptions::mapThreadOpts(ClaspOptions* this_, const std::string& key, const std::string& value) {
	ThreadOptions* thread             = &this_->config->thread;
	ThreadOptions::Distribution& opts = thread->distribute;
	if (key == "threads") {
		int i;
		if (parse(value, i) && i > 0 && i < 65) {
			this_->config->setThreads(static_cast<uint32>(i));
			return true;
		}
	}
	else if (key == "distribute") {
		std::pair<std::string, int> parsed("", 2);
		if (!parse(value, parsed) || parsed.second < 0) { return false; }
		if      (parsed.first == "all")      opts.types = (Constraint_t::learnt_conflict | Constraint_t::learnt_loop);
		else if (parsed.first == "no")       opts.types = 0;
		else if (parsed.first == "short")    opts.types = Constraint_t::max_value+1;
		else if (parsed.first == "conflict") opts.types = Constraint_t::learnt_conflict;
		else if (parsed.first == "loop")     opts.types = Constraint_t::learnt_loop;
		else                                 return false;
		opts.lbd = static_cast<uint8>(parsed.second);
		return true;
	}
	else if (key == "integrate") {
		std::pair<std::string, int> parsed("", 1024);
		if (!parse(value, parsed) || parsed.second < 0) { return false; }
		if      (parsed.first == "all")       opts.filter = ThreadOptions::Distribution::filter_no;
		else if (parsed.first == "gp")        opts.filter = ThreadOptions::Distribution::filter_gp;
		else if (parsed.first == "unsat")     opts.filter = ThreadOptions::Distribution::filter_sat;
		else if (parsed.first == "heuristic") opts.filter = ThreadOptions::Distribution::filter_heuristic;
		else                       return false;
		opts.grace = static_cast<uint32>(parsed.second);
		return true;
	}
	else if (key == "global-restarts") {
		std::pair<int, ScheduleStrategy> arg;
		if (parse(value, arg)) {
			thread->restarts.maxR  = uint32(arg.first);
			thread->restarts.sched = arg.second;
			return true;
		}
	}
	return false; // value is not valid
}
#else
void ClaspOptions::initThreadOptions(ProgramOptions::OptionContext&) {}
bool ClaspOptions::mapThreadOpts(ClaspOptions*, const std::string&, const std::string&) { return false; }
#endif

bool ClaspOptions::validateOptions(const ProgramOptions::ParsedOptions& vm, Messages& m) {
	bool ok = mode->validateOptions(vm, m) && search->validateOptions(vm, m);
	mode.reset(0);
	search.reset(0);
	if (ok) {
		satPreDefault    = vm.count("sat-prepro") == 0;
		const char* port = 0; std::string mem;
		if (!portfolio.empty()) {
			if (portfolio == "default") {
				port = portfolio_g;
			}
			else {
				std::ifstream test(portfolio.c_str());
				if (!test) {
					m.error = "Could not open portfolio file '";
					m.error += portfolio;
					m.error += "'";
					return false;
				}
				if (!parsePortfolio(test, config->numThreads(), mem)) {
					m.error = "Portfolio file has unrecognized format!";
					return false;
				}
				port = mem.data();
			}
		}
		ok = populateThreadConfigs(port, vm, m);
	}
	return ok && config->validate(m.error);
}

void ClaspOptions::applyDefaults(Input::Format f) {
	if (f != Input::SMODELS && satPreDefault) {
		SatElite::SatElite* pre = new SatElite::SatElite();
		pre->options.maxIters = 20;
		pre->options.maxOcc   = 25;
		pre->options.maxTime  = 120;
		pre->options.maxFrozen= 1.0;
		config->ctx.satPrepro.reset(pre);		
	}
}
const char* ClaspOptions::getInputDefaults(Input::Format f) const {
	if (f == Input::SMODELS) { return "--eq=5"; }
	else                     { return "--sat-prepro=20,25,120"; }
}

bool ClaspOptions::parsePortfolio(std::istream& in, uint32 max, std::string& mem) const {
	mem.reserve(128);
	for (std::string line; std::getline(in, line) && max; ) {
		if (line.empty()) continue;
		if (line[0] == '[') {
			mem += line;
			mem += '\0';
			--max;
		}
		else if (line[0] != '#') { return false; }
	}
	mem += '\0';
	return true;
}

bool ClaspOptions::populateThreadConfigs(const char* port, const ProgramOptions::ParsedOptions& vm, Messages& m) {
	for (uint32 i = 1; i < config->numThreads(); ++i) {
		// copy all from command-line
		// NOTE: this also copies any default values
		config->threadConfig(i)->initFrom(*config->threadConfig(0));
	}
	if (port != 0) { 
		bool addSeed = false;
		const char* p= port;
		for (uint32 i = 0; i != config->numThreads(); ++i) {
			if (!parseConfig(p, i, vm,m)){ return false; }
			if (addSeed)                 { config->threadConfig(i)->solver().strategies().rng.srand(i); }
			if (*p)                      { p += strlen(p) + 1; }
			if (!*p)                     { p  = port; addSeed = true; }
		}
	}
	else { config->thread.forceGP = true; }
	return true;
}

bool ClaspOptions::parseConfig(const char* in, uint32 id, const ProgramOptions::ParsedOptions& vm, Messages& m) {
	assert(in);
	std::string configOpts;
	const char* nameBeg = in;
	const char* nameEnd = in;
	if (*nameBeg == '[') {
		for (nameEnd = ++nameBeg; *nameEnd && *nameEnd != ']';) {
			++nameEnd;
		}
	}
	if (nameEnd[0] != ']' || nameEnd[1] != ':') {
		nameEnd    = nameBeg;
		configOpts = in;
	}
	else {
		configOpts = nameEnd+2;
	}
	try {
		ProgramOptions::OptionContext root;
		SearchOptions opts(config->threadConfig(id));
		opts.initOptions(root);
		ProgramOptions::ParsedOptions parsed(vm);
		// HACK: ignore all restarts once command-line has restarts
		bool hasR            = vm.count("dynamic-restarts") || vm.count("restarts") || vm.count("local-restarts");
		ScheduleStrategy old = opts.local->solve.restart.sched;
		uint8 oldType        = opts.local->solve.restart.type;
		// add all from current config
		parsed.assign(ProgramOptions::parseCommandString(configOpts, root));
		if (!opts.validateOptions(parsed, m)) {
			if (!hasR || m.error.find("restarts") == std::string::npos) {
				throw std::runtime_error(m.error);
			}
			m.error.clear();
		}
		if (hasR) {
			opts.local->solve.restart.sched = old;
			opts.local->solve.restart.type  = oldType;
		}
		return true;
	}
	catch(const std::exception& e) {
		std::string err("Config '");
		err.append(nameBeg, (nameEnd-nameBeg));
		err += "': ";
		err += e.what();
		m.error = err;
		return false;
	}
}

const char* portfolio_g = {
	     /* 0 */"[DEFAULT]: "
	 "\0"/* 1 */"[CRAFTED]: --berk-max=512 --berk-once --otfs=1 --recursive-str --dinit=800,10000 --dsched=20000,1.1 --reverse-arcs=2 --opt-heu=1"
	 "\0"/* 2 */"[INDUST]:  --berk-max=512 --berk-once --restarts=256 --reverse-arcs=2 --recursive-str --dinit=800,10000 --dsched=10000,1.1 --dscore=2 --update-lbd --dalgo=2 --opt-heu=2"
	 "\0"/* 3 */"[LUBY-SP]: --heu=VSIDS --restarts=128 --save-p --otfs=1 --rand-w=false --contr=0 --opt-heu=3"
	 "\0"/* 4 */"[STRONG]:  --heu=VSIDS --otfs=2 --reverse-arcs=2 --recursive-str --rand-w=false"
	 "\0"/* 5 */"[LOCAL-R]: --berk-max=512 --restarts=100,1.5,1000 --local-restarts --rand-w=false --contr=0"
	 "\0"/* 6 */"[SLOW]:    --berk-max=512 --restarts=16000 --initial-look=50"
	 "\0"/* 7 */"[RANDY]:   --heu=VSIDS --rand-f=0.02 --rand-p=20,100 --recursive-str --shuffle=10,2"
	 "\0"/* 8 */"[SIMPLE]:  --heu=VMTF --str=no --contr=0 --restarts=100,1.3 --dinit=800,10000"
	 ""
};
#undef STATE_DEFAULTED

}
