/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2012 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#define _POSIX_SOURCE /* struct sigaction */
#define _POSIX_C_SOURCE 200809L /* getline() */
#define _GNU_SOURCE /* getline() */

#include <stdio.h> /* getline() */
#include <signal.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h> /* access() */
#include <errno.h>

/* wait() */
#include <sys/types.h>
#include <sys/wait.h>

#include "cdw_sys.h"
#include "gettext.h"
#include "cdw_debug.h"
#include "cdw_string.h"
#include "cdw_cdio.h"
#include "cdw_ofs.h"

/**
   \file cdw/src/utilities/cdw_sys.c

   File with various functions interfacing operating system.
   Currently this file defines functions:
   \li accessing /proc file system,
   \li accessing /lib/modules/ directory,
   \li handling signals.
*/

#ifdef CDW_MAIN_WINDOW_CAN_RESIZE
extern sig_atomic_t cdw_main_window_resize;
#endif


/* data structures and functions related to handling signals */
static void cdw_sys_exit_signal_handler(int signal_number);
static void cdw_sys_exit_signal_message(void);
static void cdw_sys_sigwinch_handler(int signal_number);
static void cdw_sys_sigchild_handler(__attribute__((unused)) int signal_number);

static sig_atomic_t child_exit_status;
static sig_atomic_t cdw_sys_signal_number;

static cdw_rv_t cdw_sys_check_file_system_support_via_proc_fs(cdio_fs_t cdio_fs);
static cdw_rv_t cdw_sys_check_file_system_support_via_modules(cdio_fs_t cdio_fs);
static char *   cdw_sys_get_osrelease(void);
static char *   cdw_sys_get_kernel_modules_location(void);

/* Generic type of signal handler, regardless of actions performed
   by specific implementation of a signal handler. */
typedef void (*cdw_sa_handler)(int);

#define CDW_SYS_SIGNAL_LABEL_LEN 10

/* Signal information taken from signal(7) man page. */
struct {
	int id;
	const char label[CDW_SYS_SIGNAL_LABEL_LEN];
	cdw_sa_handler handler;
} cdw_sys_signals[] = {
	{ SIGHUP,  "SIGHUP",  NULL },                          /*      1       Term    Hangup detected on controlling terminal or death of controlling process */
	{ SIGINT,  "SIGINT",  NULL },                          /*      2       Term    Interrupt from keyboard */
        { SIGQUIT, "SIGQUIT", cdw_sys_exit_signal_handler },   /*      3       Core    Quit from keyboard */
	{ SIGILL,  "SIGILL",  cdw_sys_exit_signal_handler },   /*      4       Core    Illegal Instruction */
	{ SIGABRT, "SIGABRT", cdw_sys_exit_signal_handler },   /*      6       Core    Abort signal from abort(3) */
        { SIGFPE,  "SIGFPE",  cdw_sys_exit_signal_handler },   /*      8       Core    Floating point exception */
        { SIGKILL, "SIGKILL", NULL },                          /*      9       Term    Kill signal, can't be caught */
        { SIGSEGV, "SIGSEGV", cdw_sys_exit_signal_handler },   /*     11       Core    Invalid memory reference */
	{ SIGPIPE, "SIGPIPE", cdw_sys_exit_signal_handler },   /*     13       Term    Broken pipe: write to pipe with no readers */
        { SIGALRM, "SIGALRM", NULL },                          /*     14       Term    Timer signal from alarm(2) */
	{ SIGTERM, "SIGTERM", cdw_sys_exit_signal_handler },   /*     15       Term    Termination signal */
	{ SIGUSR1, "SIGUSR1", NULL },                          /*  30,10,16    Term    User-defined signal 1 */
	{ SIGUSR2, "SIGUSR2", NULL },                          /*  31,12,17    Term    User-defined signal 2 */
	{ SIGCHLD, "SIGCHLD", cdw_sys_sigchild_handler    },   /*  20,17,18    Ign     Child stopped or terminated; to be handled by cdw separately */
	{ SIGCONT, "SIGCONT", NULL },                          /*  19,18,25    Cont    Continue if stopped */
	{ SIGSTOP, "SIGSTOP", NULL },                          /*  17,19,23    Stop    Stop process, can't be caught */
	{ SIGTSTP, "SIGTSTP", cdw_sys_exit_signal_handler },   /*  18,20,24    Stop    Stop typed at tty */
	{ SIGTTIN, "SIGTTIN", NULL },                          /*  21,21,26    Stop    tty input for background process */
	{ SIGTTOU, "SIGTTOU", NULL },                          /*  22,22,27    Stop    tty output for background process */


#if _POSIX_VERSION == 200112L
	{ SIGBUS,  "SIGBUS",  cdw_sys_exit_signal_handler },   /*  10,7,10     Core    Bus error (bad memory access) */
	{ SIGPOLL, "SIGPOLL", NULL },                          /*              Term    Pollable event (Sys V).; Synonym for SIGIO */
	{ SIGPROF, "SIGPROF", NULL },                          /*  27,27,29    Term    Profiling timer expired */
	{ SIGSYS,  "SIGSYS",  cdw_sys_exit_signal_handler },   /*  12,31,12    Core    Bad argument to routine (SVr4) */
	{ SIGTRAP, "SIGTRAP", NULL },                          /*      5       Core    Trace/breakpoint trap */
	{ SIGURG,  "SIGURG",  NULL },                          /*  16,23,21    Ign     Urgent condition on socket (4.2BSD) */
	{ SIGVTALRM, "SIGVTALRM", NULL },                      /*  26,26,28    Term    Virtual alarm clock (4.2BSD) */
	{ SIGXCPU, "SIGXCPU", NULL },                          /*  24,24,30    Core    CPU time limit exceeded (4.2BSD) */
	{ SIGXFSZ, "SIGXFSZ", NULL },                          /*  25,25,31    Core    File size limit exceeded (4.2BSD) */
#endif


#ifdef SIGIOT
	{ SIGIOT,  "SIGIOT",  cdw_sys_exit_signal_handler },   /*     6        Core    IOT trap. A synonym for SIGABRT */
#endif
#ifdef SIGEMT
	{ SIGEMT,  "SIGEMT",  NULL },                          /*   7,-,7      Term */
#endif
#ifdef SIGSTKFLT
	{ SIGSTKFLT, "SIGSTKFLT", NULL },                      /*   -,16,-     Term    Stack fault on coprocessor (unused) */
#endif
#ifdef SIGIO
	{ SIGIO,   "SIGIO",   NULL },                          /*  23,29,22    Term    I/O now possible (4.2BSD) */
#endif
#ifdef SIGCLD
	{ SIGCLD,  "SIGCLD",  cdw_sys_sigchild_handler },      /*   -,-,18     Ign     A synonym for SIGCHLD */
#endif
#ifdef SIGPWR
	{ SIGPWR,  "SIGPWR",  NULL },                          /*  29,30,19    Term    Power failure (System V) */
#endif
#ifdef SIGINFO
	{ SIGINFO, "SIGINFO", NULL },                          /*     29,-,-           A synonym for SIGPWR */
#endif
#ifdef SIGLOST
	{ SIGLOST, "SIGLOST", NULL },                          /*   -,-,-      Term    File lock lost */
#endif
#ifdef SIGWINCH
	{ SIGWINCH, "SIGWINCH", cdw_sys_sigwinch_handler },    /*  28,28,20    Ign     Window resize signal (4.3BSD, Sun) */
#endif
#ifdef SIGUNUSED
	{ SIGUNUSED, "SIGUNUSED", cdw_sys_exit_signal_handler },  /*   -,31,-     Core    Synonymous with SIGSYS */
#endif

	{ -1,      "",        NULL }}; /* guard */




/**
  \brief Remove zombie process created by run_command()

  \param signal_number - number of handled signal, unused
*/
void cdw_sys_sigchild_handler(__attribute__((unused)) int signal_number)
{
	/* clean child process */
	int status;
	wait(&status);
	child_exit_status = status;

	return;
}





void cdw_sys_exit_signal_handler(int signal_number)
{
	cdw_sys_signal_number = signal_number;

	/* there was an 'exit' signal, so let's better leave now;
	   exit() triggers functions registered with atexit(); these
	   are <module_name>_clean() functions */
	exit(EXIT_FAILURE);
}





void cdw_sys_sigwinch_handler(int signal_number)
{
	cdw_assert (signal_number == SIGWINCH, "ERROR: wrong signal to handle\n");

	/* I've tried different things:
	    - struct winsize size;
	    - endwin();
	    - resize_term(), resizeterm();
	    - clearok();
	   and the conclusion is: there is no simple solution
	   to the problem of resizing windows in ncurses app;


	   2012-02-20
	   The function is called in SIGWINCH signal. I've tested
	   this by adding only call to exit() in signal handler.
	   So perhaps my attempts didn't work because I was trying
	   to put call to regular library function inside a signal
	   handler. */

#ifdef CDW_MAIN_WINDOW_CAN_RESIZE
	/* code in main event loop should check this flag and act
	   accordingly (resize main window and all widgets embedded
	   in the window) */
	cdw_main_window_resize = 1;
#endif

	return;
}





void cdw_sys_exit_signal_message(void)
{
	int i = 0;
	while (cdw_sys_signals[i].id != -1) {
		if (cdw_sys_signals[i].id == cdw_sys_signal_number) {
			char message[] = "cdw: ";
			write(2, message, sizeof (message));
			write(2, cdw_sys_signals[i].label, sizeof (cdw_sys_signals[i].label));

			/* yes, I know that I shouldn't call printf() in
			   signal handlers, but since it will be called
			   shortly before calling exit(), there shouldn't
			   be much harm done */

			/* 2TRANS: this is a message printed to console when cdw needs to
			   close unexpectedly. "signal" is a signal from operating system;
			   "%s" is a signal's name; keep leading and ending "\n";
			   use "\n" in the middle of message as needed to keep line
			   lengths no more than 76 characters */
			fprintf(stderr, _("\n\ncdw received %s signal and it needed to clean up after itself\nand leave. This may mean that there is a bug in cdw. Don't be mad...\n\n"),
				cdw_sys_signals[i].label);
			break;
		}
		i++;
	}

	return;
}





int cdw_sys_signal_handlers_init(void)
{
	int i = 0;
	while (cdw_sys_signals[i].id != -1) {
		if (cdw_sys_signals[i].handler) {
			cdw_vdm ("INFO: setting %s handler\n", cdw_sys_signals[i].label);
			struct sigaction action;
			memset(&action, 0, sizeof(action));
			action.sa_handler = cdw_sys_signals[i].handler;
			action.sa_flags = 0;
			int rv = sigaction(cdw_sys_signals[i].id, &action, (struct sigaction *) NULL);
			if (rv == -1) {
				cdw_vdm ("ERROR: failed call to sigaction(%s, ...)\n", cdw_sys_signals[i].label);
				return -1;
			}
		}
		i++;
	}

	/* signal_message isn't a module cleaner function, so I
	   don't call atexit() in main(); this call is made to
	   ensure that final error message is printed to stderr
	   when cdw_sys_exit_signal_handler() calls exit() */

	atexit(cdw_sys_exit_signal_message);

	return 0;
}





cdw_rv_t cdw_sys_check_file_system_support(cdio_fs_t cdio_fs)
{
	cdw_assert (cdw_ofs_is_iso(cdio_fs) || cdio_fs == CDIO_FS_UDF,
		    "ERROR: unrecognized fs: %d\n", cdio_fs);
	if (cdw_ofs_is_iso(cdio_fs)) {
		/* there may be few subtypes of ISO9660 fs,
		   let's simplify this for code using cdio_fs;
		   CDIO_FS_ISO_UDF also counts as ISO9660 */
		cdio_fs = CDIO_FS_ISO_9660;
	}
#if 0
	/* force loading module for iso9660, so that it
	   becomes visible in /proc/filesystems */
	/* this works, but only if you are root */
	int rv = mount("/dev/scd0", "/media/cdrom0",
		       "iso9660", MS_MGC_VAL | MS_RDONLY | MS_NOSUID | MS_NOATIME, "");

#endif

	/* first let's try the polite way, searching /proc/filesystems;
	   if support for some file system is provided via kernel
	   module, and the module hasn't been loaded yet, it won't
	   be present in /proc/filesystems;
	   if it is compiled into kernel then it should be there */
	cdw_rv_t crv1 = cdw_sys_check_file_system_support_via_proc_fs(cdio_fs);
	if (crv1 == CDW_OK) {
		/* operating system supports given file system type */
		return CDW_OK;
	} else {
		/* either there is no information about given file system
		   in /proc/filesystems file (crv == CDW_NO), there is no
		   such file (crv == CDW_CANCEL), or ther was some error when
		   trying to access /proc/filesystems (crv == CDW_GEN_ERROR);
		   let's try different method */
		cdw_rv_t crv2 = cdw_sys_check_file_system_support_via_modules(cdio_fs);
		if (crv2 == CDW_OK) {
			/* operating system supports given file system
			   type (support is provided by kernel module) */
			return CDW_OK;
		} else if (crv2 == CDW_NO) {
			/* operating system does not support given file
			   system type */
			return CDW_NO;
		} else if (crv2 == CDW_CANCEL) {
			/* cdw can't tell if there is a support for given
			   file system type or not */
			return CDW_CANCEL;
		} else { /* CDW_GEN_ERROR */
			if (crv1 == CDW_CANCEL) {
				/* we got here because first method didn't
				   give definitive answer, so there is still
				   a chance that operating system supports
				   given file system type; "cancel" means
				   here "we don't know, but you may try" */
				return CDW_CANCEL;
			} else { /* crv1 == CDW_ERROR */
				/* two methods returned CDW_ERROR,
				   there is something very wrong;
				   assume that given file system is not
				   supported */
				return CDW_ERROR;
			}
		}
	}
}





cdw_rv_t cdw_sys_check_file_system_support_via_proc_fs(cdio_fs_t cdio_fs)
{
	FILE *fp = fopen("/proc/filesystems", "r");
	int e = errno;
	if (fp == (FILE *) NULL) {
		cdw_vdm ("ERROR: failed to open /proc/filesystems, error = \"%s\"\n", strerror(e));
		if (e == ENOENT) {
			/* incorrect assumption that all GNU/Linux
			   systems have similar proc file systems,
			   with /proc/filesystems file present */
			return CDW_CANCEL;
		} else {
			/* some other error */
			return CDW_ERROR;
		}
	}

	char *line = (char *) NULL;
	size_t len = 0;
	ssize_t r = 0;
	cdw_rv_t retval = CDW_NO; /* "no" = "support for file system not found (yet)" */
	while ((r = getline(&line, &len, fp)) != -1) {
		if (line[r - 1] == '\n') {
			line[r - 1] = '\0';
		}
		cdw_sdm ("INFO: line = \"%s\"\n", line);
		/* only basic file system types: iso9660 and udf */
		if (cdio_fs == CDIO_FS_ISO_9660 && strstr(line, "iso9660")) {
			cdw_vdm ("INFO: detected support for iso9660 in /proc/filesystems\n");
			retval = CDW_OK;
			break;

		} else if (cdio_fs == CDIO_FS_UDF && strstr(line, "udf")) {
			cdw_vdm ("INFO: detected support for udf in /proc/filesystems\n");
			retval = CDW_OK;
			break;
		} else {
			;
		}
#ifndef NDEBUG
		if (strstr(line, "ext3")) {
			cdw_vdm ("INFO: detected support for ext3 in /proc/filesystems\n");
		}
#endif
	}

	free(line);
	line = (char *) NULL;
	fclose(fp);
	fp = (FILE *) NULL;

	if (retval == CDW_OK) {
		/* support for given file system was found */
		return CDW_OK;
	} else if (retval == CDW_NO) {
		/* procedure of checking went without problems,
		   but no support for given file system was found */
		return CDW_NO;
	} else {
		/* some error? */
		return CDW_ERROR;
	}
}





cdw_rv_t cdw_sys_check_file_system_support_via_modules(cdio_fs_t cdio_fs)
{
	/* support for some file systems may be provided by kernel modules,
	   which (most of the time?) aren't loaded until they are needed;
	   they should be available in a specific place, let's hope that
	   it's the same place in all GNU/Linux distributions */

	/* let's check if our assumption about all modules'
	   location is correct */

	char *modules_location = cdw_sys_get_kernel_modules_location();
	if (modules_location == (char *) NULL) {
		cdw_vdm ("ERROR: can't get modules location path\n");
		return CDW_ERROR;
	}

	/* at this point we know that "/lib/modules/<os release>/kernel"
	   location is valid, let's check if there are modules implementing
	   support for specific file systems */
	char *path = (char *) NULL;
	if (cdio_fs == CDIO_FS_ISO_9660) {
		path = cdw_string_concat(modules_location, "/fs/isofs/isofs.ko", (char *) NULL);
	} else if (cdio_fs == CDIO_FS_UDF) {
		path = cdw_string_concat(modules_location, "/fs/udf/udf.ko", (char *) NULL);
	} else {
		;
	}
	free(modules_location);
	modules_location = (char *) NULL;

	if (path == (char *) NULL) {
		cdw_vdm ("ERROR: failed to concat path\n");
		return CDW_ERROR;
	}
	cdw_vdm ("INFO: inspecting path \"%s\"\n", path);
	int rv = access(path, F_OK);
	int e = errno;
	free(path);
	path = (char *) NULL;
	if (rv == 0) {
		cdw_vdm ("INFO: found \"%s\" module\n", cdio_fs == CDIO_FS_ISO_9660 ? "isofs" : "udf");
		return CDW_OK;
	} else {
		if (e == ENOENT) {
			/* no kernel module file, no support
			   for given file system */
			cdw_vdm ("INFO: can't find \"%s\" module, error = \"%s\"\n",
				 cdio_fs == CDIO_FS_ISO_9660 ? "isofs" : "udf",
				 strerror(e));
			return CDW_NO;
		} else {
			cdw_vdm ("ERROR: can't access \"%s\" module, error = \"%s\"\n",
				 cdio_fs == CDIO_FS_ISO_9660 ? "isofs" : "udf",
				 strerror(e));
			return CDW_ERROR;
		}
	}
}





char *cdw_sys_get_osrelease(void)
{
	/* osrelease file stores string that is the same as
	   result of "uname -r", e.g. "2.6.32F" */
	FILE *fp = fopen("/proc/sys/kernel/osrelease", "r");
	int e = errno;
	if (fp == (FILE *) NULL) {
		cdw_vdm ("ERROR: failed to open /proc/sys/kernel/osrelease, error = \"%s\"\n", strerror(e));
		if (e == ENOENT) {
			/* incorrect assumption that all GNU/Linux
			   systems have similar proc file systems,
			   with /proc/osrelease file present */
			return (char *) NULL;
		} else {
			/* some other error */
			return (char *) NULL;
		}
	}

	size_t len = 0;
	char *line = (char *) NULL;
	ssize_t r = getline(&line, &len, fp);
	e = errno;
	fclose(fp);
	if (r <= 0) {
		cdw_vdm ("ERROR: failed to get line, r = %zd, error = \"%s\"\n", r, strerror(e));
		return (char *) NULL;
	} else {
		if (line[r - 1] == '\n') {
			line[r - 1] = '\0';
		}
		cdw_vdm ("INFO: osrelease = \"%s\"\n", line);
		return line;
	}
}





char *cdw_sys_get_kernel_modules_location(void)
{
	char *osrelease = cdw_sys_get_osrelease();
	if (osrelease == (char *) NULL) {
		cdw_vdm ("ERROR: can't get osrelease\n");
		return (char *) NULL;
	}
	char *path = cdw_string_concat("/lib/modules/", osrelease, "/kernel/", (char *) NULL);
	free(osrelease);
	osrelease = (char *) NULL;

	if (path == (char *) NULL) {
		cdw_vdm ("ERROR: failed to concat modules release path\n");
		return (char *) NULL;
	} else {
		int rv = access(path, F_OK);
		int e = errno;
		if (rv != 0) {
			cdw_vdm ("ERROR: can't access path \"%s\", error = \"%s\"\n", path, strerror(e));
			free(path);
			path = (char *) NULL;
			/* even if the path exists, if we can't access it
			   using access(), it is useless */
			return (char *) NULL;
		} else {
			return path;
		}
	}
}





/**
   \brief Get copy of environment PATH variable

   \date Function's top-level comment reviewed on 2012-02-16
   \date Function's body reviewed on 2012-02-16

   Function returns a copy of environment PATH variable.
   Function may return NULL pointer if PATH environment variable is not
   defined on user's system.

   Returned pointer is owned by caller.

   \return valid pointer on success
   \return NULL pointer on failure
*/
char *cdw_sys_env_path(void)
{
	const char *tmp = getenv("PATH");
	if (!tmp) {
		cdw_vdm ("WARNING: getenv(\"PATH\") returns NULL\n");
		return (char *) NULL;
	}

#if 0   /* test data */
	const char *test_string = ":/home/acerion/bin: :/usr/local/bin::/usr/bin:/bin:/usr/games:";
	char *path = strdup(test_string);
#else
	char *path = strdup(tmp);
#endif
	if (!path) {
		cdw_vdm ("ERROR: failed to strdup() string \"%s\"\n", tmp);
		return (char *) NULL;
	} else {
		return path;
	}
}
