package Biber::Constants;
use 5.014000;
use strict;
use warnings;

use Encode::Alias;
use Readonly;

use base 'Exporter';

our @EXPORT = qw{
  $CONFIG_DEFAULT_BIBER
  %CONFIG_DEFAULT_BIBLATEX
  %CONFIG_SCOPE_BIBLATEX
  %CONFIG_BIBLATEX_PER_ENTRY_OPTIONS
  %NOSORT_TYPES
  %STRUCTURE_DATATYPES
  $BIBER_CONF_NAME
  $BIBLATEX_VERSION
  $BIBER_SORT_FINAL
  $BIBER_SORT_NULL
  $LABEL_FINAL
  };

# Version of biblatex which this release works with. Matched against version
# passed in control file
Readonly::Scalar our $BIBLATEX_VERSION => '1.7';

# Global flags needed for sorting
our $BIBER_SORT_FINAL = 0;
our $BIBER_SORT_NULL  = 0;

# the name of the Biber configuration file, which should be
# either returned by kpsewhich or located at "$HOME/.$BIBER_CONF_NAME"
our $BIBER_CONF_NAME = 'biber.conf';

## Biber CONFIGURATION DEFAULTS

# Locale - first try environment ...
my $locale;
if ($ENV{LC_COLLATE}) {
  $locale = $ENV{LC_COLLATE};
}
elsif ($ENV{LANG}) {
  $locale = $ENV{LANG};
}
elsif ($ENV{LC_ALL}) {
  $locale = $ENV{LC_ALL};
}

# ... if nothing, set a default
unless ($locale) {
  if ( $^O =~ /Win/) {
    $locale = 'English_United States.1252';
  }
  else {
    $locale = 'en_US.UTF-8';
  }
}

# nosort type category shortcuts
our %NOSORT_TYPES = (
                     type_name => {
                                   author => 1,
                                   afterword => 1,
                                   annotator => 1,
                                   bookauthor => 1,
                                   commentator => 1,
                                   editor => 1,
                                   editora => 1,
                                   editorb => 1,
                                   editorc => 1,
                                   foreword => 1,
                                   holder => 1,
                                   introduction => 1,
                                   namea => 1,
                                   nameb => 1,
                                   namec => 1,
                                   shortauthor => 1,
                                   shorteditor => 1,
                                   translator => 1
                                  },
                     type_title => {
                                    booktitle => 1,
                                    eventtitle => 1,
                                    issuetitle => 1,
                                    journaltitle => 1,
                                    maintitle => 1,
                                    origtitle => 1,
                                    title => 1
                                   }
);

# datatypes for structure validation
our %STRUCTURE_DATATYPES = (
                            integer => qr/\A\d+\z/xms
);

# Biber option defaults. Not really needed since they are passed by .bcf but
# useful to know the format. Also needed for options not yet passed in .bcf by biblatex

our $CONFIG_DEFAULT_BIBER = {
  bblencoding        => { content => 'UTF-8' },
  bibencoding        => { content => 'UTF-8' },
  bblsafechars       => { content => 0 },
  bblsafecharsset    => { content => 'base' },
  bltxml             => { content => 0 },
  collate            => { content => 1 },
  collate_options    => { option => [ { name => 'level', value => 4 } ] },
  graph              => { content => 0 },
  debug              => { content => 0 },
  decodecharsset     => { content => 'base' },
  fastsort           => { content => 0 },
  mincrossrefs       => { content => 2 },
  nodieonerror       => { content => 0 },
  nolog              => { content => 0 },
  nostdmacros        => { content => 0 },
  nosort             => { option => [ { name => 'type_name', value => q/\A\p{L}{2}\p{Pd}/ },
                                      { name => 'type_name', value => q/[\x{2bf}\x{2018}]/ } ] },
  onlylog            => { content => 0 },
  quiet              => { content => 0 },
  sortcase           => { content => 1 },
  sortfirstinits     => { content => 0 },
  sortlocale         => { content => $locale },
  sortupper          => { content => 1 },
  trace              => { content => 0 },
  validate_config    => { content => 0 },
  validate_control   => { content => 0 },
  validate_structure => { content => 0 },
  wraplines          => { content => 0 },
};

# default global options for biblatex
# in practice these will be obtained from the control file,
# but we need this for things not yet implemented by biblatex but expected by biber
our %CONFIG_DEFAULT_BIBLATEX =
  (
   labelalphatemplate => {
  labelelement => [
             {
               labelpart => [
                 { content => "shorthand", final => 1 },
                 { content => "label" },
                 {
                   content         => "labelname",
                   ifnamecount     => 1,
                   substring_side  => "left",
                   substring_width => 3,
                 },
                 { content => "labelname", substring_side => "left", substring_width => 1, substring_compound => 1 },
               ],
               order => 1,
             },
             {
               labelpart => [
                 { content => "year", substring_side => "right", substring_width => 2 },
               ],
               order => 2,
             },
           ],
  type  => "global",
},
   structure       =>
 { constraints => [
                   {
                     constraint => [
                                     {
                                       fieldxor => [
                                         {
                                           field => [
                                             { coerce => "true", content => "date" },
                                             { content => "year" },
                                           ],
                                         },
                                       ],
                                       type => "mandatory",
                                     },
                                   ],
                     entrytype  => [
                                     { content => "article" },
                                     { content => "book" },
                                     { content => "inbook" },
                                     { content => "bookinbook" },
                                     { content => "suppbook" },
                                     { content => "booklet" },
                                     { content => "collection" },
                                     { content => "incollection" },
                                     { content => "suppcollection" },
                                     { content => "manual" },
                                     { content => "misc" },
                                     { content => "mvbook" },
                                     { content => "mvcollection" },
                                     { content => "online" },
                                     { content => "patent" },
                                     { content => "periodical" },
                                     { content => "suppperiodical" },
                                     { content => "proceedings" },
                                     { content => "inproceedings" },
                                     { content => "reference" },
                                     { content => "inreference" },
                                     { content => "report" },
                                     { content => "set" },
                                     { content => "thesis" },
                                     { content => "unpublished" },
                                   ],
                   },
                   {
                     constraint => [
                                     {
                                       field => [{ content => "entryset" }, { content => "crossref" }],
                                       type  => "mandatory",
                                     },
                                   ],
                     entrytype  => [{ content => "set" }],
                   },
                   {
                     constraint => [
                                     {
                                       field => [
                                                  { content => "author" },
                                                  { content => "journaltitle" },
                                                  { content => "title" },
                                                ],
                                       type  => "mandatory",
                                     },
                                   ],
                     entrytype  => [{ content => "article" }],
                   },
                   {
                     constraint => [
                                     {
                                       field => [{ content => "author" }, { content => "title" }],
                                       type  => "mandatory",
                                     },
                                   ],
                     entrytype  => [
                                     { content => "book" },
                                     { content => "mvbook" },
                                     { content => "mvcollection" },
                                     { content => "mvreference" },
                                   ],
                   },
                   {
                     constraint => [
                                     {
                                       field => [
                                                  { content => "author" },
                                                  { content => "title" },
                                                  { content => "booktitle" },
                                                ],
                                       type  => "mandatory",
                                     },
                                   ],
                     entrytype  => [
                                     { content => "inbook" },
                                     { content => "bookinbook" },
                                     { content => "suppbook" },
                                   ],
                   },
                   {
                     constraint => [
                                     {
                                       field   => [{ content => "title" }],
                                       fieldor => [
                                                    { field => [{ content => "author" }, { content => "editor" }] },
                                                  ],
                                       type    => "mandatory",
                                     },
                                   ],
                     entrytype  => [{ content => "booklet" }],
                   },
                   {
                     constraint => [
                                     {
                                       field => [{ content => "editor" }, { content => "title" }],
                                       type  => "mandatory",
                                     },
                                   ],
                     entrytype  => [{ content => "collection" }, { content => "reference" }],
                   },
                   {
                     constraint => [
                                     {
                                       field => [
                                                  { content => "author" },
                                                  { content => "editor" },
                                                  { content => "title" },
                                                  { content => "booktitle" },
                                                ],
                                       type  => "mandatory",
                                     },
                                   ],
                     entrytype  => [
                                     { content => "incollection" },
                                     { content => "suppcollection" },
                                     { content => "inreference" },
                                   ],
                   },
                   {
                     constraint => [{ field => [{ content => "title" }], type => "mandatory" }],
                     entrytype  => [{ content => "manual" }],
                   },
                   {
                     constraint => [{ field => [{ content => "title" }], type => "mandatory" }],
                     entrytype  => [{ content => "misc" }],
                   },
                   {
                     constraint => [
                                     {
                                       field => [{ content => "title" }, { content => "url" }],
                                       type  => "mandatory",
                                     },
                                   ],
                     entrytype  => [{ content => "online" }],
                   },
                   {
                     constraint => [
                                     {
                                       field => [
                                                  { content => "author" },
                                                  { content => "title" },
                                                  { content => "number" },
                                                ],
                                       type  => "mandatory",
                                     },
                                   ],
                     entrytype  => [{ content => "patent" }],
                   },
                   {
                     constraint => [
                                     {
                                       field => [{ content => "editor" }, { content => "title" }],
                                       type  => "mandatory",
                                     },
                                   ],
                     entrytype  => [{ content => "periodical" }],
                   },
                   {
                     constraint => [
                                     {
                                       field => [{ content => "editor" }, { content => "title" }],
                                       type  => "mandatory",
                                     },
                                   ],
                     entrytype  => [{ content => "proceedings" }, { content => "mvproceedings" }],
                   },
                   {
                     constraint => [
                                     {
                                       field => [
                                                  { content => "author" },
                                                  { content => "editor" },
                                                  { content => "title" },
                                                  { content => "booktitle" },
                                                ],
                                       type  => "mandatory",
                                     },
                                   ],
                     entrytype  => [{ content => "inproceedings" }],
                   },
                   {
                     constraint => [
                                     {
                                       field => [
                                                  { content => "author" },
                                                  { content => "title" },
                                                  { content => "type" },
                                                  { content => "institution" },
                                                ],
                                       type  => "mandatory",
                                     },
                                   ],
                     entrytype  => [{ content => "report" }],
                   },
                   {
                     constraint => [
                                     {
                                       field => [
                                                  { content => "author" },
                                                  { content => "title" },
                                                  { content => "type" },
                                                  { content => "institution" },
                                                ],
                                       type  => "mandatory",
                                     },
                                   ],
                     entrytype  => [{ content => "thesis" }],
                   },
                   {
                     constraint => [
                                     {
                                       field => [{ content => "author" }, { content => "title" }],
                                       type  => "mandatory",
                                     },
                                   ],
                     entrytype  => [{ content => "unpublished" }],
                   },
                 ],
  datetypes   => {
                   datetype => [
                     { content => "date" },
                     { content => "origdate" },
                     { content => "eventdate" },
                     { content => "urldate" },
                   ],
                 },
  entryfields => [
                   {
                     entrytype => [{ content => "ALL" }],
                     field => [
                       { content => "abstract" },
                       { content => "annotation" },
                       { content => "authortype" },
                       { content => "bookpagination" },
                       { content => "crossref" },
                       { content => "entryset" },
                       { content => "entrysubtype" },
                       { content => "execute" },
                       { content => "file" },
                       { content => "gender" },
                       { content => "hyphenation" },
                       { content => "ids" },
                       { content => "indextitle" },
                       { content => "indexsorttitle" },
                       { content => "isan" },
                       { content => "ismn" },
                       { content => "iswc" },
                       { content => "keywords" },
                       { content => "label" },
                       { content => "library" },
                       { content => "lista" },
                       { content => "listb" },
                       { content => "listc" },
                       { content => "listd" },
                       { content => "liste" },
                       { content => "listf" },
                       { content => "nameaddon" },
                       { content => "options" },
                       { content => "origday" },
                       { content => "origendday" },
                       { content => "origendmonth" },
                       { content => "origendyear" },
                       { content => "origmonth" },
                       { content => "origyear" },
                       { content => "origlocation" },
                       { content => "origpublisher" },
                       { content => "origtitle" },
                       { content => "pagination" },
                       { content => "presort" },
                       { content => "related" },
                       { content => "relatedtype" },
                       { content => "relatedstring" },
                       { content => "shortauthor" },
                       { content => "shorteditor" },
                       { content => "shorthand" },
                       { content => "shorthandintro" },
                       { content => "shortjournal" },
                       { content => "shortseries" },
                       { content => "shorttitle" },
                       { content => "sortkey" },
                       { content => "sortname" },
                       { content => "sortshorthand" },
                       { content => "sorttitle" },
                       { content => "sortyear" },
                       { content => "usera" },
                       { content => "userb" },
                       { content => "userc" },
                       { content => "userd" },
                       { content => "usere" },
                       { content => "userf" },
                       { content => "verba" },
                       { content => "verbb" },
                       { content => "verbc" },
                       { content => "xdata" },
                       { content => "xref" },
                     ],
                   },
                   {
                     entrytype => [{ content => "set" }],
                     field => [{ content => "entryset" }, { content => "crossref" }],
                   },
                   {
                     entrytype => [{ content => "article" }],
                     field => [
                       { content => "author" },
                       { content => "journaltitle" },
                       { content => "title" },
                       { content => "day" },
                       { content => "endday" },
                       { content => "endmonth" },
                       { content => "endyear" },
                       { content => "month" },
                       { content => "year" },
                       { content => "addendum" },
                       { content => "annotator" },
                       { content => "commentator" },
                       { content => "doi" },
                       { content => "editor" },
                       { content => "editora" },
                       { content => "editorb" },
                       { content => "editorc" },
                       { content => "editoratype" },
                       { content => "editorbtype" },
                       { content => "editorctype" },
                       { content => "eid" },
                       { content => "eprint" },
                       { content => "eprintclass" },
                       { content => "eprinttype" },
                       { content => "issn" },
                       { content => "issue" },
                       { content => "issuetitle" },
                       { content => "issuesubtitle" },
                       { content => "journalsubtitle" },
                       { content => "language" },
                       { content => "note" },
                       { content => "number" },
                       { content => "origlanguage" },
                       { content => "pages" },
                       { content => "pubstate" },
                       { content => "series" },
                       { content => "subtitle" },
                       { content => "titleaddon" },
                       { content => "translator" },
                       { content => "url" },
                       { content => "urlday" },
                       { content => "urlendday" },
                       { content => "urlendmonth" },
                       { content => "urlendyear" },
                       { content => "urlmonth" },
                       { content => "urlyear" },
                       { content => "version" },
                       { content => "volume" },
                     ],
                   },
                   {
                     entrytype => [{ content => "bibnote" }],
                     field => [{ content => "note" }],
                   },
                   {
                     entrytype => [{ content => "book" }],
                     field => [
                       { content => "author" },
                       { content => "title" },
                       { content => "day" },
                       { content => "endday" },
                       { content => "endmonth" },
                       { content => "endyear" },
                       { content => "month" },
                       { content => "year" },
                       { content => "addendum" },
                       { content => "afterword" },
                       { content => "annotator" },
                       { content => "chapter" },
                       { content => "commentator" },
                       { content => "doi" },
                       { content => "edition" },
                       { content => "editor" },
                       { content => "editora" },
                       { content => "editorb" },
                       { content => "editorc" },
                       { content => "editoratype" },
                       { content => "editorbtype" },
                       { content => "editorctype" },
                       { content => "eprint" },
                       { content => "eprintclass" },
                       { content => "eprinttype" },
                       { content => "foreword" },
                       { content => "introduction" },
                       { content => "isbn" },
                       { content => "language" },
                       { content => "location" },
                       { content => "maintitle" },
                       { content => "maintitleaddon" },
                       { content => "mainsubtitle" },
                       { content => "note" },
                       { content => "number" },
                       { content => "origlanguage" },
                       { content => "pages" },
                       { content => "pagetotal" },
                       { content => "part" },
                       { content => "publisher" },
                       { content => "pubstate" },
                       { content => "series" },
                       { content => "subtitle" },
                       { content => "titleaddon" },
                       { content => "translator" },
                       { content => "url" },
                       { content => "urlday" },
                       { content => "urlendday" },
                       { content => "urlendmonth" },
                       { content => "urlendyear" },
                       { content => "urlmonth" },
                       { content => "urlyear" },
                       { content => "volume" },
                       { content => "volumes" },
                     ],
                   },
                   {
                     entrytype => [{ content => "mvbook" }],
                     field => [
                       { content => "author" },
                       { content => "title" },
                       { content => "day" },
                       { content => "endday" },
                       { content => "endmonth" },
                       { content => "endyear" },
                       { content => "month" },
                       { content => "year" },
                       { content => "addendum" },
                       { content => "afterword" },
                       { content => "annotator" },
                       { content => "commentator" },
                       { content => "doi" },
                       { content => "edition" },
                       { content => "editor" },
                       { content => "editora" },
                       { content => "editorb" },
                       { content => "editorc" },
                       { content => "editoratype" },
                       { content => "editorbtype" },
                       { content => "editorctype" },
                       { content => "eprint" },
                       { content => "eprintclass" },
                       { content => "eprinttype" },
                       { content => "foreword" },
                       { content => "introduction" },
                       { content => "isbn" },
                       { content => "language" },
                       { content => "location" },
                       { content => "note" },
                       { content => "number" },
                       { content => "origlanguage" },
                       { content => "publisher" },
                       { content => "pubstate" },
                       { content => "subtitle" },
                       { content => "titleaddon" },
                       { content => "translator" },
                       { content => "url" },
                       { content => "urlday" },
                       { content => "urlendday" },
                       { content => "urlendmonth" },
                       { content => "urlendyear" },
                       { content => "urlmonth" },
                       { content => "urlyear" },
                       { content => "volume" },
                       { content => "volumes" },
                     ],
                   },
                   {
                     entrytype => [
                       { content => "inbook" },
                       { content => "bookinbook" },
                       { content => "suppbook" },
                     ],
                     field => [
                       { content => "author" },
                       { content => "title" },
                       { content => "booktitle" },
                       { content => "day" },
                       { content => "endday" },
                       { content => "endmonth" },
                       { content => "endyear" },
                       { content => "month" },
                       { content => "year" },
                       { content => "addendum" },
                       { content => "afterword" },
                       { content => "annotator" },
                       { content => "bookauthor" },
                       { content => "booksubtitle" },
                       { content => "booktitleaddon" },
                       { content => "chapter" },
                       { content => "commentator" },
                       { content => "doi" },
                       { content => "edition" },
                       { content => "editor" },
                       { content => "editora" },
                       { content => "editorb" },
                       { content => "editorc" },
                       { content => "editoratype" },
                       { content => "editorbtype" },
                       { content => "editorctype" },
                       { content => "eprint" },
                       { content => "eprintclass" },
                       { content => "eprinttype" },
                       { content => "foreword" },
                       { content => "introduction" },
                       { content => "isbn" },
                       { content => "language" },
                       { content => "location" },
                       { content => "mainsubtitle" },
                       { content => "maintitle" },
                       { content => "maintitleaddon" },
                       { content => "note" },
                       { content => "number" },
                       { content => "origlanguage" },
                       { content => "part" },
                       { content => "publisher" },
                       { content => "pages" },
                       { content => "pubstate" },
                       { content => "series" },
                       { content => "subtitle" },
                       { content => "titleaddon" },
                       { content => "translator" },
                       { content => "url" },
                       { content => "urlday" },
                       { content => "urlendday" },
                       { content => "urlendmonth" },
                       { content => "urlendyear" },
                       { content => "urlmonth" },
                       { content => "urlyear" },
                       { content => "volume" },
                       { content => "volumes" },
                     ],
                   },
                   {
                     entrytype => [{ content => "booklet" }],
                     field => [
                       { content => "author" },
                       { content => "editor" },
                       { content => "title" },
                       { content => "day" },
                       { content => "endday" },
                       { content => "endmonth" },
                       { content => "endyear" },
                       { content => "month" },
                       { content => "year" },
                       { content => "addendum" },
                       { content => "chapter" },
                       { content => "doi" },
                       { content => "eprint" },
                       { content => "eprintclass" },
                       { content => "eprinttype" },
                       { content => "howpublished" },
                       { content => "language" },
                       { content => "location" },
                       { content => "note" },
                       { content => "pages" },
                       { content => "pagetotal" },
                       { content => "pubstate" },
                       { content => "subtitle" },
                       { content => "titleaddon" },
                       { content => "type" },
                       { content => "url" },
                       { content => "urlday" },
                       { content => "urlendday" },
                       { content => "urlendmonth" },
                       { content => "urlendyear" },
                       { content => "urlmonth" },
                       { content => "urlyear" },
                     ],
                   },
                   {
                     entrytype => [{ content => "collection" }, { content => "reference" }],
                     field => [
                       { content => "editor" },
                       { content => "title" },
                       { content => "day" },
                       { content => "endday" },
                       { content => "endmonth" },
                       { content => "endyear" },
                       { content => "month" },
                       { content => "year" },
                       { content => "addendum" },
                       { content => "afterword" },
                       { content => "annotator" },
                       { content => "chapter" },
                       { content => "commentator" },
                       { content => "doi" },
                       { content => "edition" },
                       { content => "editora" },
                       { content => "editorb" },
                       { content => "editorc" },
                       { content => "editoratype" },
                       { content => "editorbtype" },
                       { content => "editorctype" },
                       { content => "eprint" },
                       { content => "eprintclass" },
                       { content => "eprinttype" },
                       { content => "foreword" },
                       { content => "introduction" },
                       { content => "isbn" },
                       { content => "language" },
                       { content => "location" },
                       { content => "mainsubtitle" },
                       { content => "maintitle" },
                       { content => "maintitleaddon" },
                       { content => "note" },
                       { content => "number" },
                       { content => "origlanguage" },
                       { content => "pages" },
                       { content => "pagetotal" },
                       { content => "part" },
                       { content => "publisher" },
                       { content => "pubstate" },
                       { content => "series" },
                       { content => "subtitle" },
                       { content => "titleaddon" },
                       { content => "translator" },
                       { content => "url" },
                       { content => "urlday" },
                       { content => "urlendday" },
                       { content => "urlendmonth" },
                       { content => "urlendyear" },
                       { content => "urlmonth" },
                       { content => "urlyear" },
                       { content => "volume" },
                       { content => "volumes" },
                     ],
                   },
                   {
                     entrytype => [{ content => "mvcollection" }, { content => "mvreference" }],
                     field => [
                       { content => "author" },
                       { content => "title" },
                       { content => "day" },
                       { content => "endday" },
                       { content => "endmonth" },
                       { content => "endyear" },
                       { content => "month" },
                       { content => "year" },
                       { content => "addendum" },
                       { content => "afterword" },
                       { content => "annotator" },
                       { content => "commentator" },
                       { content => "doi" },
                       { content => "edition" },
                       { content => "editor" },
                       { content => "editora" },
                       { content => "editorb" },
                       { content => "editorc" },
                       { content => "editoratype" },
                       { content => "editorbtype" },
                       { content => "editorctype" },
                       { content => "eprint" },
                       { content => "eprintclass" },
                       { content => "eprinttype" },
                       { content => "foreword" },
                       { content => "introduction" },
                       { content => "isbn" },
                       { content => "language" },
                       { content => "location" },
                       { content => "note" },
                       { content => "number" },
                       { content => "origlanguage" },
                       { content => "publisher" },
                       { content => "pubstate" },
                       { content => "subtitle" },
                       { content => "titleaddon" },
                       { content => "translator" },
                       { content => "url" },
                       { content => "urlday" },
                       { content => "urlendday" },
                       { content => "urlendmonth" },
                       { content => "urlendyear" },
                       { content => "urlmonth" },
                       { content => "urlyear" },
                       { content => "volume" },
                       { content => "volumes" },
                     ],
                   },
                   {
                     entrytype => [
                       { content => "incollection" },
                       { content => "suppcollection" },
                       { content => "inreference" },
                     ],
                     field => [
                       { content => "author" },
                       { content => "editor" },
                       { content => "title" },
                       { content => "booktitle" },
                       { content => "day" },
                       { content => "endday" },
                       { content => "endmonth" },
                       { content => "endyear" },
                       { content => "month" },
                       { content => "year" },
                       { content => "addendum" },
                       { content => "afterword" },
                       { content => "annotator" },
                       { content => "booksubtitle" },
                       { content => "booktitleaddon" },
                       { content => "chapter" },
                       { content => "commentator" },
                       { content => "doi" },
                       { content => "edition" },
                       { content => "editora" },
                       { content => "editorb" },
                       { content => "editorc" },
                       { content => "editoratype" },
                       { content => "editorbtype" },
                       { content => "editorctype" },
                       { content => "eprint" },
                       { content => "eprintclass" },
                       { content => "eprinttype" },
                       { content => "foreword" },
                       { content => "introduction" },
                       { content => "isbn" },
                       { content => "language" },
                       { content => "location" },
                       { content => "mainsubtitle" },
                       { content => "maintitle" },
                       { content => "maintitleaddon" },
                       { content => "note" },
                       { content => "number" },
                       { content => "origlanguage" },
                       { content => "pages" },
                       { content => "part" },
                       { content => "publisher" },
                       { content => "pubstate" },
                       { content => "series" },
                       { content => "subtitle" },
                       { content => "titleaddon" },
                       { content => "translator" },
                       { content => "url" },
                       { content => "urlday" },
                       { content => "urlendday" },
                       { content => "urlendmonth" },
                       { content => "urlendyear" },
                       { content => "urlmonth" },
                       { content => "urlyear" },
                       { content => "volume" },
                       { content => "volumes" },
                     ],
                   },
                   {
                     entrytype => [{ content => "manual" }],
                     field => [
                       { content => "title" },
                       { content => "day" },
                       { content => "endday" },
                       { content => "endmonth" },
                       { content => "endyear" },
                       { content => "month" },
                       { content => "year" },
                       { content => "addendum" },
                       { content => "author" },
                       { content => "chapter" },
                       { content => "doi" },
                       { content => "edition" },
                       { content => "editor" },
                       { content => "eprint" },
                       { content => "eprintclass" },
                       { content => "eprinttype" },
                       { content => "isbn" },
                       { content => "language" },
                       { content => "location" },
                       { content => "note" },
                       { content => "number" },
                       { content => "organization" },
                       { content => "pages" },
                       { content => "pagetotal" },
                       { content => "publisher" },
                       { content => "pubstate" },
                       { content => "series" },
                       { content => "subtitle" },
                       { content => "titleaddon" },
                       { content => "type" },
                       { content => "url" },
                       { content => "urlday" },
                       { content => "urlendday" },
                       { content => "urlendmonth" },
                       { content => "urlendyear" },
                       { content => "urlmonth" },
                       { content => "urlyear" },
                       { content => "version" },
                     ],
                   },
                   {
                     entrytype => [{ content => "misc" }],
                     field => [
                       { content => "title" },
                       { content => "day" },
                       { content => "endday" },
                       { content => "endmonth" },
                       { content => "endyear" },
                       { content => "day" },
                       { content => "endday" },
                       { content => "endmonth" },
                       { content => "endyear" },
                       { content => "month" },
                       { content => "year" },
                       { content => "addendum" },
                       { content => "author" },
                       { content => "doi" },
                       { content => "editor" },
                       { content => "eprint" },
                       { content => "eprintclass" },
                       { content => "eprinttype" },
                       { content => "howpublished" },
                       { content => "language" },
                       { content => "location" },
                       { content => "note" },
                       { content => "organization" },
                       { content => "pubstate" },
                       { content => "subtitle" },
                       { content => "titleaddon" },
                       { content => "type" },
                       { content => "url" },
                       { content => "urlday" },
                       { content => "urlendday" },
                       { content => "urlendmonth" },
                       { content => "urlendyear" },
                       { content => "urlmonth" },
                       { content => "urlyear" },
                       { content => "version" },
                     ],
                   },
                   {
                     entrytype => [{ content => "online" }],
                     field => [
                       { content => "title" },
                       { content => "url" },
                       { content => "addendum" },
                       { content => "author" },
                       { content => "editor" },
                       { content => "language" },
                       { content => "month" },
                       { content => "note" },
                       { content => "organization" },
                       { content => "pubstate" },
                       { content => "subtitle" },
                       { content => "titleaddon" },
                       { content => "urlday" },
                       { content => "urlendday" },
                       { content => "urlendmonth" },
                       { content => "urlendyear" },
                       { content => "urlmonth" },
                       { content => "urlyear" },
                       { content => "version" },
                       { content => "year" },
                     ],
                   },
                   {
                     entrytype => [{ content => "patent" }],
                     field => [
                       { content => "author" },
                       { content => "title" },
                       { content => "number" },
                       { content => "day" },
                       { content => "endday" },
                       { content => "endmonth" },
                       { content => "endyear" },
                       { content => "month" },
                       { content => "year" },
                       { content => "addendum" },
                       { content => "doi" },
                       { content => "eprint" },
                       { content => "eprintclass" },
                       { content => "eprinttype" },
                       { content => "holder" },
                       { content => "location" },
                       { content => "note" },
                       { content => "pubstate" },
                       { content => "subtitle" },
                       { content => "titleaddon" },
                       { content => "type" },
                       { content => "url" },
                       { content => "urlday" },
                       { content => "urlendday" },
                       { content => "urlendmonth" },
                       { content => "urlendyear" },
                       { content => "urlmonth" },
                       { content => "urlyear" },
                       { content => "version" },
                     ],
                   },
                   {
                     entrytype => [{ content => "periodical" }],
                     field => [
                       { content => "editor" },
                       { content => "title" },
                       { content => "day" },
                       { content => "endday" },
                       { content => "endmonth" },
                       { content => "endyear" },
                       { content => "month" },
                       { content => "year" },
                       { content => "addendum" },
                       { content => "doi" },
                       { content => "editora" },
                       { content => "editorb" },
                       { content => "editorc" },
                       { content => "editoratype" },
                       { content => "editorbtype" },
                       { content => "editorctype" },
                       { content => "eprint" },
                       { content => "eprintclass" },
                       { content => "eprinttype" },
                       { content => "issn" },
                       { content => "issue" },
                       { content => "issuesubtitle" },
                       { content => "issuetitle" },
                       { content => "language" },
                       { content => "note" },
                       { content => "number" },
                       { content => "pubstate" },
                       { content => "series" },
                       { content => "subtitle" },
                       { content => "url" },
                       { content => "urlday" },
                       { content => "urlendday" },
                       { content => "urlendmonth" },
                       { content => "urlendyear" },
                       { content => "urlmonth" },
                       { content => "urlyear" },
                       { content => "volume" },
                     ],
                   },
                   {
                     entrytype => [{ content => "mvproceedings" }],
                     field => [
                       { content => "editor" },
                       { content => "title" },
                       { content => "day" },
                       { content => "endday" },
                       { content => "endmonth" },
                       { content => "endyear" },
                       { content => "month" },
                       { content => "year" },
                       { content => "addendum" },
                       { content => "doi" },
                       { content => "eprint" },
                       { content => "eprintclass" },
                       { content => "eprinttype" },
                       { content => "eventday" },
                       { content => "eventendday" },
                       { content => "eventendmonth" },
                       { content => "eventendyear" },
                       { content => "eventmonth" },
                       { content => "eventyear" },
                       { content => "eventtitle" },
                       { content => "isbn" },
                       { content => "language" },
                       { content => "location" },
                       { content => "note" },
                       { content => "number" },
                       { content => "organization" },
                       { content => "pagetotal" },
                       { content => "publisher" },
                       { content => "pubstate" },
                       { content => "series" },
                       { content => "subtitle" },
                       { content => "titleaddon" },
                       { content => "url" },
                       { content => "urlday" },
                       { content => "urlendday" },
                       { content => "urlendmonth" },
                       { content => "urlendyear" },
                       { content => "urlmonth" },
                       { content => "urlyear" },
                       { content => "venue" },
                       { content => "volumes" },
                     ],
                   },
                   {
                     entrytype => [{ content => "proceedings" }],
                     field => [
                       { content => "editor" },
                       { content => "title" },
                       { content => "day" },
                       { content => "endday" },
                       { content => "endmonth" },
                       { content => "endyear" },
                       { content => "month" },
                       { content => "year" },
                       { content => "addendum" },
                       { content => "chapter" },
                       { content => "doi" },
                       { content => "eprint" },
                       { content => "eprintclass" },
                       { content => "eprinttype" },
                       { content => "eventday" },
                       { content => "eventendday" },
                       { content => "eventendmonth" },
                       { content => "eventendyear" },
                       { content => "eventmonth" },
                       { content => "eventyear" },
                       { content => "eventtitle" },
                       { content => "isbn" },
                       { content => "language" },
                       { content => "location" },
                       { content => "mainsubtitle" },
                       { content => "maintitle" },
                       { content => "maintitleaddon" },
                       { content => "note" },
                       { content => "number" },
                       { content => "organization" },
                       { content => "pages" },
                       { content => "pagetotal" },
                       { content => "part" },
                       { content => "publisher" },
                       { content => "pubstate" },
                       { content => "series" },
                       { content => "subtitle" },
                       { content => "titleaddon" },
                       { content => "url" },
                       { content => "urlday" },
                       { content => "urlendday" },
                       { content => "urlendmonth" },
                       { content => "urlendyear" },
                       { content => "urlmonth" },
                       { content => "urlyear" },
                       { content => "venue" },
                       { content => "volume" },
                       { content => "volumes" },
                     ],
                   },
                   {
                     entrytype => [{ content => "inproceedings" }],
                     field => [
                       { content => "author" },
                       { content => "editor" },
                       { content => "title" },
                       { content => "booktitle" },
                       { content => "day" },
                       { content => "endday" },
                       { content => "endmonth" },
                       { content => "endyear" },
                       { content => "month" },
                       { content => "year" },
                       { content => "addendum" },
                       { content => "booksubtitle" },
                       { content => "booktitleaddon" },
                       { content => "chapter" },
                       { content => "doi" },
                       { content => "eprint" },
                       { content => "eprintclass" },
                       { content => "eprinttype" },
                       { content => "eventday" },
                       { content => "eventendday" },
                       { content => "eventendmonth" },
                       { content => "eventendyear" },
                       { content => "eventmonth" },
                       { content => "eventyear" },
                       { content => "eventtitle" },
                       { content => "isbn" },
                       { content => "language" },
                       { content => "location" },
                       { content => "mainsubtitle" },
                       { content => "maintitle" },
                       { content => "maintitleaddon" },
                       { content => "note" },
                       { content => "number" },
                       { content => "organization" },
                       { content => "pages" },
                       { content => "part" },
                       { content => "publisher" },
                       { content => "pubstate" },
                       { content => "series" },
                       { content => "subtitle" },
                       { content => "titleaddon" },
                       { content => "url" },
                       { content => "urlday" },
                       { content => "urlendday" },
                       { content => "urlendmonth" },
                       { content => "urlendyear" },
                       { content => "urlmonth" },
                       { content => "urlyear" },
                       { content => "venue" },
                       { content => "volume" },
                       { content => "volumes" },
                     ],
                   },
                   {
                     entrytype => [{ content => "report" }],
                     field => [
                       { content => "author" },
                       { content => "title" },
                       { content => "type" },
                       { content => "institution" },
                       { content => "day" },
                       { content => "endday" },
                       { content => "endmonth" },
                       { content => "endyear" },
                       { content => "month" },
                       { content => "year" },
                       { content => "addendum" },
                       { content => "chapter" },
                       { content => "doi" },
                       { content => "eprint" },
                       { content => "eprintclass" },
                       { content => "eprinttype" },
                       { content => "isrn" },
                       { content => "language" },
                       { content => "location" },
                       { content => "note" },
                       { content => "number" },
                       { content => "pages" },
                       { content => "pagetotal" },
                       { content => "pubstate" },
                       { content => "subtitle" },
                       { content => "titleaddon" },
                       { content => "url" },
                       { content => "urlday" },
                       { content => "urlendday" },
                       { content => "urlendmonth" },
                       { content => "urlendyear" },
                       { content => "urlmonth" },
                       { content => "urlyear" },
                       { content => "version" },
                     ],
                   },
                   {
                     entrytype => [{ content => "thesis" }],
                     field => [
                       { content => "author" },
                       { content => "title" },
                       { content => "type" },
                       { content => "institution" },
                       { content => "day" },
                       { content => "endday" },
                       { content => "endmonth" },
                       { content => "endyear" },
                       { content => "month" },
                       { content => "year" },
                       { content => "addendum" },
                       { content => "chapter" },
                       { content => "doi" },
                       { content => "eprint" },
                       { content => "eprintclass" },
                       { content => "eprinttype" },
                       { content => "language" },
                       { content => "location" },
                       { content => "note" },
                       { content => "pages" },
                       { content => "pagetotal" },
                       { content => "pubstate" },
                       { content => "subtitle" },
                       { content => "titleaddon" },
                       { content => "url" },
                       { content => "urlday" },
                       { content => "urlendday" },
                       { content => "urlendmonth" },
                       { content => "urlendyear" },
                       { content => "urlmonth" },
                       { content => "urlyear" },
                     ],
                   },
                   {
                     entrytype => [{ content => "unpublished" }],
                     field => [
                       { content => "author" },
                       { content => "title" },
                       { content => "day" },
                       { content => "endday" },
                       { content => "endmonth" },
                       { content => "endyear" },
                       { content => "month" },
                       { content => "year" },
                       { content => "addendum" },
                       { content => "howpublished" },
                       { content => "language" },
                       { content => "location" },
                       { content => "note" },
                       { content => "pubstate" },
                       { content => "subtitle" },
                       { content => "titleaddon" },
                       { content => "url" },
                       { content => "urlday" },
                       { content => "urlendday" },
                       { content => "urlendmonth" },
                       { content => "urlendyear" },
                       { content => "urlmonth" },
                       { content => "urlyear" },
                     ],
                   },
                   {
                     entrytype => [{ content => "xdata" }],
                     field => [
                       { content => "ALL" },
                               ]
                   }
                 ],
  entrytypes  => {
                   entrytype => [
                     { content => "article" },
                     { content => "artwork" },
                     { content => "audio" },
                     { content => "bibnote" },
                     { content => "book" },
                     { content => "bookinbook" },
                     { content => "booklet" },
                     { content => "collection" },
                     { content => "commentary" },
                     { content => "customa" },
                     { content => "customb" },
                     { content => "customc" },
                     { content => "customd" },
                     { content => "custome" },
                     { content => "customf" },
                     { content => "inbook" },
                     { content => "incollection" },
                     { content => "inproceedings" },
                     { content => "inreference" },
                     { content => "image" },
                     { content => "jurisdiction" },
                     { content => "legal" },
                     { content => "legislation" },
                     { content => "letter" },
                     { content => "manual" },
                     { content => "misc" },
                     { content => "movie" },
                     { content => "music" },
                     { content => "mvcollection" },
                     { content => "mvbook" },
                     { content => "online" },
                     { content => "patent" },
                     { content => "performance" },
                     { content => "periodical" },
                     { content => "proceedings" },
                     { content => "reference" },
                     { content => "report" },
                     { content => "review" },
                     { content => "set" },
                     { content => "software" },
                     { content => "standard" },
                     { content => "suppbook" },
                     { content => "suppcollection" },
                     { content => "thesis" },
                     { content => "unpublished" },
                     { content => "video" },
                     { content => "xdata" },
                   ],
                 },
  fields      => {
                   field => [
                     { content => "abstract", datatype => "literal", fieldtype => "field" },
                     { content => "addendum", datatype => "literal", fieldtype => "field" },
                     { content => "afterword", datatype => "name", fieldtype => "list" },
                     { content => "annotation", datatype => "literal", fieldtype => "field" },
                     { content => "annotator", datatype => "name", fieldtype => "list" },
                     { content => "author", datatype => "name", fieldtype => "list" },
                     { content => "authortype", datatype => "key", fieldtype => "field" },
                     { content => "bookauthor", datatype => "name", fieldtype => "list" },
                     { content => "bookpagination", datatype => "key", fieldtype => "field" },
                     { content => "booksubtitle", datatype => "literal", fieldtype => "field" },
                     { content => "booktitle", datatype => "literal", fieldtype => "field" },
                     {
                       content   => "booktitleaddon",
                       datatype  => "literal",
                       fieldtype => "field",
                     },
                     { content => "chapter", datatype => "literal", fieldtype => "field" },
                     { content => "commentator", datatype => "name", fieldtype => "list" },
                     { content => "crossref", datatype => "entrykey", fieldtype => "field" },
                     { content => "day", datatype => "datepart", fieldtype => "field" },
                     { content => "doi", datatype => "verbatim", fieldtype => "field" },
                     { content => "edition", datatype => "literal", fieldtype => "field" },
                     { content => "editor", datatype => "name", fieldtype => "list" },
                     { content => "editora", datatype => "name", fieldtype => "list" },
                     { content => "editoratype", datatype => "key", fieldtype => "field" },
                     { content => "editorb", datatype => "name", fieldtype => "list" },
                     { content => "editorbtype", datatype => "key", fieldtype => "field" },
                     { content => "editorc", datatype => "name", fieldtype => "list" },
                     { content => "editorctype", datatype => "key", fieldtype => "field" },
                     { content => "editortype", datatype => "key", fieldtype => "field" },
                     { content => "eid", datatype => "literal", fieldtype => "field" },
                     { content => "endday", datatype => "datepart", fieldtype => "field" },
                     { content => "endmonth", datatype => "datepart", fieldtype => "field" },
                     {
                       content   => "endyear",
                       datatype  => "datepart",
                       fieldtype => "field",
                       nullok    => "true",
                     },
                     {
                       content     => "entryset",
                       datatype    => "literal",
                       fieldtype   => "field",
                       skip_output => "true",
                     },
                     { content => "entrysubtype", datatype => "literal", fieldtype => "field" },
                     { content => "eprint", datatype => "verbatim", fieldtype => "field" },
                     { content => "eprintclass", datatype => "literal", fieldtype => "field" },
                     { content => "eprinttype", datatype => "literal", fieldtype => "field" },
                     { content => "eventday", datatype => "datepart", fieldtype => "field" },
                     { content => "eventendday", datatype => "datepart", fieldtype => "field" },
                     { content => "eventendmonth", datatype => "datepart", fieldtype => "field" },
                     {
                       content   => "eventendyear",
                       datatype  => "datepart",
                       fieldtype => "field",
                       nullok    => "true",
                     },
                     { content => "eventmonth", datatype => "datepart", fieldtype => "field" },
                     { content => "eventtitle", datatype => "literal", fieldtype => "field" },
                     { content => "eventyear", datatype => "datepart", fieldtype => "field" },
                     { content => "execute", datatype => "literal", fieldtype => "field" },
                     { content => "file", datatype => "verbatim", fieldtype => "field" },
                     { content => "foreword", datatype => "name", fieldtype => "list" },
                     { content => "gender", datatype => "literal", fieldtype => "field" },
                     { content => "holder", datatype => "name", fieldtype => "list" },
                     { content => "howpublished", datatype => "literal", fieldtype => "field" },
                     { content => "hyphenation", datatype => "literal", fieldtype => "field" },
                     {
                       content   => "indexsorttitle",
                       datatype  => "literal",
                       fieldtype => "field",
                     },
                     { content => "indextitle", datatype => "literal", fieldtype => "field" },
                     { content => "institution", datatype => "literal", fieldtype => "list" },
                     { content => "introduction", datatype => "name", fieldtype => "list" },
                     { content => "isan", datatype => "literal", fieldtype => "field" },
                     { content => "isbn", datatype => "literal", fieldtype => "field" },
                     { content => "ismn", datatype => "literal", fieldtype => "field" },
                     { content => "isrn", datatype => "literal", fieldtype => "field" },
                     { content => "issn", datatype => "literal", fieldtype => "field" },
                     { content => "issue", datatype => "literal", fieldtype => "field" },
                     { content => "issuesubtitle", datatype => "literal", fieldtype => "field" },
                     { content => "issuetitle", datatype => "literal", fieldtype => "field" },
                     { content => "iswc", datatype => "literal", fieldtype => "field" },
                     {
                       content   => "journalsubtitle",
                       datatype  => "literal",
                       fieldtype => "field",
                     },
                     { content => "journaltitle", datatype => "literal", fieldtype => "field" },
                     { content => "keywords", datatype => "csv", fieldtype => "field" },
                     { content => "label", datatype => "literal", fieldtype => "field" },
                     { content => "language", datatype => "key", fieldtype => "list" },
                     { content => "library", datatype => "literal", fieldtype => "field" },
                     { content => "lista", datatype => "literal", fieldtype => "list" },
                     { content => "listb", datatype => "literal", fieldtype => "list" },
                     { content => "listc", datatype => "literal", fieldtype => "list" },
                     { content => "listd", datatype => "literal", fieldtype => "list" },
                     { content => "liste", datatype => "literal", fieldtype => "list" },
                     { content => "listf", datatype => "literal", fieldtype => "list" },
                     { content => "location", datatype => "literal", fieldtype => "list" },
                     { content => "mainsubtitle", datatype => "literal", fieldtype => "field" },
                     { content => "maintitle", datatype => "literal", fieldtype => "field" },
                     {
                       content   => "maintitleaddon",
                       datatype  => "literal",
                       fieldtype => "field",
                     },
                     { content => "month", datatype => "datepart", fieldtype => "field" },
                     { content => "namea", datatype => "name", fieldtype => "list" },
                     { content => "nameaddon", datatype => "literal", fieldtype => "field" },
                     { content => "nameatype", datatype => "key", fieldtype => "field" },
                     { content => "nameb", datatype => "name", fieldtype => "list" },
                     { content => "namebtype", datatype => "key", fieldtype => "field" },
                     { content => "namec", datatype => "name", fieldtype => "list" },
                     { content => "namectype", datatype => "key", fieldtype => "field" },
                     { content => "note", datatype => "literal", fieldtype => "field" },
                     { content => "number", datatype => "literal", fieldtype => "field" },
                     { content => "options", datatype => "csv", fieldtype => "field" },
                     { content => "organization", datatype => "literal", fieldtype => "list" },
                     { content => "origday", datatype => "datepart", fieldtype => "field" },
                     { content => "origendday", datatype => "datepart", fieldtype => "field" },
                     { content => "origendmonth", datatype => "datepart", fieldtype => "field" },
                     {
                       content   => "origendyear",
                       datatype  => "datepart",
                       fieldtype => "field",
                       nullok    => "true",
                     },
                     { content => "origlanguage", datatype => "key", fieldtype => "field" },
                     { content => "origlocation", datatype => "literal", fieldtype => "list" },
                     { content => "origmonth", datatype => "datepart", fieldtype => "field" },
                     { content => "origpublisher", datatype => "literal", fieldtype => "list" },
                     { content => "origtitle", datatype => "literal", fieldtype => "field" },
                     { content => "origyear", datatype => "datepart", fieldtype => "field" },
                     { content => "pages", datatype => "range", fieldtype => "field" },
                     { content => "pagetotal", datatype => "literal", fieldtype => "field" },
                     { content => "pagination", datatype => "key", fieldtype => "field" },
                     { content => "part", datatype => "literal", fieldtype => "field" },
                     {
                       content     => "presort",
                       datatype    => "literal",
                       fieldtype   => "field",
                       skip_output => "true",
                     },
                     { content => "publisher", datatype => "literal", fieldtype => "list" },
                     { content => "pubstate", datatype => "key", fieldtype => "field" },
                     { content => "related", datatype => "literal", fieldtype => "field" },
                     { content => "relatedtype", datatype => "literal", fieldtype => "field" },
                     { content => "relatedstring", datatype => "literal", fieldtype => "field" },
                     { content => "reprinttitle", datatype => "literal", fieldtype => "field" },
                     { content => "series", datatype => "literal", fieldtype => "field" },
                     { content => "shortauthor", datatype => "name", fieldtype => "list" },
                     { content => "shorteditor", datatype => "name", fieldtype => "list" },
                     { content => "shorthand", datatype => "literal", fieldtype => "field" },
                     {
                       content   => "shorthandintro",
                       datatype  => "literal",
                       fieldtype => "field",
                     },
                     { content => "shortjournal", datatype => "literal", fieldtype => "field" },
                     { content => "shortseries", datatype => "literal", fieldtype => "field" },
                     { content => "shorttitle", datatype => "literal", fieldtype => "field" },
                     {
                       content     => "sortkey",
                       datatype    => "literal",
                       fieldtype   => "field",
                       skip_output => "true",
                     },
                     {
                       content     => "sortname",
                       datatype    => "name",
                       fieldtype   => "list",
                       skip_output => "true",
                     },
                     {
                       content     => "sortshorthand",
                       datatype    => "literal",
                       fieldtype   => "field",
                       skip_output => "true",
                     },
                     {
                       content     => "sorttitle",
                       datatype    => "literal",
                       fieldtype   => "field",
                       skip_output => "true",
                     },
                     {
                       content     => "sortyear",
                       datatype    => "literal",
                       fieldtype   => "field",
                       skip_output => "true",
                     },
                     { content => "subtitle", datatype => "literal", fieldtype => "field" },
                     { content => "title", datatype => "literal", fieldtype => "field" },
                     { content => "titleaddon", datatype => "literal", fieldtype => "field" },
                     { content => "translator", datatype => "name", fieldtype => "list" },
                     { content => "type", datatype => "key", fieldtype => "field" },
                     { content => "url", datatype => "verbatim", fieldtype => "field" },
                     { content => "usera", datatype => "literal", fieldtype => "field" },
                     { content => "userb", datatype => "literal", fieldtype => "field" },
                     { content => "userc", datatype => "literal", fieldtype => "field" },
                     { content => "userd", datatype => "literal", fieldtype => "field" },
                     { content => "usere", datatype => "literal", fieldtype => "field" },
                     { content => "userf", datatype => "literal", fieldtype => "field" },
                     { content => "urlday", datatype => "datepart", fieldtype => "field" },
                     { content => "urlendday", datatype => "datepart", fieldtype => "field" },
                     { content => "urlendmonth", datatype => "datepart", fieldtype => "field" },
                     {
                       content   => "urlendyear",
                       datatype  => "datepart",
                       fieldtype => "field",
                       nullok    => "true",
                     },
                     { content => "urlmonth", datatype => "datepart", fieldtype => "field" },
                     { content => "urlyear", datatype => "datepart", fieldtype => "field" },
                     { content => "venue", datatype => "literal", fieldtype => "field" },
                     { content => "verba", datatype => "verbatim", fieldtype => "field" },
                     { content => "verbb", datatype => "verbatim", fieldtype => "field" },
                     { content => "verbc", datatype => "verbatim", fieldtype => "field" },
                     { content => "version", datatype => "literal", fieldtype => "field" },
                     { content => "volume", datatype => "literal", fieldtype => "field" },
                     { content => "volumes", datatype => "literal", fieldtype => "field" },
                     { content => "xdata", datatype => "entrykey", fieldtype => "list", skip_output => "true" },
                     { content => "xref", datatype => "entrykey", fieldtype => "field" },
                     { content => "year", datatype => "datepart", fieldtype => "field" },
                   ],
                 },
                }
);


# Set up some encoding aliases to map \inputen{c,x} encoding names to Encode
# It seems that inputen{c,x} has a different idea of nextstep than Encode
# so we push it to MacRoman
define_alias('ansinew'        => 'cp1252'); # inputenc alias for cp1252
define_alias('applemac'       => 'MacRoman');
define_alias('applemacce'     => 'MacCentralEurRoman');
define_alias('next'           => 'MacRoman');
define_alias('x-mac-roman'    => 'MacRoman');
define_alias('x-mac-centeuro' => 'MacCentralEurRoman');
define_alias('x-mac-cyrillic' => 'MacCyrillic');
define_alias('x-nextstep'     => 'MacRoman');
define_alias('x-ascii'        => 'ascii'); # Encode doesn't resolve this one by default
define_alias('lutf8'          => 'UTF-8'); # Luatex
define_alias('utf8x'          => 'UTF-8'); # UCS (old)

# Defines the scope of each of the BibLaTeX configuration options
#
# PRESORT is not a "real" biblatex option passed by biblatex. It is defined
# by the biblatex \DeclarePresort macro and is stored in here as it
# can be global/per-type or per-entry and therefore it's natural to store it here.
our %CONFIG_SCOPE_BIBLATEX = (
  alphaothers        => {GLOBAL => 1, PER_TYPE => 1, PER_ENTRY => 0},
  controlversion     => {GLOBAL => 1, PER_TYPE => 0, PER_ENTRY => 0},
  debug              => {GLOBAL => 1, PER_TYPE => 0, PER_ENTRY => 0},
  dataonly           => {GLOBAL => 0, PER_TYPE => 0, PER_ENTRY => 1},
  displaymode        => {GLOBAL => 1, PER_TYPE => 1, PER_ENTRY => 1},
  inheritance        => {GLOBAL => 1, PER_TYPE => 0, PER_ENTRY => 0},
  labelalpha         => {GLOBAL => 1, PER_TYPE => 1, PER_ENTRY => 0},
  labelalphatemplate => {GLOBAL => 1, PER_TYPE => 1, PER_ENTRY => 0},
  labelnamespec      => {GLOBAL => 1, PER_TYPE => 1, PER_ENTRY => 0},
  labelnumber        => {GLOBAL => 1, PER_TYPE => 1, PER_ENTRY => 0},
  labelyear          => {GLOBAL => 1, PER_TYPE => 1, PER_ENTRY => 0},
  labelyearspec      => {GLOBAL => 1, PER_TYPE => 1, PER_ENTRY => 0},
  maxalphanames      => {GLOBAL => 1, PER_TYPE => 1, PER_ENTRY => 1},
  maxbibnames        => {GLOBAL => 1, PER_TYPE => 1, PER_ENTRY => 1},
  maxcitenames       => {GLOBAL => 1, PER_TYPE => 1, PER_ENTRY => 1},
  maxitems           => {GLOBAL => 1, PER_TYPE => 1, PER_ENTRY => 1},
  minalphanames      => {GLOBAL => 1, PER_TYPE => 1, PER_ENTRY => 1},
  minbibnames        => {GLOBAL => 1, PER_TYPE => 1, PER_ENTRY => 1},
  mincitenames       => {GLOBAL => 1, PER_TYPE => 1, PER_ENTRY => 1},
  minitems           => {GLOBAL => 1, PER_TYPE => 1, PER_ENTRY => 1},
  presort            => {GLOBAL => 1, PER_TYPE => 1, PER_ENTRY => 1},
  singletitle        => {GLOBAL => 1, PER_TYPE => 1, PER_ENTRY => 0},
  skipbib            => {GLOBAL => 0, PER_TYPE => 1, PER_ENTRY => 1},
  skiplab            => {GLOBAL => 0, PER_TYPE => 1, PER_ENTRY => 1},
  skiplos            => {GLOBAL => 0, PER_TYPE => 1, PER_ENTRY => 1},
  sortalphaothers    => {GLOBAL => 1, PER_TYPE => 1, PER_ENTRY => 0},
  sortexclusion      => {GLOBAL => 0, PER_TYPE => 1, PER_ENTRY => 0},
  sorting            => {GLOBAL => 1, PER_TYPE => 0, PER_ENTRY => 0},
  sortlos            => {GLOBAL => 1, PER_TYPE => 0, PER_ENTRY => 0},
  structure          => {GLOBAL => 1, PER_TYPE => 0, PER_ENTRY => 0},
  uniquelist         => {GLOBAL => 1, PER_TYPE => 1, PER_ENTRY => 1},
  uniquename         => {GLOBAL => 1, PER_TYPE => 1, PER_ENTRY => 1},
  useauthor          => {GLOBAL => 1, PER_TYPE => 1, PER_ENTRY => 1},
  useeditor          => {GLOBAL => 1, PER_TYPE => 1, PER_ENTRY => 1},
  useprefix          => {GLOBAL => 1, PER_TYPE => 1, PER_ENTRY => 1},
  usetranslator      => {GLOBAL => 1, PER_TYPE => 1, PER_ENTRY => 1},
);

# For per-entry options, what should be set when we find them and
# what should be output to the .bbl for biblatex.
# Basically, here we have to emulate relevant parts of biblatex's options processing
# for local entry-specific options, note therefore the presence here of some
# options like max/mincitenames which are not passed in the .bcf
our %CONFIG_BIBLATEX_PER_ENTRY_OPTIONS =  (
  dataonly      => {OUTPUT => 1, INPUT => ['skiplab', 'skiplos']},
  displaymode   => {OUTPUT => 0},
  maxitems      => {OUTPUT => 1},
  minitems      => {OUTPUT => 1},
  maxbibnames   => {OUTPUT => 1},
  minbibnames   => {OUTPUT => 1},
  maxcitenames  => {OUTPUT => 1},
  mincitenames  => {OUTPUT => 1},
  maxalphanames => {OUTPUT => 0},
  minalphanames => {OUTPUT => 0},
  maxnames      => {OUTPUT => ['maxcitenames', 'maxbibnames'], INPUT  => ['maxcitenames', 'maxbibnames']},
  minnames      => {OUTPUT => ['mincitenames', 'minbibnames'], INPUT  => ['mincitenames', 'minbibnames']},
  presort       => {OUTPUT => 0},
  skipbib       => {OUTPUT => 1},
  skiplab       => {OUTPUT => 1},
  skiplos       => {OUTPUT => 1},
  uniquelist    => {OUTPUT => 0},
  useauthor     => {OUTPUT => 1},
  useeditor     => {OUTPUT => 1},
  useprefix     => {OUTPUT => 1},
  usetranslator => {OUTPUT => 1},
);


1;

__END__

=pod

=encoding utf-8

=head1 NAME

Biber::Constants - global constants for biber

=head1 AUTHOR

François Charette, C<< <firmicus at ankabut.net> >>
Philip Kime C<< <philip at kime.org.uk> >>

=head1 BUGS

Please report any bugs or feature requests on our sourceforge tracker at
L<https://sourceforge.net/tracker2/?func=browse&group_id=228270>.

=head1 COPYRIGHT & LICENSE

Copyright 2009-2012 François Charette and Philip Kime, all rights reserved.

This module is free software.  You can redistribute it and/or
modify it under the terms of the Artistic License 2.0.

This program is distributed in the hope that it will be useful,
but without any warranty; without even the implied warranty of
merchantability or fitness for a particular purpose.

=cut
