"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.OpenSearchConfigStoreClient = void 0;
var _lodash = require("lodash");
var _constants = require("../../utils/constants");
var _utils = require("../../utils/utils");
function _classPrivateFieldInitSpec(e, t, a) { _checkPrivateRedeclaration(e, t), t.set(e, a); }
function _checkPrivateRedeclaration(e, t) { if (t.has(e)) throw new TypeError("Cannot initialize the same private elements twice on an object"); }
function _classPrivateFieldGet(s, a) { return s.get(_assertClassBrand(s, a)); }
function _classPrivateFieldSet(s, a, r) { return s.set(_assertClassBrand(s, a), r), r; }
function _assertClassBrand(e, t, n) { if ("function" == typeof e ? e === t : e.has(t)) return arguments.length < 3 ? t : n; throw new TypeError("Private element is not present on this object"); } /*
                                                                                                                                                                                                     * Copyright OpenSearch Contributors
                                                                                                                                                                                                     * SPDX-License-Identifier: Apache-2.0
                                                                                                                                                                                                     */
var _openSearchClient = /*#__PURE__*/new WeakMap();
var _cache = /*#__PURE__*/new WeakMap();
/**
 * This is the default client DAO when "dynamic_config_service.enabled: true" and no plugin has registered a DAO factory.
 * This client will fetch configs from .opensearch_dashboards_config alias.
 * The alias is important as it will always point to the latest "version" of the config index
 */
class OpenSearchConfigStoreClient {
  constructor(openSearchClient) {
    _classPrivateFieldInitSpec(this, _openSearchClient, void 0);
    _classPrivateFieldInitSpec(this, _cache, new Map());
    _classPrivateFieldSet(_openSearchClient, this, openSearchClient);
  }

  /**
   * Inserts the config index and an alias that points to it
   *
   * TODO Add migration logic
   */
  async createDynamicConfigIndex() {
    const existsAliasResponse = await _classPrivateFieldGet(_openSearchClient, this).indices.existsAlias({
      name: _constants.DYNAMIC_APP_CONFIG_ALIAS
    });
    if (!existsAliasResponse.body) {
      const latestVersion = await this.searchLatestConfigIndex();
      if (latestVersion < 1) {
        await _classPrivateFieldGet(_openSearchClient, this).indices.create({
          index: (0, _utils.getDynamicConfigIndexName)(1),
          body: {
            aliases: {
              [_constants.DYNAMIC_APP_CONFIG_ALIAS]: {}
            }
          }
        });
      } else {
        await _classPrivateFieldGet(_openSearchClient, this).indices.updateAliases({
          body: {
            actions: [{
              add: {
                index: (0, _utils.getDynamicConfigIndexName)(latestVersion),
                alias: _constants.DYNAMIC_APP_CONFIG_ALIAS
              }
            }]
          }
        });
      }
    } else {
      const results = await _classPrivateFieldGet(_openSearchClient, this).indices.getAlias({
        name: _constants.DYNAMIC_APP_CONFIG_ALIAS
      });
      const indices = Object.keys(results.body);
      if (indices.length !== 1) {
        throw new Error(`Alias ${_constants.DYNAMIC_APP_CONFIG_ALIAS} is pointing to 0 or multiple indices. Please remove the alias(es) and restart the server`);
      }
      const numNonDynamicConfigIndices = indices.filter(index => !(0, _utils.isDynamicConfigIndex)(index)).length;
      if (numNonDynamicConfigIndices > 0) {
        throw new Error(`Alias ${_constants.DYNAMIC_APP_CONFIG_ALIAS} is pointing to a non dynamic config index. Please remove the alias and restart the server`);
      }
    }
  }
  async getConfig(namespace, options) {
    if (_classPrivateFieldGet(_cache, this).has(namespace)) {
      return _classPrivateFieldGet(_cache, this).get(namespace);
    }
    const result = (await this.searchConfigsRequest([namespace])).body.hits.hits;
    if (result.length <= 0) {
      _classPrivateFieldGet(_cache, this).set(namespace, undefined);
      return undefined;
    }
    const source = result[0]._source;
    this.setCacheFromSearch(result[0]);
    return source === null || source === void 0 ? void 0 : source.config_blob;
  }
  async bulkGetConfigs(namespaces, options) {
    const results = new Map();
    const configsToQuery = namespaces.filter(namespace => {
      const isCached = _classPrivateFieldGet(_cache, this).has(namespace);
      const config = _classPrivateFieldGet(_cache, this).get(namespace);
      if (config) {
        results.set(namespace, config);
      }
      return !isCached;
    });
    if (configsToQuery.length <= 0) {
      return results;
    }
    let nonExistentConfigs = [...configsToQuery];
    const configs = await this.searchConfigsRequest(configsToQuery);
    configs.body.hits.hits.forEach(config => {
      // eslint-disable-next-line @typescript-eslint/naming-convention
      const {
        config_name,
        config_blob
      } = config._source;
      nonExistentConfigs = nonExistentConfigs.filter(name => name !== config_name);
      if (config_blob) {
        results.set(config_name, config_blob);
      }
      this.setCacheFromSearch(config);
    });

    // Cache results that weren't found
    nonExistentConfigs.forEach(name => {
      _classPrivateFieldGet(_cache, this).set(name, undefined);
    });
    return results;
  }
  async listConfigs(options) {
    // Cannot get from cache since config keys can be missing
    const configs = await _classPrivateFieldGet(_openSearchClient, this).search({
      index: _constants.DYNAMIC_APP_CONFIG_ALIAS,
      body: {
        size: _constants.DYNAMIC_APP_CONFIG_MAX_RESULT_SIZE,
        query: {
          match_all: {}
        }
      }
    });
    const results = new Map(configs.body.hits.hits.filter(config => {
      var _config$_source;
      this.setCacheFromSearch(config);
      return !!((_config$_source = config._source) !== null && _config$_source !== void 0 && _config$_source.config_blob);
    }).map(config => {
      var _config$_source2, _config$_source3;
      return [(_config$_source2 = config._source) === null || _config$_source2 === void 0 ? void 0 : _config$_source2.config_name, (_config$_source3 = config._source) === null || _config$_source3 === void 0 ? void 0 : _config$_source3.config_blob];
    }));
    return results;
  }
  async createConfig(createConfigProps, options) {
    const {
      config
    } = createConfigProps;
    const name = (0, _utils.pathToString)(config);
    return await this.createConfigsRequest(new Map([[name, {
      configBlob: config
    }]]));
  }
  async bulkCreateConfigs(bulkCreateConfigProps, options) {
    return await this.createConfigsRequest(new Map(bulkCreateConfigProps.configs.map(configBlob => {
      const name = (0, _utils.pathToString)(configBlob);
      return [name, {
        configBlob
      }];
    })));
  }
  async deleteConfig(deleteConfigs, options) {
    const name = (0, _utils.pathToString)(deleteConfigs);
    return await this.deleteConfigsRequest([name]);
  }
  async bulkDeleteConfigs(bulkDeleteConfigs, options) {
    const namespaces = bulkDeleteConfigs.paths.map(path => {
      return (0, _utils.pathToString)(path);
    });
    return await this.deleteConfigsRequest(namespaces);
  }
  clearCache() {
    _classPrivateFieldGet(_cache, this).clear();
  }

  /**
   * Adds config names to the cache from search hits
   *
   * @param config
   */
  setCacheFromSearch(config) {
    // eslint-disable-next-line @typescript-eslint/naming-convention
    const {
      config_blob,
      config_name
    } = config._source;
    _classPrivateFieldGet(_cache, this).set(config_name, config_blob);
  }

  /**
   * Adds config names to the cache from a config document
   *
   * @param config
   */
  setCache(config) {
    _classPrivateFieldGet(_cache, this).set(config.config_name, config.config_blob);
  }

  /**
   * Sends a bulk update/request to create/update the new configs
   *
   * @param configMap config name and config blob key/pair values
   */
  async createConfigsRequest(configMap) {
    const existingConfigs = await this.searchConfigsRequest([...configMap.keys()], true);
    const existingConfigNames = [];

    // Update the existing configs with the new config blob
    const bulkConfigs = existingConfigs.body.hits.hits.flatMap(config => {
      var _config$_source4, _configMap$get;
      const configName = (_config$_source4 = config._source) === null || _config$_source4 === void 0 ? void 0 : _config$_source4.config_name;
      existingConfigNames.push(configName);
      const configBlob = (_configMap$get = configMap.get(configName)) === null || _configMap$get === void 0 ? void 0 : _configMap$get.configBlob.updatedConfig;
      this.setCache({
        ...config._source,
        config_blob: configBlob
      });
      return [{
        update: {
          _id: config._id,
          _index: _constants.DYNAMIC_APP_CONFIG_ALIAS,
          retry_on_conflict: 2,
          routing: '',
          version: config._version + 1,
          version_type: 'external'
        }
      }, {
        doc: {
          // Only need to update the blob
          config_blob: configBlob
        }
      }];
    });

    // Create the rest
    const configsToCreate = [...configMap.keys()].filter(name => !existingConfigNames.includes(name));
    configsToCreate.forEach(name => {
      const {
        configBlob
      } = configMap.get(name);
      const newConfigDocument = {
        config_name: name,
        config_blob: configBlob.updatedConfig
      };
      this.setCache(newConfigDocument);
      bulkConfigs.push({
        create: {
          _id: (0, _lodash.uniqueId)(),
          _index: _constants.DYNAMIC_APP_CONFIG_ALIAS,
          retry_on_conflict: 2,
          routing: '',
          version: 1,
          version_type: 'external'
        }
      }, newConfigDocument);
    });
    return await _classPrivateFieldGet(_openSearchClient, this).bulk({
      index: _constants.DYNAMIC_APP_CONFIG_ALIAS,
      body: bulkConfigs
    });
  }

  /**
   * Deletes documents whose config name matches the query
   *
   * @param namespaces list of config names to search
   * @returns
   */
  async deleteConfigsRequest(namespaces) {
    namespaces.forEach(name => _classPrivateFieldGet(_cache, this).delete(name));
    return await _classPrivateFieldGet(_openSearchClient, this).deleteByQuery({
      index: _constants.DYNAMIC_APP_CONFIG_ALIAS,
      body: {
        query: {
          bool: {
            should: [{
              terms: {
                config_name: namespaces
              }
            }]
          }
        }
      }
    });
  }

  /**
   * Returns documents whose config name matches the query
   *
   * @param namespaces list of config names to search
   * @param excludeConfigBlob whether to include the config blob in the response
   * @returns
   */
  async searchConfigsRequest(namespaces, excludeConfigBlob = false) {
    return await _classPrivateFieldGet(_openSearchClient, this).search({
      ...(excludeConfigBlob && {
        _source: ['config_name']
      }),
      index: _constants.DYNAMIC_APP_CONFIG_ALIAS,
      body: {
        query: {
          bool: {
            should: [{
              terms: {
                config_name: namespaces
              }
            }]
          }
        }
      }
    });
  }

  /**
   * Finds the most updated dynamic config index
   *
   * @returns the latest version number or 0 if not found
   */
  async searchLatestConfigIndex() {
    const configIndices = await _classPrivateFieldGet(_openSearchClient, this).cat.indices({
      index: `${_constants.DYNAMIC_APP_CONFIG_INDEX_PREFIX}_*`,
      format: 'json'
    });
    if (configIndices.body.length < 1) {
      return 0;
    }
    const validIndices = configIndices.body.map(hit => {
      var _hit$index;
      return (_hit$index = hit.index) === null || _hit$index === void 0 ? void 0 : _hit$index.toString();
    }).filter(index => index && (0, _utils.isDynamicConfigIndex)(index));
    return validIndices.length === 0 ? 0 : validIndices.map(configIndex => {
      return configIndex ? (0, _utils.extractVersionFromDynamicConfigIndex)(configIndex) : 0;
    }).reduce((currentMax, currentNum) => {
      return currentMax && currentNum && currentMax > currentNum ? currentMax : currentNum;
    });
  }
}
exports.OpenSearchConfigStoreClient = OpenSearchConfigStoreClient;