﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/workmail/WorkMailRequest.h>
#include <aws/workmail/WorkMail_EXPORTS.h>

#include <utility>

namespace Aws {
namespace WorkMail {
namespace Model {

/**
 */
class GetMailboxDetailsRequest : public WorkMailRequest {
 public:
  AWS_WORKMAIL_API GetMailboxDetailsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "GetMailboxDetails"; }

  AWS_WORKMAIL_API Aws::String SerializePayload() const override;

  AWS_WORKMAIL_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The identifier for the organization that contains the user whose mailbox
   * details are being requested.</p>
   */
  inline const Aws::String& GetOrganizationId() const { return m_organizationId; }
  inline bool OrganizationIdHasBeenSet() const { return m_organizationIdHasBeenSet; }
  template <typename OrganizationIdT = Aws::String>
  void SetOrganizationId(OrganizationIdT&& value) {
    m_organizationIdHasBeenSet = true;
    m_organizationId = std::forward<OrganizationIdT>(value);
  }
  template <typename OrganizationIdT = Aws::String>
  GetMailboxDetailsRequest& WithOrganizationId(OrganizationIdT&& value) {
    SetOrganizationId(std::forward<OrganizationIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The identifier for the user whose mailbox details are being requested.</p>
   * <p>The identifier can be the <i>UserId</i>, <i>Username</i>, or <i>email</i>.
   * The following identity formats are available:</p> <ul> <li> <p>User ID:
   * 12345678-1234-1234-1234-123456789012 or
   * S-1-1-12-1234567890-123456789-123456789-1234</p> </li> <li> <p>Email address:
   * user@domain.tld</p> </li> <li> <p>User name: user</p> </li> </ul>
   */
  inline const Aws::String& GetUserId() const { return m_userId; }
  inline bool UserIdHasBeenSet() const { return m_userIdHasBeenSet; }
  template <typename UserIdT = Aws::String>
  void SetUserId(UserIdT&& value) {
    m_userIdHasBeenSet = true;
    m_userId = std::forward<UserIdT>(value);
  }
  template <typename UserIdT = Aws::String>
  GetMailboxDetailsRequest& WithUserId(UserIdT&& value) {
    SetUserId(std::forward<UserIdT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_organizationId;
  bool m_organizationIdHasBeenSet = false;

  Aws::String m_userId;
  bool m_userIdHasBeenSet = false;
};

}  // namespace Model
}  // namespace WorkMail
}  // namespace Aws
