/*
 * Decompiled with CFR 0.152.
 */
package com.azure.core.util.polling;

import com.azure.core.http.HttpHeader;
import com.azure.core.http.HttpHeaderName;
import com.azure.core.http.rest.Response;
import com.azure.core.implementation.ImplUtils;
import com.azure.core.util.SharedExecutorService;
import com.azure.core.util.logging.ClientLogger;
import com.azure.core.util.polling.AsyncPollResponse;
import com.azure.core.util.polling.LongRunningOperationStatus;
import com.azure.core.util.polling.PollResponse;
import com.azure.core.util.polling.PollingContext;
import com.azure.core.util.polling.implementation.PollingUtils;
import java.net.MalformedURLException;
import java.time.Duration;
import java.util.Objects;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.function.BiFunction;
import java.util.function.Function;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;

class PollingUtil {
    private static final ClientLogger LOGGER = new ClientLogger(PollingUtil.class);

    PollingUtil() {
    }

    static <T> PollResponse<T> pollingLoop(PollingContext<T> pollingContext, Duration timeout, LongRunningOperationStatus statusToWaitFor, Function<PollingContext<T>, PollResponse<T>> pollOperation, Duration pollInterval, boolean isWaitForStatus) {
        boolean timeBound = timeout != null;
        long timeoutInMillis = timeBound ? timeout.toMillis() : -1L;
        long startTime = System.currentTimeMillis();
        PollResponse intermediatePollResponse = pollingContext.getLatestResponse();
        boolean firstPoll = true;
        while (!intermediatePollResponse.getStatus().isComplete()) {
            Future<PollResponse> pollOp;
            long elapsedTime = System.currentTimeMillis() - startTime;
            if (timeBound && elapsedTime >= timeoutInMillis) {
                if (intermediatePollResponse.getStatus().equals(statusToWaitFor) || isWaitForStatus) {
                    return intermediatePollResponse;
                }
                throw LOGGER.logExceptionAsError(new RuntimeException(new TimeoutException("Polling didn't complete before the timeout period.")));
            }
            if (intermediatePollResponse.getStatus().equals(statusToWaitFor)) {
                return intermediatePollResponse;
            }
            if (firstPoll) {
                firstPoll = false;
                pollOp = SharedExecutorService.getInstance().submit(() -> (PollResponse)pollOperation.apply(pollingContext));
            } else {
                Duration delay = PollingUtil.getDelay(intermediatePollResponse, pollInterval);
                pollOp = SharedExecutorService.getInstance().schedule(() -> (PollResponse)pollOperation.apply(pollingContext), delay.toMillis(), TimeUnit.MILLISECONDS);
            }
            try {
                long pollTimeout = timeBound ? timeoutInMillis - elapsedTime : -1L;
                intermediatePollResponse = ImplUtils.getResultWithTimeout(pollOp, pollTimeout);
                pollingContext.setLatestResponse(intermediatePollResponse);
            }
            catch (InterruptedException | ExecutionException | TimeoutException e) {
                if (isWaitForStatus) {
                    return intermediatePollResponse;
                }
                throw LOGGER.logExceptionAsError(new RuntimeException(e));
            }
        }
        return intermediatePollResponse;
    }

    static <T, U> Flux<AsyncPollResponse<T, U>> pollingLoopAsync(PollingContext<T> pollingContext, Function<PollingContext<T>, Mono<PollResponse<T>>> pollOperation, BiFunction<PollingContext<T>, PollResponse<T>, Mono<T>> cancelOperation, Function<PollingContext<T>, Mono<U>> fetchResultOperation, Duration pollInterval) {
        return Flux.using(() -> pollingContext, cxt -> Mono.defer(() -> (Mono)pollOperation.apply((PollingContext)cxt)).delaySubscription(PollingUtil.getDelay(cxt.getLatestResponse(), pollInterval)).switchIfEmpty(Mono.error(() -> new IllegalStateException("PollOperation returned Mono.empty()."))).repeat().takeUntil(currentPollResponse -> currentPollResponse.getStatus().isComplete()).concatMap(currentPollResponse -> {
            cxt.setLatestResponse(currentPollResponse);
            return Mono.just(new AsyncPollResponse(cxt, cancelOperation, fetchResultOperation));
        }), cxt -> {});
    }

    private static <T> Duration getDelay(PollResponse<T> pollResponse, Duration pollInterval) {
        Duration retryAfter = pollResponse.getRetryAfter();
        if (retryAfter == null) {
            return pollInterval;
        }
        return retryAfter.compareTo(Duration.ZERO) > 0 ? retryAfter : pollInterval;
    }

    static <T, U> PollResponse<T> toPollResponse(AsyncPollResponse<T, U> asyncPollResponse) {
        return new PollResponse<T>(asyncPollResponse.getStatus(), asyncPollResponse.getValue(), asyncPollResponse.getRetryAfter());
    }

    static boolean matchStatus(AsyncPollResponse<?, ?> currentPollResponse, LongRunningOperationStatus statusToWaitFor) {
        if (currentPollResponse == null || statusToWaitFor == null) {
            return false;
        }
        return statusToWaitFor == currentPollResponse.getStatus();
    }

    static boolean locationCanPoll(Response<?> initialResponse, String endpoint, ClientLogger logger) {
        HttpHeader locationHeader = initialResponse.getHeaders().get(HttpHeaderName.LOCATION);
        if (locationHeader != null) {
            try {
                ImplUtils.createUrl(PollingUtils.getAbsolutePath(locationHeader.getValue(), endpoint, logger));
                return true;
            }
            catch (MalformedURLException e) {
                logger.info("Failed to parse Location header into a URL.", e);
                return false;
            }
        }
        return false;
    }

    static boolean operationResourceCanPoll(Response<?> initialResponse, HttpHeaderName operationLocationHeader, String endpoint, ClientLogger logger) {
        HttpHeader header = initialResponse.getHeaders().get(operationLocationHeader);
        if (header != null) {
            try {
                ImplUtils.createUrl(PollingUtils.getAbsolutePath(header.getValue(), endpoint, logger));
                return true;
            }
            catch (MalformedURLException e) {
                return false;
            }
        }
        return false;
    }

    static void validateTimeout(Duration timeout, ClientLogger logger) {
        Objects.requireNonNull(timeout, "'timeout' cannot be null.");
        if (timeout.isNegative() || timeout.isZero()) {
            throw logger.logExceptionAsWarning(new IllegalArgumentException("Negative or zero value for timeout is not allowed."));
        }
    }

    static Duration validatePollInterval(Duration pollInterval, ClientLogger logger) {
        Objects.requireNonNull(pollInterval, "'pollInterval' cannot be null.");
        if (pollInterval.isNegative() || pollInterval.isZero()) {
            throw logger.logExceptionAsWarning(new IllegalArgumentException("Negative or zero value for pollInterval is not allowed."));
        }
        return pollInterval;
    }
}

