﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/sesv2/SESV2Request.h>
#include <aws/sesv2/SESV2_EXPORTS.h>

#include <utility>

namespace Aws {
namespace SESV2 {
namespace Model {

/**
 * <p>A request to delete an event destination from a configuration
 * set.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/sesv2-2019-09-27/DeleteConfigurationSetEventDestinationRequest">AWS
 * API Reference</a></p>
 */
class DeleteConfigurationSetEventDestinationRequest : public SESV2Request {
 public:
  AWS_SESV2_API DeleteConfigurationSetEventDestinationRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DeleteConfigurationSetEventDestination"; }

  AWS_SESV2_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The name of the configuration set that contains the event destination to
   * delete.</p>
   */
  inline const Aws::String& GetConfigurationSetName() const { return m_configurationSetName; }
  inline bool ConfigurationSetNameHasBeenSet() const { return m_configurationSetNameHasBeenSet; }
  template <typename ConfigurationSetNameT = Aws::String>
  void SetConfigurationSetName(ConfigurationSetNameT&& value) {
    m_configurationSetNameHasBeenSet = true;
    m_configurationSetName = std::forward<ConfigurationSetNameT>(value);
  }
  template <typename ConfigurationSetNameT = Aws::String>
  DeleteConfigurationSetEventDestinationRequest& WithConfigurationSetName(ConfigurationSetNameT&& value) {
    SetConfigurationSetName(std::forward<ConfigurationSetNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the event destination to delete.</p>
   */
  inline const Aws::String& GetEventDestinationName() const { return m_eventDestinationName; }
  inline bool EventDestinationNameHasBeenSet() const { return m_eventDestinationNameHasBeenSet; }
  template <typename EventDestinationNameT = Aws::String>
  void SetEventDestinationName(EventDestinationNameT&& value) {
    m_eventDestinationNameHasBeenSet = true;
    m_eventDestinationName = std::forward<EventDestinationNameT>(value);
  }
  template <typename EventDestinationNameT = Aws::String>
  DeleteConfigurationSetEventDestinationRequest& WithEventDestinationName(EventDestinationNameT&& value) {
    SetEventDestinationName(std::forward<EventDestinationNameT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_configurationSetName;

  Aws::String m_eventDestinationName;
  bool m_configurationSetNameHasBeenSet = false;
  bool m_eventDestinationNameHasBeenSet = false;
};

}  // namespace Model
}  // namespace SESV2
}  // namespace Aws
